package org.eaglei.services.repository;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;

/**
 * 
 * @author Ricardo De Lima
 * @author Daniela Bourges
 * 
 *         May 12, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 */

public class SPARQLQueryUtil implements SPARQLOntConstants, SPARQLConstants {

	private static final Log logger = LogFactory.getLog( SPARQLQueryUtil.class );

    public static EIEntity getStatusEntity(final EIURI statusUri) {
        if ( statusUri == null ) {
            return EIEntity.NULL_ENTITY;
        }
        if ( DRAFT_URI.equals( statusUri.toString() ) ) {
            return DRAFT_ENTITY;
        } else if ( CURATION_URI.equals( statusUri.toString() ) ) {
            return CURATION_ENTITY;
        } else if ( PUBLISH_URI.equals( statusUri.toString() ) ) {
            return PUBLISH_ENTITY;
        } else if ( WITHDRAW_URI.equals( statusUri.toString() ) ) {
            return WITHDRAW_ENTITY;
        }
        return EIEntity.NULL_ENTITY;
    }

	public static String getReferencedByQuery(final EIURI resourceUri) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( namespaces() );
		sparql.append( selectClause() );
		sparql.append( commonPattern() );
		sparql.append( typePattern() );
		sparql.append( referencedByPattern( resourceUri ) );
		//sparql.append( typeRestrictionPattern( EIURI.NULL_EIURI, false ) );
		//sparql.append( stateRestrictionPattern( EIURI.NULL_EIURI ) );
		//sparql.append( labRestrictionPattern( EIURI.NULL_EIURI ) );
		//sparql.append( ownerRestrictionPattern( user, strictOwnerFilter ) );
		sparql.append( closingClause() );
		return sparql.toString();
	}

	public static String referencedByPattern(final EIURI resourceUri) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "?" );
		sparql.append( SUBJECT_VARIABLE );
		sparql.append( " ?anyPredicate <" );
		sparql.append( resourceUri.toString() );
		sparql.append( "> . " );
		return sparql.toString();
	}

	private static String namespaces() {
		final StringBuilder namespaces = new StringBuilder();
		namespaces.append( "PREFIX dcterms: <" );
		namespaces.append( DC_URI );
		namespaces.append( "> " );
		namespaces.append( "PREFIX rdf: <" );
		namespaces.append( RDF_URI );
		namespaces.append( "> " );
		namespaces.append( "PREFIX rdfs: <" );
		namespaces.append( RDFS_URI );
		namespaces.append( "> " );
		namespaces.append( "PREFIX repo: <" );
		namespaces.append( EAGLE_I_REPO_URI );
		namespaces.append( "> " );
		return namespaces.toString();
	}

	private static String selectClause() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "SELECT DISTINCT " );
        sparql.append( "?anyPredicate " );
		for (final String variable : resultSetVariables) {
			sparql.append( "?" );
			sparql.append( variable );
			sparql.append( " " );
		}
		sparql.append( "WHERE {" );
		return sparql.toString();
	}

	private static String commonPattern() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "?" );
		sparql.append( SUBJECT_VARIABLE );
		sparql.append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" );
		sparql.append( LABEL_VARIABLE );
		sparql.append( " . ?" );
		sparql.append( SUBJECT_VARIABLE );
		sparql.append( " dcterms:created ?" );
		sparql.append( DATE_VARIABLE );
		sparql.append( " . " );
		return sparql.toString();
	}

	private static String typePattern() {
		return "graph ?g{?" + SUBJECT_VARIABLE + " a ?" + TYPE_VARIABLE + "} filter(?g != repo:NG_Inferred) . ";
	}

	public static String allTypesPattern(final boolean excludeSomeTypes) {
		// Need to use a variable different than ?t,
		// because that gets only the triples that are not inferred types
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "?" );
		sparql.append( SUBJECT_VARIABLE );
		sparql.append( " a ?rt . " );
		sparql.append( "?rt <http://eagle-i.org/ont/app/1.0/inClassGroup> <http://eagle-i.org/ont/app/1.0/ClassGroup/resourceRoot> " );
		// FIXME manually exclude Organizations and Persons as resources
		// should be done by annotations in ontology
		if ( excludeSomeTypes ) {
			sparql.append( "filter(?rt != <" );
			sparql.append( OBI_ORGANIZATION );
			sparql.append( "> && ?rt != <" );
			sparql.append( FOAF_PERSON );
			sparql.append( ">) ." );
		}
		return sparql.toString();
	}

	private static String ownerRestrictionPattern(final String user, final boolean strictOwnerFilter) {
		final StringBuilder sparql = new StringBuilder();
		if ( strictOwnerFilter ) {
			sparql.append( "?" );
			sparql.append( SUBJECT_VARIABLE );
			sparql.append( " repo:hasWorkflowOwner ?" );
			sparql.append( OWNER_VARIABLE );
			sparql.append( ". FILTER(?" );
			sparql.append( OWNER_VARIABLE );
			sparql.append( "= <" );
		} else {
			sparql.append( "OPTIONAL {?" );
			sparql.append( SUBJECT_VARIABLE );
			sparql.append( " repo:hasWorkflowOwner ?" );
			sparql.append( OWNER_VARIABLE );
			sparql.append( "} . FILTER(!bound(?" );
			sparql.append( OWNER_VARIABLE );
			sparql.append( ") || ?" );
			sparql.append( OWNER_VARIABLE );
			sparql.append( " = <" );
		}
		sparql.append( user );
		sparql.append( ">) . " );
		return sparql.toString();
	}

	private static String typeRestrictionPattern(final EIURI classUri, final boolean excludeSomeTypes) {
		final StringBuilder sparql = new StringBuilder();
		if ( !isNull( classUri ) ) {
			sparql.append( "?" );
			sparql.append( SUBJECT_VARIABLE );
			sparql.append( " a <" );
			sparql.append( classUri.toString() );
			sparql.append( "> ." );
		} else {
			sparql.append( allTypesPattern( excludeSomeTypes ) );
		}
		return sparql.toString();
	}

	private static String stateRestrictionPattern(final EIURI state) {
		final StringBuilder sparql = new StringBuilder();
		if ( !isNull( state ) ) {
			sparql.append( "?" );
			sparql.append( SUBJECT_VARIABLE );
			sparql.append( " repo:hasWorkflowState <" );
			sparql.append( state.toString() );
			sparql.append( "> . ?" );
			sparql.append( SUBJECT_VARIABLE );
			sparql.append( " repo:hasWorkflowState ?" );
			sparql.append( STATE_VARIABLE );
			sparql.append( " . " );
		} else {
			sparql.append( "OPTIONAL {?" );
			sparql.append( SUBJECT_VARIABLE );
			sparql.append( " repo:hasWorkflowState ?" );
			sparql.append( STATE_VARIABLE );
			sparql.append( "} . " );
		}
		return sparql.toString();
	}

	public static String labRestrictionPattern(final EIURI lab) {
		final StringBuilder sparql = new StringBuilder();
		if ( !isNull( lab ) ) {
			sparql.append( "?" );
			sparql.append( SUBJECT_VARIABLE );
			sparql.append( " ?labProperty ?" );
			sparql.append( LAB_VARIABLE );
			sparql.append( " . ?" );
			sparql.append( LAB_VARIABLE );
			sparql.append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" );
			sparql.append( LAB_NAME_VARIABLE );
			sparql.append( " . ?labProperty <http://eagle-i.org/ont/app/1.0/inPropertyGroup> <http://eagle-i.org/ont/app/1.0/PropertyGroup/relatedLab> . filter(?" );
			sparql.append( LAB_VARIABLE );
			sparql.append( " = <" );
			sparql.append( lab.toString() );
			sparql.append( ">) . " );
		} else {
			sparql.append( "OPTIONAL {?" );
			sparql.append( SUBJECT_VARIABLE );
			sparql.append( " ?labProperty ?" );
			sparql.append( LAB_VARIABLE );
			sparql.append( " . ?labProperty <http://eagle-i.org/ont/app/1.0/inPropertyGroup> <http://eagle-i.org/ont/app/1.0/PropertyGroup/relatedLab> . ?" );
			sparql.append( LAB_VARIABLE );
			sparql.append( " a <" );
			sparql.append( EI_LAB );
			sparql.append( "> . ?" );
			sparql.append( LAB_VARIABLE );
			sparql.append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" );
			sparql.append( LAB_NAME_VARIABLE );
			sparql.append( "} . " );
		}
		return sparql.toString();
	}

	private static String closingClause() {
		return "} order by repo:upperCaseStr(?" + LABEL_VARIABLE + ")";
	}

	private static boolean isNull(final EIURI uri) {
		return uri == null || uri.equals( EIURI.NULL_EIURI ) || "".equals( uri.toString() );
	}
	
	
	/* ----- Copied from EIInstanceMinimalFactory --------------*/
	
    public static List<EIInstanceMinimal> createReferencedByInstances(final EIOntModel ontModel, final EIInstance objectInstance, final ResultSet resultSet) {
        if ( resultSet == null ) {
            return Collections.EMPTY_LIST;
        }
        final List<EIProperty> objInstProperties = ontModel.getProperties(objectInstance.getInstanceType().getURI());
        final Map<EIURI, EIInstanceMinimal> instances = new HashMap<EIURI, EIInstanceMinimal>();
        // Need to preserve ordering to avoid a sort operation; this list will contain live elements
        // FIXME is list still ordered with workflow?
        final List<EIInstanceMinimal> instanceList = new ArrayList<EIInstanceMinimal>();
        while ( resultSet.hasNext() ) {
            final QuerySolution solution = resultSet.next();
            // Only type and label are mandatory
            if ( solution.contains( SUBJECT_VARIABLE ) && solution.contains( TYPE_VARIABLE ) && solution.contains( LABEL_VARIABLE ) ) {
                final EIURI instanceUri = getUriFromSolution( solution, SUBJECT_VARIABLE );
                final EIURI instanceType = getUriFromSolution( solution, TYPE_VARIABLE );
                final String label = getStringFromSolution( solution, LABEL_VARIABLE );
                // type is not an EIClass ; we can skip solution
                if ( safeGetClass( ontModel, instanceType ) == null ) {
                    continue;
                }
                final EIEntity instanceEntity = EIEntity.create( instanceUri, label );
                
                // Perform on-the-fly inferencing of inverse property values
                boolean inferredValue = false;
                EIURI referencePredicate = getUriFromSolution( solution, ANY_PREDICATE_VARIABLE );
                if (referencePredicate != EIURI.NULL_EIURI) {
                    if ( logger.isDebugEnabled() ) {
                        logger.debug( "Referencing property: " + referencePredicate );
                    }
                    String referencePredicateStr = referencePredicate.toString();
                    for (EIProperty objInstProperty : objInstProperties) {
                        if (objInstProperty instanceof EIObjectProperty) {
                            String inverseProperty = ((EIObjectProperty) objInstProperty).getInverseURI();
                            if (referencePredicateStr.equals(inverseProperty)) {
                                Set<EIEntity> values = objectInstance.getObjectProperty(objInstProperty.getEntity());
                                inferredValue = true;
                                if (values == null || ! values.contains(instanceEntity)) {
                                    objectInstance.addObjectProperty(objInstProperty.getEntity(), instanceEntity);
                                    if ( logger.isDebugEnabled() ) {
                                        logger.debug( "Inferred inverse property: " + objInstProperty.getEntity() + "   value: " + instanceEntity);
                                    }
                                    break;
                                }
                            }
                        }
                    }
                }
                if (inferredValue) {
                    continue;
                }

                EIInstanceMinimal ei;
                if ( !instances.containsKey( instanceUri ) ) {
                    ei = createEmptyMinimal( ontModel, instanceType, instanceEntity );
                    instances.put( instanceUri, ei );
                    instanceList.add( ei );
                    if ( logger.isDebugEnabled() ) {
                        logger.debug( "Creating new EIInstance: " + ei );
                    }

                } else {
                    ei = instances.get( instanceUri );
                    ei.addEIType( EIEntity.create( instanceType, "" ) );
                    if ( logger.isDebugEnabled() ) {
                        logger.debug( "Adding type: " + instanceType + " to existing EIInstance :" + ei );
                    }
                }
                // Continue with setting optional parts
                ei.setWFOwner( getEntityFromSolution( solution, OWNER_VARIABLE, OWNER_NAME_VARIABLE ) );
                ei.setWFState( getEntityFromSolution( solution, STATE_VARIABLE, "" ) );
                ei.setCreationDate( getStringFromSolution( solution, DATE_VARIABLE ) );
                ei.setLab( getEntityFromSolution( solution, LAB_VARIABLE, LAB_NAME_VARIABLE ) );
            } else {
                continue;
            }
        }
        // This adds way too much time
        // Collections.sort(instanceList);
        return instanceList;
    }

    private static EIInstanceMinimal createEmptyMinimal(final EIOntModel ontModel, final EIURI typeUri, final EIEntity instanceEntity) {
        if ( instanceEntity == null ) {
            return null;
        }
        final EIClass instanceClass = safeGetClass( ontModel, typeUri );
        if ( instanceClass == null ) {
            return null;
        }
        final String label = instanceClass.getEntity().getLabel();
        instanceClass.getEntity().setLabel( label );
        final EIInstanceMinimal ei = EIInstanceMinimal.create( instanceClass.getEntity(), instanceEntity );
        return ei;
    }

    private static EIEntity getEntityFromSolution(final QuerySolution solution, final String uriVariable, final String labelVariable) {
        if ( solution.contains( uriVariable ) ) {
            final EIURI uri = EIURI.create( solution.getResource( uriVariable ).getURI() );
            if ( STATE_VARIABLE.equals( uriVariable ) ) {
                return getStatusEntity( uri );
            }
            String label = "<none>";
            if ( solution.contains( labelVariable ) ) {
                label = solution.getLiteral( labelVariable ).getString();
            }
            return EIEntity.create( uri, label );
        } else {
            return EIEntity.NULL_ENTITY;
        }
    }

    private static EIURI getUriFromSolution(final QuerySolution solution, final String variable) {
        if ( solution.contains( variable ) ) {
            return EIURI.create( solution.getResource( variable ).getURI() );
        } else {
            return EIURI.NULL_EIURI;
        }
    }

    private static String getStringFromSolution(final QuerySolution solution, final String variable) {
        if ( solution.contains( variable ) ) {
            return solution.getLiteral( variable ).getString();
        } else {
            return "";
        }
    }

    private static EIClass safeGetClass(final EIOntModel ontModel, final EIURI typeUri) {
        if ( ontModel.isModelClassURI( typeUri.toString() ) ) {
            return ontModel.getClass( typeUri );
        } else {
            return null;
        }
    }


}
