package org.eaglei.ui.gwt.instance.server;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.model.SearchEIInstancePreview;
import org.eaglei.search.logging.AsynchronousLoggerContact;
import org.eaglei.search.logging.AsynchronousLoggerInstance;
import org.eaglei.search.logging.AsynchronousLoggerSearch;
import org.eaglei.search.provider.AuthSearchRequest;
import org.eaglei.services.repository.RepositoryInstanceProvider;
import org.eaglei.services.repository.RepositoryProviderException;
import org.eaglei.services.repository.RepositoryProviderException.RepositoryProviderExceptionType;
import org.eaglei.services.repository.SecurityProvider;
import org.eaglei.ui.gwt.instance.rpc.InstanceServiceRemote;
import org.eaglei.ui.gwt.rpc.InvalidSessionIdException;
import org.eaglei.ui.gwt.rpc.LoggedException;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

/**
 * Implementation of InstanceService.
 */
public class InstanceServlet extends RemoteServiceServlet implements InstanceServiceRemote {

	private static final long serialVersionUID = 1L;
	private static final Log logger = LogFactory.getLog(InstanceServlet.class);    
    private RepositoryInstanceProvider instanceProvider;
    private SecurityProvider securityProvider;
    protected static final String READ_VIEW = "user";
    
    private static EIOntModel ontModel;
    
    private AsynchronousLoggerInstance asyncLoggerInstance;
    private AsynchronousLoggerContact asyncLoggerContact;
    private static final String INSTANCE_LOG_TABLE_NAME = "INSTANCE_SERVLET_LOG";
    private static final String CONTACT_LOG_TABLE_NAME = "CONTACT_LOG";

    public InstanceServlet() {
    }

    @Override
    public void init() {
        WebApplicationContext ctx = 
            WebApplicationContextUtils.getWebApplicationContext(getServletContext());
        instanceProvider = ctx.getBean(RepositoryInstanceProvider.class);
        securityProvider = ctx.getBean("securityProvider", SecurityProvider.class);
        ontModel = ctx.getBean( EIOntModel.class );
        
        // Create asynchronous logger         	
        String buildID = "Unknown";
        InputStream inputStream = null;
        // Get build ID from properties file created during build.
        try {
        	inputStream = ctx.getResource("/buildversion.properties").getInputStream();
        	
          	//TEMPORARY
        	if (inputStream == null) {
        		logger.debug("/buildversion.properties not found");
        		inputStream = ctx.getResource("/institution/buildversion.properties").getInputStream();
        		if (inputStream == null) {
        			logger.debug("/institution/buildversion.properties not found");
        			inputStream = ctx.getResource("/central/buildversion.properties").getInputStream();
        			if (inputStream == null) {
        				logger.debug("/central/buildversion.properties not found");
        			}
        		}
        	}
        	//TEMPORARY
        	
	        // If we find the file, set the build ID
	        if (inputStream != null) {
				Properties props = new Properties();
				props.load(inputStream);
				buildID = props.getProperty("buildversion", "Unknown");
			} 
        } catch(IOException ioe) {
			logger.error("Failed to find buildversion.properties");
        }

        
        // DEBUG
        logger.debug("InstanceServlet: Context Display name = " + ctx.getDisplayName());
        logger.debug("InstanceServlet: Context ID = " + ctx.getId());
        if (ctx.getParent() == null) {
        	logger.debug("InstanceServlet: Context Parent is null.");
        } else {
        	logger.debug("InstanceServlet: Context Parent Display name = " + ctx.getParent().getDisplayName());
        }
        if (ctx.getServletContext() == null) {
        	logger.debug("InstanceServlet: ServletContext is null.");
        } else {
	        logger.debug("InstanceServlet: ServletContext path = " + ctx.getServletContext().getContextPath());
	        logger.debug("InstanceServlet: ServletContext context name = " + ctx.getServletContext().getServletContextName());
	    }
        //DEBUG
        
        String ontologyVersion = ontModel.getVersion();
        asyncLoggerInstance = new AsynchronousLoggerInstance(INSTANCE_LOG_TABLE_NAME, buildID, ontologyVersion);
        asyncLoggerContact = new AsynchronousLoggerContact(CONTACT_LOG_TABLE_NAME, buildID, ontologyVersion);
    }

    @Override
    public void destroy() {
    }
    
	/* (non-Javadoc)
	 * @see org.eaglei.services.repository.InstanceProvider#getToolTips(org.eaglei.security.Session, org.eaglei.model.EIURI)
	 */

    public Map<EIURI,String> getToolTips(String sessionID, EIURI instanceID) throws InvalidSessionIdException, LoggedException {
        try {
        	securityProvider.isValid(sessionID, true);
        }
        catch (RepositoryProviderException e) {
        	//TODO: Should this be InvalidSessionIdException or just reuse RepositoryProviderException?
        	throw new InvalidSessionIdException("Error validating session.", e);
        }
    	try {
    		return instanceProvider.getToolTips(sessionID, instanceID);

    	} catch (Throwable t) {
    		logger.error("Error getting tool tips", t);
    		throw new LoggedException(t.getLocalizedMessage());
    	}     
    }
    
    public EIInstance getEIInstance(String sessionID, EIURI instanceID) throws InvalidSessionIdException, LoggedException {
    	try {
        	securityProvider.isValid(sessionID, true);
        }
        catch (RepositoryProviderException e) {
        	//TODO: Should this be InvalidSessionIdException or just reuse RepositoryProviderException?
        	throw new InvalidSessionIdException("Error validating session.", e);
        }
    	try {
    		long start = System.currentTimeMillis();
    		EIInstance result = instanceProvider.getEIInstance( sessionID, instanceID );
    		
    		// Log entry for this search
    		long end = System.currentTimeMillis();
    		asyncLoggerInstance.log(securityProvider.getSession(sessionID), start, end, instanceID, result);
    		
    		return result;
    	} catch (Throwable t) {
    		logger.error("Error getting instance", t);
    		throw new LoggedException(t.getLocalizedMessage());
    	}   
    }
    
    public boolean contactMessage(String sessionID, EIURI instanceID, String label, boolean test_mode, 
            String from_name, String from_email, String subject, String message) throws InvalidSessionIdException, LoggedException {
    	try {
        	securityProvider.isValid(sessionID, true);
        }
        catch (RepositoryProviderException e) {
        	//TODO: Should this be InvalidSessionIdException or just reuse RepositoryProviderException?
        	throw new InvalidSessionIdException("Error validating session.", e);
        }
        try {
            String client_ip = getThreadLocalRequest().getLocalAddr();
            asyncLoggerContact.log(sessionID, client_ip, instanceID, label, from_name, from_email, subject);
            return instanceProvider.contactMessage(sessionID, client_ip, instanceID, label, test_mode, from_name, from_email, subject, message);
        } catch (Throwable t) {
            logger.error("Error sending contact message", t);
            throw new LoggedException(t.getLocalizedMessage());
        }     
    }
    
    
    public EIInstance setReferencingResources(String sessionID, EIInstance instance) throws Exception {
    	try {
        	securityProvider.isValid(sessionID, true);
        }
        catch (RepositoryProviderException e) {
        	//TODO: Should this be InvalidSessionIdException or just reuse RepositoryProviderException?
        	throw new InvalidSessionIdException("Error validating session.", e);
        }
    	 
    	 try {
    		return  instanceProvider.setReferencingResources( sessionID, instance );
    	 } catch(Throwable t) {
    		 logger.error("Error setting Referencing resources", t);
             throw new LoggedException(t.getLocalizedMessage());
    	 }
    }

	@Override
	public List<EIInstanceMinimal> listReferencingResources(String sessionID, String userURI, EIURI resourceUri, AuthSearchRequest queryRequest, boolean strictOwnerFilter) throws Exception {
		try {
        	securityProvider.isValid(sessionID, true);
        }
        catch (RepositoryProviderException e) {
        	//TODO: Should this be InvalidSessionIdException or just reuse RepositoryProviderException?
        	throw new InvalidSessionIdException("Error validating session.", e);
        }
		try {
			return instanceProvider.listReferencingResources( sessionID, userURI, resourceUri, queryRequest, strictOwnerFilter );

		} catch(Throwable t) {
			 logger.error("Error listing Referencing resources", t);
             throw new LoggedException(t.getLocalizedMessage());
		}
	}
	
	@Override
	public SearchEIInstancePreview getObjectPropertyValuePreview(final String sessionID, final EIURI instanceUri) throws RepositoryProviderException {
		try {
			return instanceProvider.getObjectPropertyValuePreview(sessionID, instanceUri);
		}
		catch (RepositoryProviderException e) {
			throw e;
		}
		catch (Throwable t) {
			throw new RepositoryProviderException("Unable to get instance preview", t, RepositoryProviderExceptionType.GENERAL );
		}
	}
    

}
