package org.eaglei.ui.gwt.instance.widgets;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.eaglei.model.EIEntity;
import org.eaglei.model.SearchEIInstancePreview;
import org.eaglei.services.repository.RepositoryProviderException;
import org.eaglei.ui.gwt.instance.rpc.InstanceServiceRemoteAsync;
import org.eaglei.ui.gwt.security.SessionContext;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyDownEvent;
import com.google.gwt.event.dom.client.KeyDownHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Anchor;

/** Class that acts as a link but uses a popup to display
 * a preview on hover.
 * 
 * @author JJB25
 *
 */
public class EIInstancePreviewLink extends Anchor {
	
	/** The popup attached to this link. */
	private EIInstancePreviewPopup popup;
	/** The instance service. */
	private final InstanceServiceRemoteAsync instanceService;
	/** Logger for this class. */
	private final Logger log = Logger.getLogger( "EIInstancePreviewLink" );
	/** A flag indicating whether the mouse is still hovering over the link. */
	private boolean mouseOver = false;
	
	/** Constructor for this class.
	 * 
	 * @param text Text for the link.
	 * @param service The instance service.
	 * @param propertyValue The object property that this link goes to.
	 */
	public EIInstancePreviewLink(String text, InstanceServiceRemoteAsync service, EIEntity propertyValue) {
		super(text, true);
		instanceService = service;
		addMouseOverHandler(new EIInstancePreviewMouseOverHandler(propertyValue));
		addMouseOutHandler(new EIInstancePreviewMouseOutHandler());
		addClickHandler(new EIPreviewPopUpClickHandler());
		addKeyDownHandler(new EIFocusKeyHandler());
	}
	
	/** Handler for mouse over events (i.e. hovers). Creates and displays
	 * a preview popup based on a preview retrieved from the repository.
	 * 
	 * @author JJB25
	 *
	 */
	private class EIInstancePreviewMouseOverHandler implements MouseOverHandler {
		
		/** The property that we are retrieving a preview of. */
		private EIEntity property;
		
		/** Constructs a new handler.
		 * 
		 * @param propertyValue The property that we are retrieving a preview of.
		 */
		public EIInstancePreviewMouseOverHandler(EIEntity propertyValue) {
			property = propertyValue;
		}

		@Override
		/** Uses the instance service to retrieve a preview (if it hasn't already been done).
		 *  Then generates a new preview popup using that preview return. If a popup has already
		 *  been generated then it is simply displayed.
		 */
		public void onMouseOver(MouseOverEvent event) {
			mouseOver = true;
			setFocus(true);
			if (popup == null) {
				try {
					instanceService.getObjectPropertyValuePreview(SessionContext.getSessionId(), property.getURI(), new EIPreviewPopupCallback());
				}
				catch (RepositoryProviderException e) {
					log.log(Level.SEVERE, "Unable to retrieve instance preview from callback.", e);
				}
			}
			else if (!popup.isShowing()) {
				popup.switchFocusMode(false);
			}
			if (popup != null) {
				int left = EIInstancePreviewLink.this.getAbsoluteLeft();
				int top = EIInstancePreviewLink.this.getAbsoluteTop() + EIInstancePreviewLink.this.getOffsetHeight() + 5;
				popup.setPopupPosition(left, top);
				popup.show();
			}
		}
	}
	
	/** Handler class for mouse out events.
	 * 
	 * @author JJB25
	 *
	 */
	private class EIInstancePreviewMouseOutHandler implements MouseOutHandler {
		
		@Override
		/** Handler for mouse out events. Sets the mouse over flag to false
		 * and closes the popup if it exists.
		 * 
		 */
		public void onMouseOut(MouseOutEvent event) {
			mouseOver = false;
			setFocus(false);
			if (popup != null) {
				popup.hide();
			}
		}
	}
	
	/** Callback class for retrieving the search instance preview from the server. 
	 * 
	 * @author JJB25
	 *
	 */
	private class EIPreviewPopupCallback implements AsyncCallback<SearchEIInstancePreview> {

		@Override
		public void onFailure(Throwable caught) {
			log.log(Level.SEVERE, "Unable to retrieve instance preview from callback.", caught);
		}

		@Override
		/** Called when the callback suceeds. Creates a popup with the returned preview and
		 * shows it if the mouse over flag is still set (needed given the asynchronous nature 
		 * of the call).
		 * 
		 */
		public void onSuccess(SearchEIInstancePreview result) {
			popup = new EIInstancePreviewPopup(result);
			if (mouseOver) {
				int left = EIInstancePreviewLink.this.getAbsoluteLeft();
				int top = EIInstancePreviewLink.this.getAbsoluteTop() + EIInstancePreviewLink.this.getOffsetHeight() + 5;
				popup.setPopupPosition(left, top);
				popup.show();
			}
			log.info("Retrieved instance preview from callback with " + result.getPropertyCount() + " properties.");
		}
		
	}
	
	/** Handler for key down events while hovering over the link.
	 * 
	 * @author JJB25
	 *
	 */
	private class EIFocusKeyHandler implements KeyDownHandler {

		@Override
		public void onKeyDown(KeyDownEvent event) {
			// Key Code 113 = F2 key
			if (event.getNativeKeyCode() == 113 && mouseOver && popup != null) {
				popup.switchFocusMode(true);
			}
		}
	}
	
	/** Handler for click events. Needed to set the mouse over
	 * flag to false and close the popup if it is open.
	 * 
	 * @author JJB25
	 *
	 */
	private class EIPreviewPopUpClickHandler implements ClickHandler {

		@Override
		public void onClick(ClickEvent event) {
			mouseOver = false;
			if (popup != null) {
				popup.hide();
			}
		}
		
	}
}
