package org.eaglei.ui.gwt.instance.widgets;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import static org.eaglei.model.EIOntConstants.PG_CONTACT_LOCATION;
import static org.eaglei.model.EIOntConstants.PG_EMAIL_CONTACT;
import static org.eaglei.model.EIOntConstants.PG_LAB_RELATED;
import static org.eaglei.model.EIOntConstants.PG_PRIMARY;
import static org.eaglei.model.EIOntConstants.PG_SECONDARY;
import static org.eaglei.ui.gwt.instance.EagleIEntityConstants.FOAF_PERSON_URI;
import static org.eaglei.ui.gwt.instance.EagleIEntityConstants.RESOURCE_DESCRIPTION_ENTITY;
import static org.eaglei.ui.gwt.instance.EagleIEntityConstants.SYNONYM_ENTITY;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIProperty;
import org.eaglei.model.SearchEIInstancePreview;
import org.eaglei.model.gwt.rpc.ClientModelManager;
import org.eaglei.model.gwt.rpc.ClientModelManager.PropertyCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.PropertyDefinitionCallback;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.google.gwt.user.client.ui.SimplePanel;

/**
 * Popup class for showing previews of an instance's properties.
 * 
 * @author JJB25
 * 
 */
public class EIInstancePreviewPopup extends PopupPanel implements ValueChangeHandler<String> {

	// Display text constants
	/** Text for the 'more info' Label. */
	private static final String MORE_INFO_LABEL_TEXT1 = "Press F2 to focus.";
	/** Text for the 'more info' Label. */
	private static final String MORE_INFO_LABEL_TEXT2 = "Press F2 to show more information.";
	/** Text for the type property Label. */
	private static final String TYPE_PROPERTY_LABEL_TEXT = "Type: ";
	/** Text for the name property Label. */
	private static final String NAME_PROPERTY_LABEL_TEXT = "Name: ";
	
	// Style name constants
	/** Style name for the 'more info' Label. */
	private static final String MORE_INFO_LABEL_STYLE = "moreInfo";
	/** Style for the property value Labels. */
	private static final String PROPERTY_VALUE_STYLE = "previewPopupRowValue";
	/** Style for the property label Labels. */
	private static final String PROPERTY_LABEL_STYLE = "previewPopupLabel";
	/** Style name for style applied to the entire popup. */
	private static final String POPUP_STYLE = "previewPopup";
	/** Style name for the close button. */
	private static final String CLOSE_BUTTON_STYLE = "previewCloseButton";
	/** Style name for style applied to the entire popup. */
	private static final String POPUP_CONTENT_STYLE = "previewPopupContent";
	/** Style name for style applied to the entire popup. */
	private static final String POPUP_CONTENT_SCROLL_STYLE = "previewPopupContentScroll";
	

	/** Logger for this class. */
	private static Logger log = Logger.getLogger( "EIInstancePreviewPopup" );
	/** Panel containing the entire display of the popup. */
	private final FlowPanel displayPanel;
	/** Panel containing the instance preview portion of the popup. */
	private FlowPanel instancePanel;
	/** Panel containing the close button and focus label. */
	private final SimplePanel controlPanel;
	/** Panel for scrolling the instance preview in focus mode. */
	private ScrollPanel scrollPanel;
	/** Constant defining how many properties will be shown other than name and type. */
	private static final int MAX_DISPLAYED_PROPERTIES = 3;
	/** Optional label that will display if the number of properties exceeds the displayable maximum. */
	private final Label moreInfoLabel;
	/** Label for the label of the name of the object. */
	private final Label nameFieldLabel;
	/** Label for the value of the name of the object. */
	private final Label nameFieldValue;
	/** Label for the label of the type of the object. */
	private final Label typeFieldLabel;
	/** Label for the value of the type of the object. */
	private final Label typeFieldValue;
	/** Flag for focus mode. */
	private boolean focusMode;
	/** Button for closing the popup in focus mode. */
	private Button closeButton;
	/** Preview of the instance retrieved from the repository. */
	private final SearchEIInstancePreview preview;
	/** A list of retrieved properties that has been ordered for display. */
	private ArrayList<EIProperty> orderedProperties;

	/**
	 * Constructor for this class. Creates a name property label Label and value Label, 
	 * a type property label Label and value Label, and initiates server calls to retrieve 
	 * other properties.
	 * 
	 * @param preview The instance preview retrieved from the repository.
	 */
	public EIInstancePreviewPopup(SearchEIInstancePreview preview) {
		super(true);
		this.preview = preview;
		displayPanel = new FlowPanel();
		displayPanel.setStyleName( POPUP_STYLE );
		add(displayPanel);
		controlPanel = new SimplePanel();
		displayPanel.add( controlPanel );
		moreInfoLabel = new Label( MORE_INFO_LABEL_TEXT1 );
		moreInfoLabel.setStyleName( MORE_INFO_LABEL_STYLE );
		controlPanel.add(moreInfoLabel);
		scrollPanel = new ScrollPanel();
		scrollPanel.setAlwaysShowScrollBars(false);
		scrollPanel.setStyleName(POPUP_CONTENT_SCROLL_STYLE);
		nameFieldLabel = new Label( NAME_PROPERTY_LABEL_TEXT );
		nameFieldLabel.setStyleName( PROPERTY_LABEL_STYLE );
		nameFieldValue = new Label( preview.getInstanceLabel() );
		nameFieldValue.setStyleName( PROPERTY_VALUE_STYLE );
		typeFieldLabel = new Label( TYPE_PROPERTY_LABEL_TEXT );
		typeFieldLabel.setStyleName( PROPERTY_LABEL_STYLE );
		typeFieldValue = new Label( preview.getInstanceType().getLabel() );
		typeFieldValue.setStyleName( PROPERTY_VALUE_STYLE );
		closeButton = new Button();
		closeButton.setStyleName(CLOSE_BUTTON_STYLE);
		closeButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				EIInstancePreviewPopup.this.close();
			}
		});
		focusMode = false;
		History.addValueChangeHandler(this);
		initializeInstancePanel();
		if ( preview.getPropertyCount() > 0 ) {
			initializeProperties();
		}
		else {
			// Needed so we don't dereference a null when redisplaying
			orderedProperties = new ArrayList<EIProperty>(0);
		}
	}

	/**
	 * Makes calls to the model service to get a list of properties for a 
	 * given class and then get definitions for those properties. Finally
	 * the property list is ordered, validated, and added to the popup.
	 * 
	 */
	private void initializeProperties() {
		ClientModelManager.INSTANCE.getProperties( preview.getRealInstance().getInstanceClass(), new PropertyCallback() {
			@Override
			public void onSuccess(final EIClass result) {
				ClientModelManager.INSTANCE.getPropertyDefinitions( result.getProperties(), new PropertyDefinitionCallback() {
					@Override
					public void onSuccess(final List<EIProperty> properties) {
						reorderProperties( properties );
						removeInvalidProperties();
						addProperties();
					}

				} );
			}
		} );
	}
	
	/** Iterates through the ordered properties to find invalid
	 * properties. These are then removed from the property data
	 * and from the ordered property list.
	 * 
	 */
	private void removeInvalidProperties() {
		Map<EIEntity, Set<EIEntity>> objectPropMap = preview.getRealInstance().getObjectProperties();
		Set<EIProperty> propsToRemove = new HashSet<EIProperty>();
		for (EIProperty property : orderedProperties) {
			if (propertyIsInvalid(property.getEntity(), objectPropMap)) {
				preview.getRealInstance().getObjectProperties().remove(property.getEntity());
				propsToRemove.add(property);
			}
		}
		orderedProperties.removeAll(propsToRemove);
	}

	/**
	 * Reorders the properties according to property groups, for preferred form display. <br />
	 * <br />
	 * Currently: <br />
	 * - additional name <br />
	 * - resource description <br />
	 * - lab information <br />
	 * - people (based on range of property) <br />
	 * - contact information (based on privacy annotations) <br />
	 * - dates (total hack) <br />
	 * - primary properties of resource type (based on new annotation) <br />
	 * - secondary properties (whatever is left)<br />
	 * <br />
	 * ORDER IS SUBJECT TO CHANGE
	 * 
	 * @param properties
	 *            List of EIProperties to be ordered.
	 * 
	 */
	private void reorderProperties(List<EIProperty> properties) {
		// TODO specify group order in a config file; specify order within a group
		final List<EIProperty> personProperties = new ArrayList<EIProperty>();
		final List<EIProperty> contactProperties = new ArrayList<EIProperty>();
		final List<EIProperty> providerProperties = new ArrayList<EIProperty>();
		final List<EIProperty> dateProperties = new ArrayList<EIProperty>();
		final List<EIProperty> primaryProperties = new ArrayList<EIProperty>();
		final List<EIProperty> secondaryProperties = new ArrayList<EIProperty>();
		final List<EIProperty> restOfProperties = new ArrayList<EIProperty>();
		final Set<EIEntity> dataprops = preview.getRealInstance().getDatatypeProperties().keySet();
		orderedProperties = new ArrayList<EIProperty>( properties.size() );
		for (final EIProperty property : properties) {
			if ( property.getEntity().equals( RESOURCE_DESCRIPTION_ENTITY ) && dataprops.contains( property.getEntity() ) ) {
				orderedProperties.add( 0, property );
				continue;
			}
			if ( property.getEntity().equals( SYNONYM_ENTITY ) && dataprops.contains( property.getEntity() ) ) {
				orderedProperties.add( 0, property );
				continue;
			}
			if ( hasPersonRange( property ) ) {
				personProperties.add( property );
				continue;
			}

			if ( property.getEntity().getLabel().contains( "date" ) ) {
				dateProperties.add( property );
				continue;
			}

			final Set<String> annotations = property.getAnnotations();
			if ( annotations.contains( PG_LAB_RELATED ) ) {
				providerProperties.add( property );
				continue;
			}
			if ( annotations.contains( PG_CONTACT_LOCATION ) || annotations.contains( PG_EMAIL_CONTACT ) ) {
				contactProperties.add( property );
				continue;
			}

			if ( annotations.contains( PG_PRIMARY ) ) {
				primaryProperties.add( property );
				continue;
			}

			if ( annotations.contains( PG_SECONDARY ) ) {
				secondaryProperties.add( property );
				continue;
			}

			restOfProperties.add( property );
		}

		orderedProperties.addAll( providerProperties );
		orderedProperties.addAll( personProperties );
		orderedProperties.addAll( contactProperties );
		Collections.reverse( dateProperties );
		orderedProperties.addAll( dateProperties );
		orderedProperties.addAll( primaryProperties );
		orderedProperties.addAll( secondaryProperties );
		orderedProperties.addAll( restOfProperties );
	}

	/**
	 * Adds all the properties in the ordered property list to the instance panel. 
	 * This includes parsing the property value set to create a string representation 
	 * of the set and creating Labels for both the property labels and the property values.
	 * 
	 */
	private void addProperties() {
		StringBuilder strBuild = new StringBuilder();
		Map<EIEntity, Set<String>> dataPropMap = preview.getRealInstance().getDatatypeProperties();
		Map<EIEntity, Set<EIEntity>> objectPropMap = preview.getRealInstance().getObjectProperties();

		for (int i = 0, count = 0; i < orderedProperties.size() && (focusMode?true:count < MAX_DISPLAYED_PROPERTIES); i++) {
			EIProperty property = orderedProperties.get( i );
			EIEntity propertyEntity = property.getEntity();
			boolean isObjectProp = objectPropMap.containsKey( propertyEntity );
			boolean isDataProp = dataPropMap.containsKey( propertyEntity );
			if ( !isObjectProp && !isDataProp ) {
				// Valid property for this type but no data
				log.log( Level.FINER, "No data found for property: " + propertyEntity.getLabel() );
				continue;
			}

			final Label propertyFieldLabel = new Label( propertyEntity.getLabel() + ": " );
			propertyFieldLabel.setStyleName( PROPERTY_LABEL_STYLE );
			instancePanel.add( propertyFieldLabel );

			Object[] propertyValArry = isObjectProp ? objectPropMap.get( propertyEntity ).toArray() : dataPropMap.get( propertyEntity ).toArray();
			log.log( Level.FINER, propertyValArry.length + " values found for property: " + propertyEntity.getLabel() );
			
			for (int j = 0; j < propertyValArry.length; j++) {
				if ( strBuild.length() > 0 ) {
					if (propertyValArry.length > 2) {
						strBuild.append( ", " );
					}
					if ( j == propertyValArry.length - 1 ) {
						strBuild.append( " and " );
					}
				}
				String val = isObjectProp? ((EIEntity)propertyValArry[j]).getLabel(): (String) propertyValArry[j];
				strBuild.append( InstanceWidgetUtils.formatText( val ) );
			}
			
			log.finer( "adding property value " + strBuild.toString() );
			final Label propertyFieldValue = new Label( strBuild.toString() );
			propertyFieldValue.setStyleName( PROPERTY_VALUE_STYLE );
			instancePanel.add( propertyFieldValue );
			strBuild.delete( 0, strBuild.length() );
			count++;
		}
		if ( !focusMode) {
			if (preview.getPropertyCount() > MAX_DISPLAYED_PROPERTIES) {
				moreInfoLabel.setText(MORE_INFO_LABEL_TEXT2);
			}
			else {
				moreInfoLabel.setText(MORE_INFO_LABEL_TEXT1);
			}	
		}
	}

	/** Checks to see if the passed in property is invalid (i.e. value is null, 
	 * empty String, or URI).
	 * 
	 * @param propertyEntity The property entity to be checked.
	 * @param objectPropMap The map of object properties.
	 * @return True if the property is invalid, false otherwise.
	 */
	private boolean propertyIsInvalid(EIEntity propertyEntity, Map<EIEntity, Set<EIEntity>> objectPropMap) {
		if ( objectPropMap.containsKey(propertyEntity) ) {
			EIEntity[] valueArry = objectPropMap.get(propertyEntity).toArray(new EIEntity[0]);
			EIEntity valueEntity = valueArry[0];
			if ( valueEntity.getLabel() == null || valueEntity.getLabel().equals( valueEntity.getURI().toString() ) || valueEntity.getLabel().equals( "" ) ) {
				log.finer( "property " + propertyEntity.getLabel() + " had invalid value" );
				return true;
			}
		} 
		return false;
	}

	/**
	 * Determines whether the passed in property has a person as the property value range.
	 * 
	 * @param property
	 *            The property whose range is being tested.
	 * @return True if the property range is a person, false otherwise.
	 */
	private boolean hasPersonRange(final EIProperty property) {
		if ( !( property instanceof EIObjectProperty ) ) {
			return false;
		}
		final List<EIClass> ranges = ( (EIObjectProperty)property ).getRangeList();
		if ( ranges.size() == 1 && ranges.get( 0 ).getEntity().getURI().equals( FOAF_PERSON_URI ) ) {
			return true;
		} else {
			return false;
		}
	}

	/** Switches the focus mode of the popup if needed. The focus
	 * mode will determine whether the popup auto-hides, is fully
	 * expanded, and is scrollable.
	 * 
	 * @param shouldBeFocused Flag to indicate whether the popup
	 * should now be in focus mode or not.
	 */
	public void switchFocusMode(boolean shouldBeFocused) {
		if (!focusMode && shouldBeFocused) {
			focusMode = true;
			instancePanel.removeFromParent();
			displayPanel.add(scrollPanel);
			moreInfoLabel.removeFromParent();
			controlPanel.add(closeButton);
			initializeInstancePanel();
			addProperties();
			scrollPanel.add(instancePanel);
			setAutoHideEnabled(false);
		}
		else if (focusMode && !shouldBeFocused) {
			focusMode = false;
			instancePanel.removeFromParent();
			scrollPanel.removeFromParent();
			closeButton.removeFromParent();
			controlPanel.add(moreInfoLabel);
			initializeInstancePanel();
			addProperties();
			displayPanel.add(instancePanel);
			setAutoHideEnabled(true);
		}
	}
	
	/** Convenience method to bypass the normal hide method. Unlike
	 * the implementation of hide this will call the super hide
	 * method regardless of whether the popup is in focus mode.
	 * 
	 */
	private void close() {
		super.hide();
	}

	@Override
	public void hide() {
		if (!focusMode) {
			super.hide();
		}
	}

	@Override
	public void onValueChange(ValueChangeEvent<String> event) {
		close();
	}
	
	private void initializeInstancePanel() {
		instancePanel = new FlowPanel();
		instancePanel.setStyleName(POPUP_CONTENT_STYLE);
		displayPanel.add( instancePanel );
		instancePanel.add( nameFieldLabel );
		instancePanel.add( nameFieldValue );
		instancePanel.add( typeFieldLabel );
		instancePanel.add( typeFieldValue );
	}
}
