package org.eaglei.ui.gwt.search.results;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import org.eaglei.model.EIEntity;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchRequest.DataSet;
import org.eaglei.ui.gwt.ApplicationContext;
import org.eaglei.ui.gwt.ApplicationResources;
import org.eaglei.ui.gwt.SearchApplicationContext;
import org.eaglei.ui.gwt.search.SearchContext;
import org.eaglei.ui.gwt.search.rpc.ClientSearchResultSet;

import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.Hyperlink;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.Widget;

public class ResourcesGrid extends FlowPanel implements NavBar.NavListener {
    
    public static class CellPanel extends FlowPanel {
        
        CellPanel() {
            setStyleName("resultsTableCell");
        }
        
        void setContent(Widget primary, Widget secondary) {
            clear();
            add(primary);
            primary.addStyleName("primary");
            if (secondary != null) {
                add(secondary);
                secondary.addStyleName("secondary");
            }
        }
    }
    
    
    public static class DataSetRenderer {
        
        protected String[] columnHeaderNames;
        protected DataSet dataset;
        protected ResourcesGrid resultsTable;
        
        public DataSetRenderer(ResourcesGrid resultsTable, DataSet dataset, String[] columnHeaderNames) {
            this.resultsTable = resultsTable;
            this.dataset = dataset;
            this.columnHeaderNames = columnHeaderNames;
        }
        
        public DataSet getDataset() {
            return this.dataset;
        }
        
        public String[] getHeaderColumnNames() {
            return columnHeaderNames;
        }
        
        public void displayHeaderRow() {
        	FlowPanel headerRow = new FlowPanel();
            for (String colName : getHeaderColumnNames()) {
            	HTML contents = new HTML("<b>" + colName + "</b>");
            	contents.setStyleName( "listGridHeader" );
            	headerRow.add(contents);
            }
        }
        
        public void createInstanceRow(int row, DataSet dataset, SearchResult searchResult) {           
        	FlowPanel rowPanel = new FlowPanel();
        	rowPanel.setStyleName("ResultPanelRow");
            // Name cell
            Widget instanceLink;
            if (dataset == DataSet.EI_RESOURCES) {
                instanceLink = new Hyperlink(searchResult.getEntity().getLabel(), 
                        SearchApplicationContext.INSTANCE_PAGE_TOKEN + "uri=" + searchResult.getEntity().getURI().toString());
            } else {
                instanceLink = new Anchor(searchResult.getEntity().getLabel(), searchResult.getEntity().getURI().toString());
                ((Anchor) instanceLink).setTarget("_blank"); // display in new tab
            }
            instanceLink.setStyleName("ResultNameLabel");
            
            HTML snippit = null;
            if (searchResult.getHighlight() != null) {
                snippit = new HTML(searchResult.getHighlight());
            }
            CellPanel cp = new CellPanel();
            cp.setContent(instanceLink, snippit);
            rowPanel.add(cp);
            //resultsTable.setWidget(row, col++, cp);

            // Type cell
            Label typeLabel = new Label(searchResult.getType().getLabel());
            Label rootTypeLabel = null;
            EIEntity rootTypeEntity = resultsTable.currentResults.getMapURIToRootEntity().get(searchResult.getType().getURI());
            if (rootTypeEntity != null) {
                rootTypeLabel = new Label(rootTypeEntity.getLabel());
            }
            cp = new CellPanel();
            cp.setContent(typeLabel, rootTypeLabel);
            if (rootTypeLabel != null) {
                rootTypeLabel.addStyleName("rootType");
            }
            //resultsTable.setWidget(row, col++, cp);
            rowPanel.add(cp);
            
            // Location cell
            EIEntity lab = searchResult.getProvider();
            Hyperlink labLink;
            if (lab != null) {
                labLink = new Hyperlink(lab.getLabel(), 
                    SearchApplicationContext.INSTANCE_PAGE_TOKEN + "uri=" + lab.getURI().toString());
                labLink.setStyleName("resultsLabLink");
            } else {
                labLink = null;
            }
            Label instLabel = new Label(searchResult.getInstitution().getLabel());
            cp = new CellPanel();
            if (ApplicationContext.IS_INSTITUTION) {
                if (labLink != null) {
                    // If in institution mode, display Lab as the primary label
                    cp.setContent(labLink, null);
                }
            } else {
                // If in Central mode, display Institution as primary, lab as secondary.
                cp.setContent(instLabel, labLink);
            }
            //resultsTable.setWidget(row, col++, cp);
            rowPanel.add(cp);
            resultsTable.add(rowPanel);
       }
        
 
    }
    
    public static final int PAGE_SIZE = SearchRequest.DEFAULT_PAGE_SIZE;
    private static final int FIRST_RESULT_ROW = 2;
	private static final int BOTTOM_THRESHOLD = 7;
    
	private final Map<DataSet, DataSetRenderer> mapDSToRenderer = new HashMap<DataSet, ResourcesGrid.DataSetRenderer>();;
	private ClientSearchResultSet currentResults;
	private DataSetRenderer currentRenderer;
	private final NavBar navBar1;
	private final InnerNavBar navBar2;
	private Label status;
	private final FlowPanel statusRow;
	private final FlowPanel progressRow;
	
	public ResourcesGrid(NavBar navBar1, final InnerNavBar navBar2) {
        // Header columns
        setStyleName("ResultPanel");
        this.navBar1 = navBar1;
        this.navBar2 = navBar2;
        navBar1.addNavListener(this);
        navBar2.addNavListener(this);
        status = new Label();
        statusRow = new FlowPanel();
        statusRow.add( status ); 
        add(statusRow); 
       
        progressRow = new FlowPanel();
        progressRow.add( new Image(ApplicationResources.INSTANCE.loading()) );
        add(progressRow); 
	}
	
	public ClientSearchResultSet getCurrentResults() {
	    return this.currentResults;
	}
	
	public void addRenderer(DataSetRenderer renderer) {
        this.mapDSToRenderer.put(renderer.getDataset(), renderer);
	}
	
	public void displaySearchPending() {
        currentResults = null;
        navBar1.setVisible(false);
        navBar2.setVisible(false);
        statusRow.setVisible( false );
        progressRow.setVisible( true );
	    for (int i = getWidgetCount()-1; i>=FIRST_RESULT_ROW; i--) {
	        remove(i);
	    }
	}

	public void displayStatusMessage(String message) {
		displayStatusMessage(message, false);
	}
	
	public void displayStatusMessage(String message, boolean showNavBar) {
        status.setText(message);
        currentResults = null;
        navBar1.setVisible(showNavBar);
        navBar2.setVisible(false);
        statusRow.setVisible( true );
        progressRow.setVisible( false );
        for (int i = getWidgetCount()-1; i>=FIRST_RESULT_ROW; i--) {
            remove(i);
        }
	}
	
	public void displaySearchResults(ClientSearchResultSet results) {
		if ( results.getResultSet().getRequest().getDataset() == DataSet.NIF_RESOURCES ) {
			displayStatusMessage("This feature (NIF search) is temporarily unavailable.", false);
			return;
		}
		
		if (results.getResultSet().getResults().size() == 0) {
	        // Update the nav bars.
	        navBar1.update(results);
	        navBar2.update(results);
	        if ( results.getResultSet().getResults().size() < BOTTOM_THRESHOLD ) {
	        	navBar2.setVisible( false );
	        }
            displayStatusMessage("No results found.", true);
            return;
        }
                
        if (currentRenderer == null 
                || currentRenderer.getDataset() != results.getResultSet().getRequest().getDataset()) {
            currentRenderer = mapDSToRenderer.get(results.getResultSet().getRequest().getDataset());
            currentRenderer.displayHeaderRow();
        }
        
	    currentResults = results;
 	    progressRow.setVisible( false );

        // Update the nav bars.
        navBar1.update(results);
        navBar2.update(results);
        navBar1.setVisible(true);
        navBar2.setVisible(results.getResultSet().getResults().size() >= BOTTOM_THRESHOLD);
       
        // Add the results
        statusRow.setVisible( false	);
        int row = FIRST_RESULT_ROW;
        DataSet dataset = results.getResultSet().getRequest().getDataset();
        for (SearchResult result : results.getResultSet().getResults()) {
            currentRenderer.createInstanceRow(row++, dataset, result);
        }
        
	}
	
    @Override
    public void onPreviousPage() {
        // Move back a page.
        int startIndex = currentResults.getResultSet().getStartIndex() - PAGE_SIZE;
        assert(startIndex >= 0) : "paged beyond the start of the result list";
        executeNewPageSearch(startIndex);
    }

    @Override
    public void onNextPage() {
        // Move forward a page.
        int startIndex = currentResults.getResultSet().getStartIndex() + PAGE_SIZE;
        assert(startIndex < currentResults.getResultSet().getTotalCount()) : "paged beyond the end of the result list";
        executeNewPageSearch(startIndex);
    }

    private void executeNewPageSearch(int startIndex) {
		SearchRequest newRequest = new SearchRequest(currentResults.getResultSet().getRequest().toURLParams());
        newRequest.setStartIndex(startIndex);
        SearchContext.INSTANCE.search(newRequest);
    }

}
