package org.eaglei.ui.gwt.search.server;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.search.harvest.PollingDataHarvester;
import org.eaglei.search.logging.AsynchronousLoggerCount;
import org.eaglei.search.logging.AsynchronousLoggerSearch;
import org.eaglei.search.provider.ClassCountResult;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchProvider;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.repository.RepositoryProviderException;
import org.eaglei.services.repository.SecurityProvider;
import org.eaglei.ui.gwt.rpc.InvalidSessionIdException;
import org.eaglei.ui.gwt.rpc.LoggedException;
import org.eaglei.ui.gwt.search.rpc.ClientSearchResultSet;
import org.eaglei.ui.gwt.search.rpc.SearchServiceRemote;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

/**
 * Implementation of SearchService.
 */
public class SearchServlet extends RemoteServiceServlet implements SearchServiceRemote {

    /**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	private static final Log logger = LogFactory.getLog(SearchServlet.class);
    private static final boolean DEBUG = logger.isDebugEnabled(); 
    
    /*
     * Provider supports execution of search requests (through network or directly against local repo)
     */
    private SearchProvider searchProvider;
    private PollingDataHarvester dataHarvester;
    private SecurityProvider securityProvider;
    private EIOntModel eagleiOntModel;
    private InstitutionRegistry institutionRegistry;
    private List<EIClass> searchCategoryClasses;
    private HashSet<EIURI> searchCategoryURIs;
 
    private AsynchronousLoggerSearch asyncLoggerSearch;
    private static final String SEARCH_LOG_TABLE_NAME = "SEARCH_SERVLET_LOG";
    private AsynchronousLoggerCount asyncLoggerCount;
    private static final String COUNT_LOG_TABLE_NAME = "SEARCH_SERVLET_COUNT_LOG";
    
    private String ontologyVersion = "";

    public SearchServlet() {
    }
    
    @Override
    public void init() {
        WebApplicationContext ctx = 
            WebApplicationContextUtils.getWebApplicationContext(getServletContext());
        searchProvider = ctx.getBean("rootSearchProvider", SearchProvider.class);
        securityProvider = ctx.getBean("securityProvider", SecurityProvider.class);
        eagleiOntModel = ctx.getBean(EIOntModel.class);
        institutionRegistry = ctx.getBean(InstitutionRegistry.class);
        List<String> searchCategories = (List<String>) ctx.getBean("searchCategories");
        searchCategoryClasses = new ArrayList<EIClass>(searchCategories.size());
        searchCategoryURIs = new HashSet<EIURI>(searchCategories.size());
        for (String uriStr : searchCategories) {
            EIURI uri = EIURI.create(uriStr);
            searchCategoryClasses.add(eagleiOntModel.getClass(uri));
            searchCategoryURIs.add(uri);
        }
        try {
            searchProvider.init();
            logger.info("SearchProvider created and initialized");
        } catch (Throwable t) {
            logger.error("Initialization of search provider failed", t);
            throw new RuntimeException("Initialization of search provider failed", t);
        }

        // Create asynchronous logger         	
        String buildID = "Unknown";
        InputStream inputStream = null;
        // Get build ID from properties file created during build.
        try {
        	inputStream = ctx.getResource("/buildversion.properties").getInputStream();
        	
        	//TEMPORARY
        	if (inputStream == null) {
        		logger.info("/buildversion.properties not found");
        		inputStream = ctx.getResource("/institution/buildversion.properties").getInputStream();
        		if (inputStream == null) {
        			logger.info("/institution/buildversion.properties not found");
        		}
        	}
        	//TEMPORARY
        	
	        // If we find the file, set the build ID
	        if (inputStream != null) {
				Properties props = new Properties();
				props.load(inputStream);
				buildID = props.getProperty("buildversion", "Unknown");
			} 
        } catch(IOException ioe) {
			logger.error("Failed to find buildversion.properties");
        }

     // DEBUG
        logger.debug("SearchServlet: Context Display name = " + ctx.getDisplayName());
        logger.debug("SearchServlet: Context ID = " + ctx.getId());
        if (ctx.getParent() == null) {
        	logger.debug("SearchServlet: Context Parent is null.");
        } else {
        	logger.debug("SearchServlet: Context Parent Display name = " + ctx.getParent().getDisplayName());
        }
        if (ctx.getServletContext() == null) {
        	logger.debug("SearchServlet: ServletContext is null.");
        } else {
	        logger.debug("SearchServlet: ServletContext path = " + ctx.getServletContext().getContextPath());
	        logger.debug("SearchServlet: ServletContext context name = " + ctx.getServletContext().getServletContextName());
	    }
        //DEBUG
        
        ontologyVersion = eagleiOntModel.getVersion();
        asyncLoggerSearch = new AsynchronousLoggerSearch(SEARCH_LOG_TABLE_NAME, buildID, ontologyVersion);
        asyncLoggerCount = new AsynchronousLoggerCount(COUNT_LOG_TABLE_NAME, buildID, ontologyVersion);

        dataHarvester = (PollingDataHarvester) ctx.getBean("dataHarvester");
        dataHarvester.startPolling();
    }

    @Override
    public void destroy() {
        //TODO does Spring need to be gracefully shutdown?
    }

    /* (non-Javadoc)
     * @see org.eaglei.search.client.rpc.SearchService#getInstitutions(java.lang.String)
     */
    public List<EIEntity> getInstitutions() throws LoggedException, InvalidSessionIdException {
        //SessionManager.validate(sessionId);
        try {
           
            return institutionRegistry.getInstitutions();
           
        } catch (Throwable t) {
            logger.error("Error in getInstitutions", t);
            throw new LoggedException(t.getLocalizedMessage());
        }        
    }
    
    public List<EIClass> getTopLevelSearchCategories() throws LoggedException {
        //SessionManager.validate(sessionId);
        try {
           
            return searchCategoryClasses;
           
        } catch (Throwable t) {
            logger.error("Error in getTopLevelSearchCategories", t);
            throw new LoggedException(t.getLocalizedMessage());
        }        
    }

    
    /* (non-Javadoc)
     * @see org.eaglei.search.client.rpc.SearchService#getSearchResults(java.lang.String, org.eaglei.search.request.SearchRequest)
     */
    public ClientSearchResultSet search(final String sessionId, final SearchRequest request) throws InvalidSessionIdException, LoggedException {
    	try {
        	securityProvider.isValid(sessionId, true);
        }
        catch (RepositoryProviderException e) {
        	//TODO: Should this be InvalidSessionIdException or just reuse RepositoryProviderException?
        	throw new InvalidSessionIdException("Error validating session.", e);
        }
        
        if (!dataHarvester.hasInitialData()) {
        	throw new LoggedException("Search application is initializing...  Please retry in a few minutes.");
        }

        long start = System.currentTimeMillis();

        try {
            // TODO session id errors must be propagated
            // to the client as a checked exception or error
            // code in the return object.
            if (request == null) {
                logger.error("Null search request");
                throw new LoggedException();
            }
            if (DEBUG) {
                logger.debug("Search request:  " + request.toString());
            }

            // execute the search
            SearchResultSet resultSet = searchProvider.query(request);
            
            // NOTE: the following could probably be deleted once we fully convert
            //       to the new CountResult api.
            // Lookup the entity for the type binding.
            EIEntity bindingEntity =
                (request.getBinding() != null) ? eagleiOntModel.getClass(request.getBinding().getType()).getEntity() : null;
            
            // Compute the tab category of the binding
            EIURI bindingCategoryURI = null;
            if (request.getBinding() != null) {
                if (searchCategoryURIs.contains(request.getBinding().getType())) {
                    bindingCategoryURI = request.getBinding().getType();
                } else {
                    List<EIClass> superclasses = eagleiOntModel.getSuperClasses(request.getBinding().getType());
                    for (int i=superclasses.size()-1; i>=0; i--) {
                        EIURI superclassURI = superclasses.get(i).getEntity().getURI();
                        if (searchCategoryURIs.contains(superclassURI)) {
                            bindingCategoryURI = superclassURI;
                            break;
                        }
                    }
                }
            }
            
            // Generate a lookup table for the root class of each search result type
            HashMap<EIURI, EIEntity> mapURIToRootEntity = new HashMap<EIURI, EIEntity>();
            for (SearchResult result : resultSet.getResults()) {
                List<EIClass> superclasses = eagleiOntModel.getSuperClasses(result.getType().getURI());
                if (superclasses.size() > 0) {
                    mapURIToRootEntity.put(result.getType().getURI(), superclasses.get(superclasses.size()-1).getEntity());
                }
            }

            // Log entry for this search
            long end = System.currentTimeMillis();
            long count = resultSet.getTotalCount();
            asyncLoggerSearch.log(securityProvider.getSession(sessionId), 
            		start, end, request, count);
            
            /*
            for (SearchResult result : resultSet.getResults()) {
            	logger.debug("Result: " + result.getEntity().getLabel() +"  score: " + result.getRank());
            }
*/

            return new ClientSearchResultSet(resultSet, bindingEntity, bindingCategoryURI, mapURIToRootEntity);
           
        } catch (Throwable t) {
            logger.error("Unexpected error in search: " + request.toString(), t);
            throw new LoggedException(t.getLocalizedMessage());
        }
    }
    
    public SearchCounts count(final String sessionId, SearchCountRequest request) throws InvalidSessionIdException, LoggedException {
    	try {
        	securityProvider.isValid(sessionId, true);
        }
        catch (RepositoryProviderException e) {
        	//TODO: Should this be InvalidSessionIdException or just reuse RepositoryProviderException?
        	throw new InvalidSessionIdException("Error validating session.", e);
        }
        
        if (!dataHarvester.hasInitialData()) {
        	throw new LoggedException("Search application is initializing...  Please retry in a few minutes.");
        }

        try {
            // TODO session id errors must be propagated
            // to the client as a checked exception or error
            // code in the return object.
            if (request == null) {
                logger.error("Null search request");
                throw new LoggedException();
            }
            if (DEBUG) {
                logger.debug(request.toString());
            }

            // Implementing server-side for a little perf benefit.
            for (EIClass c : searchCategoryClasses) {
                EIURI uri = c.getEntity().getURI();
                if (request.getRequest().getBinding() == null 
                        || ! uri.equals(request.getRequest().getBinding().getType())) {
                    request.addCountType(uri);
                }
            }
            if (request.getRequest().getBinding() != null) {
                request.addCountType(null);  // All results
            }
            
            // execute the search
            return searchProvider.count(request);                
           
        } catch (Throwable t) {
            logger.error("Unexpected error in count: " + request.toString(), t);
            throw new LoggedException(t.getLocalizedMessage());
        }        
    }

    @Override
    public ClassCountResult getResourceCounts(String sessionId, SearchRequest request)
            throws InvalidSessionIdException, LoggedException {
    	try {
        	securityProvider.isValid(sessionId, true);
        }
        catch (RepositoryProviderException e) {
        	//TODO: Should this be InvalidSessionIdException or just reuse RepositoryProviderException?
        	throw new InvalidSessionIdException("Error validating session.", e);
        }
        
        if (!dataHarvester.hasInitialData()) {
        	throw new LoggedException("Search application is initializing...  Please retry in a few minutes.");
        }

        try {
            if (DEBUG) {
                logger.debug("Count request:  " + request.toString());
            }

            // execute the search
            long start = System.currentTimeMillis();
            ClassCountResult result = searchProvider.getResourceCount(request); 
            
            // Log entry for this search
            long end = System.currentTimeMillis();
            asyncLoggerCount.log(securityProvider.getSession(sessionId), start, end, request);
            
            return result;
           
        } catch (Throwable t) {
            logger.error("Unexpected error in count: " + request.toString(), t);
            throw new LoggedException(t.getLocalizedMessage());
        }        
    }

    @Override
    public ClassCountResult getProviderTypeCounts(String sessionId, SearchRequest request)
            throws InvalidSessionIdException, LoggedException {
    	try {
        	securityProvider.isValid(sessionId, true);
        }
        catch (RepositoryProviderException e) {
        	//TODO: Should this be InvalidSessionIdException or just reuse RepositoryProviderException?
        	throw new InvalidSessionIdException("Error validating session.", e);
        }
        
        if (!dataHarvester.hasInitialData()) {
        	throw new LoggedException("Search application is initializing...  Please retry in a few minutes.");
        }

        try {
            ClassCountResult result = searchProvider.getProviderTypeCount(request); 
            return result;
           
        } catch (Throwable t) {
            logger.error("Unexpected error in provider type count: " + request.toString(), t);
            throw new LoggedException(t.getLocalizedMessage());
        }        
    }

}
