package org.eaglei.ui.gwt.suggest.server;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.lexical.EntityMatch;
import org.eaglei.lexical.EntityMatchRequest;
import org.eaglei.lexical.SuggestionProvider;
import org.eaglei.lexical.lucene.EntityMatchImpl;
import org.eaglei.model.EIURI;
import org.eaglei.search.harvest.DataHarvester;
import org.eaglei.search.provider.AutoSuggestConstants;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;


/**
 * Dataset-based auto-suggest servlet implementation.
 * 
 * @author tbashor
 */
public class DataSuggestServlet extends HttpServlet {

	private static final long serialVersionUID = 1L;
	
	private static final Map<String, String> mapURIToPlural = new HashMap<String, String>() {
		private static final long serialVersionUID = 184521309834214679L;
		{
			put( "http://purl.obolibrary.org/obo/ERO_0000004", "in Instruments" );
			put( "http://purl.obolibrary.org/obo/ERO_0000005", "in Services" );
			put( "http://purl.obolibrary.org/obo/OBI_0000272", "in Protocols" );
			put( "http://purl.obolibrary.org/obo/ERO_0000071", "in Software" );
			put( "http://purl.obolibrary.org/obo/ERO_0000006", "in Reagents" );
			put( "http://purl.obolibrary.org/obo/ERO_0000015", "in Human Studies" );
			put( "http://purl.obolibrary.org/obo/OBI_0100026", "in Organisms and Viruses" );
			put( "http://purl.obolibrary.org/obo/ERO_0000595", "in Research Opportunities" );
			put( "http://purl.obolibrary.org/obo/ERO_0000020", "in Biological Specimens" );
			put( "http://purl.org/obo/owl/SO%23SO_0001059", "in Genetic Alterations" );
			put( "http://purl.obolibrary.org/obo/ERO_0000002", "in Core Laboratories" );
		}
	};
	
    protected static final Log logger = LogFactory.getLog(DataSuggestServlet.class);
    protected static final boolean DEBUG = logger.isDebugEnabled(); 

	/*
	 * Suggestion provider.
	 */
    private SuggestionProvider provider;
    
    private DataHarvester dataHarvester;

    /**
     * Default constructor. 
     */
    public DataSuggestServlet() {
        // TODO Auto-generated constructor stub
    }
    
    @Override
    public void init() {
        WebApplicationContext ctx = 
            WebApplicationContextUtils.getWebApplicationContext(getServletContext());
        provider = (SuggestionProvider) ctx.getBean("autoSuggestProvider");
        dataHarvester = (DataHarvester) ctx.getBean("dataHarvester");
    }
    
	/**
	 * @see HttpServlet#doGet(HttpServletRequest request, HttpServletResponse response)
	 */
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
	    doPost(request, response);
	}

	/**
	 * @see HttpServlet#doPost(HttpServletRequest request, HttpServletResponse response)
	 */
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		EntityMatchRequest entityRequest = null;
		try {
	        String id = request.getParameter("id");
	        String asyncCallback = request.getParameter("callback");
	        
	        entityRequest = createRequest(request);
	        entityRequest.setMaxMatches(50);
	        
	        //String institution = request.getParameter(AutoSuggestConstants.RESOURCE_PROVIDER_INSTITUTION_PARAM);
	        List<EntityMatch> suggestions = null;
	        if (dataHarvester.hasInitialData()) {
	        	suggestions = provider.suggest(entityRequest);
	        }
	        
	        StringBuilder sb = new StringBuilder();
	        sb.append(asyncCallback);
	        sb.append("({");  // function wrap start
	        writeJsonNameValueString(sb, "id", id);
	        sb.append(',');
	        /*  Commenting out return of the user query input.
	         *  It is only used for debugging, and putting user input
	         *  into json is a potential security risk.
	        writeJsonNameValueString(sb, "q", q);
	        sb.append(',');
	        */
	        sb.append("suggestions");
	        sb.append(":[");
	        if (suggestions != null && suggestions.size() > 0) {
	            int i=0;
	            while (true) {
	                EntityMatchImpl suggestion = (EntityMatchImpl) suggestions.get(i++);
	                String bindingURI = suggestion.getBindingType() != null ? suggestion.getBindingType() : "<null>";
	                String bindingLabel;
	                if (suggestion.getBindingType() != null) {
	                	bindingLabel = mapURIToPlural.get(suggestion.getBindingType());
	                	if (bindingLabel == null) {
	                		// Not found in the lookup
	                		// Display URI so we see the error.
	                		bindingLabel = "in " + suggestion.getBindingType();
	                		logger.error("No plural display label found for binding uri: " + suggestion.getBindingType());
	                	}
	                } else {
	                	bindingLabel = "<null>";
	                }
	                sb.append("\"");
	                sb.append(suggestion.getHighlight());
	                sb.append("\"");
	                sb.append(',');
	                writeJsonValueString(sb, suggestion.getMatchLabel());
	                sb.append(',');
	                //writeJsonValueString(sb, suggestion.getEntity() != null ? suggestion.getEntity().getURI().toString() : "<null>");
	                writeJsonValueString(sb, bindingURI);
	                sb.append(',');
	                writeJsonValueString(sb, bindingLabel);
	                if (i == suggestions.size()) {
	                    break;
	                } else {
	                    sb.append(',');
	                }
	            }
	        }
	        sb.append("]");
	        sb.append("})");  // function wrap end
	        String output = sb.toString();
	        
	        response.setContentType("text/javascript");
	        response.setCharacterEncoding("UTF-8");
	        response.addHeader("Pragma", "no-cache");
	        response.setStatus(200);
	        
	        PrintWriter out = response.getWriter();
	
	        out.println(output);
		} catch (Throwable t) {
			logger.error("Unexpected error getting suggestions: request: " + 
					entityRequest != null ? entityRequest.toString() : "null", t);
			throw new ServletException(t);
		}
	}
	
    private static void writeJsonNameValueString(StringBuilder sb, String name, String value) {
        sb.append(name);
        sb.append(":");
        writeJsonValueString(sb, value);
    }

    private static void writeJsonValueString(StringBuilder sb, String value) {
        sb.append("\"");
        sb.append(value);
        sb.append("\"");
    }
    
    private static EntityMatchRequest createRequest(HttpServletRequest httpRequest) {
        String q = httpRequest.getParameter(AutoSuggestConstants.QUERY_PARAM);
        // Needed to bracket the query with some delimiter, otherwise trailing whitespace is lost.
        // Strip the brackets here, we don't need them for the json response.
        q = q.substring(1, q.length() - 1);
        //System.out.println("encoded: "+q);
        try {
        	// Necessary since TomCat will automatically consider the parameter to be ISO-8859-1 encoded
        	// even though it is UTF-8 encoded. Converting to bytes and then back to a UTF-8 encoded string
        	// solves the problem.
			q = new String(q.getBytes("ISO-8859-1"), "UTF-8");
		} catch (UnsupportedEncodingException e1) {
			e1.printStackTrace();
		}
        //System.out.println("decoded: "+q);
        String classid = httpRequest.getParameter(AutoSuggestConstants.BINDING_TYPE_PARAM);
        EntityMatchRequest searchRequest = null;
        if (classid == null) {
            searchRequest = new EntityMatchRequest(q);
        } else {                
            String decodedClassId;
			try {
				decodedClassId = URLDecoder.decode(classid, "UTF-8");
	            searchRequest = new EntityMatchRequest(q, EIURI.create(decodedClassId));
			} catch (UnsupportedEncodingException e) {
				logger.error(e);
			}
        }
        String property = httpRequest.getParameter(AutoSuggestConstants.PROPERTY_PARAM);
        if (property != null) {
        	searchRequest.setProperty(property);
        }
    	return searchRequest;
    }

}
