/*
 * Copyright (c) 2015-2016, President and Fellows of Harvard College
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

function Column(options) {
    this.dbColumn = options.dbColumn;
    this.columnName = options.columnName;
    this.columnWidth = options.width;
    this.headerClass = options.headerClass;
    this.rowElementClass = options.rowElementClass;
    this.extractDataFunction = options.extractDataFunction;
    this.defaultSortOrder = options.defaultSortOrder;
    this.rowElementHyperLink = options.rowElementHyperLink;
    this.columnType = options.columnType ? new options.columnType(this) : new Column.Text(this);
    this.filter = options.filter;
    this.id = options.id;
    this.orderBy = options.defaultSortOrder;
    this.onExpand = options.onExpand;

    this.onChange = options.onChange;

    this.getSortingClassAssign = function() {
        return 'class="' + this.getSortingClass() + '"';
    };

    this.getSortingClass = function() {

        // sortable iff dbColumn is defined
        if (this.dbColumn === undefined) {
            return "not_sortable";
        }

        var ascOrDesc = this.getCurrentOrderBy();

        if (ascOrDesc === Column.DESC) {
            return Column.sorting_desc;
        }
        else if (ascOrDesc === Column.ASC) {
            return Column.sorting_asc;
        }
        else {
            return Column.sortable;
        }
    };

    this.refreshClass = function() {
        $('#' + this.getTdId()).removeClass();
        $('#' + this.getTdId()).addClass(this.getSortingClass());
    };

    this.getDefaultOrderBy = function() {
        if (this.defaultAscOrDesc === Column.ASC) {
            return Column.ASC;
        }
        else {
            return Column.DESC;
        }
    };

    this.getCurrentOrderBy = function() {
        return this.orderBy;
    };
    this.setCurrentOrderBy = function(value) {
        this.orderBy = value;
    };

    this.toggleOrInitOrderBy = function() {

        if (this.getCurrentOrderBy() === Column.ASC) {
            this.setCurrentOrderBy(Column.DESC);

        } else { // either DESC or undefined
            this.setCurrentOrderBy(Column.ASC);
        }
        return this.getCurrentOrderBy();
    };

    this.getColumnNameString = function() {
        if (this.columnName) {
            return "<strong>" + this.columnName + "</strong>";
        }
        else {
            return "";
        }
    };

    this.getColumnName = function() {
        if (this.columnName) {
            return this.columnName;
        }
        else {
            return "";
        }
    };

    this.getHeaderClass = function() {
        if (this.headerClass) {
            return this.headerClass;
        }
        else {
            return "";
        }
    };

    this.getRowElementClass = function() {
        return this.rowElementClass || "";
    };

    this.getColumnWidth = function() {
        return this.columnWidth ||  "";
    };

    this.getExtractDataFunction = function(value) {
        if (this.extractDataFunction) {
            return this.extractDataFunction(value);

        }
        else {
            return value;
        }
    };

    this.getType = function() {
        return this.columnType;
    };

    this.getTdId = function() {
        return this.id;
    };

    this.generateTableHeadElement = function(tableRow, onSort) {
        var tdElement = $("<td></td>", {id: this.getTdId()});
        if(this.getColumnWidth())
        {
            tdElement.css({"width": this.getColumnWidth() + "%"});
        }
        tableRow.append(tdElement);

        this.columnType.renderHeader(tdElement, onSort);
    };

    this.generateTableElement = function(val, currentSelected, tableRow) {
        var column = this;
        var tdElement = $('<td></td>');
        tableRow.append(tdElement);

        this.columnType.render(tdElement, val);
    };
}

/*
 * Creates a column that contains a link
 */
Column.Hyperlink = function(column) {
    this.basicColumnData = column;
    this.options = {};

    this.setOptions = function(options)
    {
        this.options = options;
    };

    this.createRowElementHyperLink = function (hyperlink) {
        return "javascript:" + hyperlink;
    };

    this.renderHeader = function(tdElement) {};

    this.render = function (tdElement, val) {
        var linkText = this.basicColumnData.extractDataFunction(val);
        var hyperlink = this.basicColumnData.rowElementHyperLink ? this.basicColumnData.rowElementHyperLink(val) : "";
        var hrefElement = $('<a></a>', {
            href: this.createRowElementHyperLink(hyperlink),
            text: linkText
        }).appendTo(tdElement);
        tdElement.addClass(this.basicColumnData.getRowElementClass());
    };
};

/*
 * Creates a column that can contain any text or html
 */
Column.Text = function(column) {
    this.basicColumnData = column;
    this.options = {};

    this.setOptions = function(options)
    {
        this.options = options;
    };

    this.renderHeader = function(tdElement, onSort)
    {
        var headerColumn = this;
        tdElement.addClass(this.basicColumnData.getSortingClass());
        tdElement.addClass(this.basicColumnData.getHeaderClass());
        tdElement.html(this.basicColumnData.getColumnNameString());
        tdElement.click(function()
        {
            onSort(headerColumn.basicColumnData);
        });
    };

    this.render = function (tdElement, val) {
        var value = this.basicColumnData.extractDataFunction(val);
        tdElement.html(value);
    };
};

/*
 * Creates a column that contains a checkbox
 */
Column.Checkbox = function(column) {
    this.basicColumnData = column;
    this.options = {};

    this.setOptions = function(options)
    {
        this.options = options;
    };

    this.renderHeader = function (tdElement) {
        var checkboxElement = $("<input/>", {
            id: this.getHeaderElementId(),
            type: "checkbox"
        });
        tdElement.append(checkboxElement);
        var checkboxColumn = this;
        checkboxElement.click(function() {
            var checked = $(this).prop("checked");
            if(checked) {
                checkboxColumn.checkAll();
            }
            else {
                checkboxColumn.uncheckAll();
            }
        });
    };

    this.getHeaderElementId = function() {
        return this.basicColumnData.getTdId() + "-selectAll";
    };

    this.getRowElementClass = function() {
       return this.basicColumnData.getTdId() + "-checkbox";
    };

    this.render = function (tdElement, val) {
        var inputElement = $("<input/>", {
            type: "checkbox",
            class: this.getRowElementClass()
        });

        tdElement.append(inputElement);

        var checkboxColumn = this;
        inputElement.click(function()
        {
            var checked = $(this).prop("checked");
            checkboxColumn.updateHeaderCheckbox(checked);
        });

        inputElement.change(function()
        {
            var checked = $(this).prop("checked");
            var value = checkboxColumn.basicColumnData.extractDataFunction(val);
            checkboxColumn.options.onSelect(value, checked);
            checkboxColumn.basicColumnData.onChange();
        });
    };

    this.updateHeaderCheckbox = function(checked){
        var headerElementId = this.getHeaderElementId();

        var hasUnselectedCheckBoxes = false;

        $("." + this.getRowElementClass()).each(function() {
            var checked = $(this).prop("checked");

            if(!checked) {
                hasUnselectedCheckBoxes = true;
            }
        });

        if(!hasUnselectedCheckBoxes) {
            $("#" + headerElementId).prop("checked", true);
        }
        else {
            $("#" + headerElementId).prop("checked", false);
        }
    };

    this.checkAll = function() {
        $("." + this.getRowElementClass()).prop("checked", true);
        $("." + this.getRowElementClass()).change();
    };

    this.uncheckAll = function() {
        $("." + this.getRowElementClass()).prop("checked", false);
        $("." + this.getRowElementClass()).change();
    }
};

/*
 * Creates a column with a plus/minus toggle that allows a row below to be
 * expanded and collapsed
 */
Column.Expandable = function(column) {
    this.basicColumnData = column;

    this.options = {};

    this.setOptions = function(options)
    {
        this.options = options;
    };

    this.renderHeader = function(tdElement){};

    this.render = function(tdElement, val){
        var expandColumn = this;

        var toggleImage = $("<img/>", {
            src: expandOrCollapseImgUrls.EXPAND,
            title: expandOrCollapseImgTitle.EXPAND
        }).appendTo(tdElement);

        var value = this.basicColumnData.extractDataFunction(val);
        toggleImage.click(function() {
            var src = $(this).attr("src");

            if (src === expandOrCollapseImgUrls.EXPAND) {
                $(this).attr("src", expandOrCollapseImgUrls.COLLAPSE);
                expandColumn.options.onExpand(tdElement, value);
            }
            else {
                $(this).attr("src", expandOrCollapseImgUrls.EXPAND);
                expandColumn.options.onCollapse(tdElement, value);
            }
        });
    }
};

Column.ASC = 'ASC';
Column.DESC = 'DESC';

Column.sortable = "sortable";
Column.sorting_desc = "sorting_desc";
Column.sorting_asc = "sorting_asc";





