package net.shrine.sheriff.controller;

import edu.harvard.med.cbmi.auth.Ecauthnport;
import edu.harvard.med.cbmi.auth.Ecauthnsvc;
import net.shrine.serializers.HTTPClient;
import org.apache.log4j.Logger;
import org.spin.extension.JDOMTool;
import org.spin.query.message.identity.IdentityServiceException;

import java.util.List;

/**
 * [ SUMMARY ]
 * <p/>
 * [ Author ]
 * Andrew McMurry
 * Date: Jul 13, 2009
 * <p/>
 * Harvard Medical School Center for BioMedical Informatics
 *
 * @link http://cbmi.med.harvard.edu
 * <p/>
 * [ In partnership with ]
 * @link http://chip.org
 * @link http://lcs.mgh.harvard.edu
 * @link http://www.brighamandwomens.org
 * @link http://bidmc.harvard.edu
 * @link http://dfhcc.harvard.edu
 * @link http://spin.nci.nih.gov/
 * <p/>
 * <p/>
 * -------------------------------------------------
 * [ Licensing ]
 * All works licensed by the Lesser GPL
 * @link http://www.gnu.org/licenses/lgpl.html
 * -------------------------------------------------
 */
public class AuthFacade
{
    private static final Logger log = Logger.getLogger(AuthFacade.class);

    private static AuthFacade instance;

    ///**
    // * Ecommons Authentication Client
    // */
    private final String application;
    private final String issuer;
    private final String profilesURL;

    private final Ecauthnsvc service;
    private final Ecauthnport port;

    /**
     * PROFILES API request template XML message
     */
    public static final String PROFILES_REQUEST = "<Profiles Operation=\"GetPersonList\" xmlns=\"http://connects.profiles.schema/profiles/query\">"
            + "<QueryDefinition><PersonID>%s</PersonID></QueryDefinition>"
            + "  <OutputOptions SortType=\"LastFirstName\" StartRecord=\"1\" MaxRecords=\"1\" /></Profiles>";


    public static AuthFacade createInstance(String application, String issuer, String profilesURL)
    {
        instance = new AuthFacade(application, issuer, profilesURL);
        return instance;
    }

    public static AuthFacade getInstance()
    {
        return instance;
    }

    private AuthFacade(String application, String issuer, String profilesURL)
    {
        this.application = application;
        this.issuer = issuer;
        this.profilesURL = profilesURL;
        this.service = new Ecauthnsvc();
        this.port = service.getEcauthnport();
    }
    /**
     * PROFILES XML Namespace
     */
    public static final String PROFILES_NAMESPACE = "xmlns=\"http://connects.profiles.schema/profiles/personlist\"";

    // ---

    public boolean authenticateEcommons(String eCommonsID, String eCommonsPWD) throws IdentityServiceException
    {
        return this.port.authenticate(application, issuer,
                eCommonsID, eCommonsPWD);
    }

    public boolean authorizeFacultyStatus(String eCommonsID) throws IdentityServiceException
    {
        return authorizeFacultyStatus(eCommonsID, profilesURL);
    }

    public boolean authorizeFacultyStatus(String eCommonsID, String url) throws IdentityServiceException
    {
        boolean isApproved = false;

        try
        {
            String requestXML = String.format(PROFILES_REQUEST, eCommonsID);

            log.debug("authorizeFacultyStatus() posting to " + url + " " + requestXML);

            String responseXML = HTTPClient.post(requestXML, url);

            log.debug("authorizeFacultyStatus() returns: " + responseXML);

            // Stripping namespace
            responseXML = responseXML.replaceFirst(PROFILES_NAMESPACE, "");

            JDOMTool jdom = new JDOMTool(responseXML);

            List<String> facultyTypes = jdom.getElementValues("//FacultyType");

            if(facultyTypes != null)
            {
                log.debug("FacultyStatus found , count=" + facultyTypes.size());

                for(String facultyType : facultyTypes)
                {
                    log.debug("//FacultyType " + facultyType);

                    if(facultyType.toUpperCase().contains("FACULTY"))
                    {
                        isApproved = true;
                    }
                    if(facultyType.toUpperCase().contains("PROFESSOR"))
                    {
                        isApproved = true;
                    }
                    if(facultyType.toUpperCase().contains("INSTRUCTOR"))
                    {
                        isApproved = true;
                    }
                }
            }
        }
        catch(Exception e)
        {
            throw new IdentityServiceException("failed to authorize faculty status", e);
        }

        return isApproved;
    }

    public String getEmailAddress(String ecommonsID)
    {
        return this.port.getEmail(issuer, ecommonsID);
    }

    public String getName(String ecommonsID)
    {
        return this.port.getName(issuer, ecommonsID);
    }
}
