package net.shrine.sheriff.controller;

import net.shrine.sheriff.model.ApprovalStatus;
import net.shrine.sheriff.model.Constants;
import net.shrine.sheriff.model.Sheriff;
import net.shrine.sheriff.model.SheriffDAO;
import net.shrine.sheriff.model.SheriffEntry;
import net.shrine.sheriff.model.SheriffException;
import org.apache.log4j.Logger;
import org.spin.tools.config.ConfigException;
import org.spin.tools.crypto.signature.Identity;
import org.spin.tools.crypto.signature.XMLSignatureUtil;

import javax.jws.WebService;
import javax.xml.crypto.dsig.XMLSignatureException;
import java.util.List;

@WebService(serviceName = Constants.serviceName,
        portName = Constants.portName,
        targetNamespace = Constants.namespace,
        endpointInterface = "net.shrine.sheriff.model.Sheriff"
)
public class SheriffImpl implements Sheriff
{
    protected final Logger log = Logger.getLogger(SheriffImpl.class);

    protected final boolean DEBUG = log.isDebugEnabled();
    protected final boolean INFO = log.isInfoEnabled();

    protected final SheriffDAO dao = new SheriffDAO();

    public SheriffEntry readApprovedTopic(Identity identity, Integer queryTopicID)
            throws SheriffException
    {

        String username = getUsernameFromIdentity(identity);

        if(INFO)
        {
            log.info("Attempting to validate approved sheriff topic \"" + queryTopicID + "\" for " + username);
        }

        SheriffEntry entry = dao.readById(new Long(queryTopicID.intValue()));

        if((entry != null) &&
                (entry.getECommonsID().equalsIgnoreCase(username)) &&
                (entry.getApproval() == ApprovalStatus.Approved))
        {
            if(INFO)
            {
                log.info("QUERY: validated approved sheriff topic \"" + entry.getQueryName() + "\" for " + username);
            }
            return entry;
        }
        else
        {
            throw new SheriffException("Validation FAILED for sheriff topic \""
                    + queryTopicID + "\" for " + username + ": "
                    + String.valueOf(entry));
        }

    }

    public List<SheriffEntry> readApprovedEntries(Identity identity) throws SheriffException
    {
        try
        {
            String username = getUsernameFromIdentity(identity);

            if(INFO)
            {
                log.info("Attempting to read approved sheriff entries for " + username);
            }

            List<SheriffEntry> entries = dao.readApprovedEntries(identity.getUsername());

            if(INFO)
            {
                log.info("LOGIN: Returning " + entries.size() + " entries for " + username);
            }

            return entries;
        }
        catch(SheriffException se)
        {
            //just rethrow, do not re-wrap.
            throw se;
        }
        catch(Exception e)
        {
            throw new SheriffException("Unhandled exception during readApprovedEntries", e);
        }
    }

    private String getUsernameFromIdentity(Identity identity) throws SheriffException
    {

        if(identity == null)
        {
            throw new SheriffException("Identity not provided to read approved query topics.");
        }

        String username = identity.getUsername();

        if(username == null)
        {
            throw new SheriffException("Username not provided to read approved query topics.");
        }

        try
        {
            if(!XMLSignatureUtil.getDefaultInstance().verifySignature(identity))
            {
                throw new SheriffException("XML Digital Signature was invalid or untrusted");
            }
        }
        catch(XMLSignatureException e)
        {
            throw new SheriffException("Error reading signature ", e);
        }
        catch(ConfigException e){
            throw new SheriffException("Error getting instance of XMLSignatureUtil");
        }

        return username;
    }
}
