package net.shrine.sheriff.controller;

import org.apache.log4j.Logger;
import org.spin.extension.MailTool;
import org.spin.tools.config.ConfigException;

import javax.mail.MessagingException;

public class SheriffMailUtil
{
    private static SheriffMailUtil instance = null;
    protected static final Logger log = Logger.getLogger(SheriffMailUtil.class);

    private static final boolean INFO = log.isInfoEnabled();
    private static final boolean DEBUG = log.isDebugEnabled();

    private MailTool mailTool = null;
    private boolean sendEmailNotifications;
    private String dataStewardEmail;
    private AuthFacade authFacade;

    private SheriffMailUtil(boolean sendEmailNotifications,
            String dataStewardEmail, AuthFacade authFacade)
    {
        if(INFO)
        {
            log.info("Initializing SheriffMailUtil...");
        }

        this.sendEmailNotifications = sendEmailNotifications;

        this.dataStewardEmail = dataStewardEmail;

        if(this.isSendEmailNotifications())
        {
            try
            {
                mailTool = MailTool.getInstance();
            }
            catch(ConfigException cfe)
            {
                mailTool = null;
                this.setSendEmailNotifications(false);
                log.error("Failed to initialize mailTool, setting SEND_EMAIL_NOTIFICATIONS to FALSE.", cfe);
            }
        }
    }

    public static SheriffMailUtil createInstance(boolean sendEmail, String dataStewardEmail)
    {
        AuthFacade authFacade = AuthFacade.getInstance();
        if(authFacade == null)
        {
            throw new Error("AuthFacade is not initialized.");
        }
        instance = new SheriffMailUtil(sendEmail, dataStewardEmail, authFacade);
        return instance;
    }

    public static SheriffMailUtil getInstance()
    {
        return instance;
    }

    protected String getDataStewardEmail()
    {
        return dataStewardEmail;
    }

    public boolean isSendEmailNotifications()
    {
        return sendEmailNotifications;
    }

    protected void setSendEmailNotifications(boolean b)
    {
        sendEmailNotifications = false;
    }

    public void sendSysAdminNotification(String subject, String message)
    {
        try
        {
            if(isSendEmailNotifications())
            {
                if(INFO)
                {
                    log.info("Sending sendSysAdminNotification: \"" + subject + "\", SendEmailNotifications=" + isSendEmailNotifications());
                }
                mailTool.sendAdminNotification(subject, message);
            }
            else
            {
                if(INFO)
                {
                    log.info("NOT sending sendSysAdminNotification: \"" + subject + "\", SendEmailNotifications=" + isSendEmailNotifications());
                }
            }
        }
        catch(MessagingException e)
        {
            log.error("Failed to send notification to system administrator.", e);
            e.printStackTrace();
        }
    }

    public void sendDataStewardNotification(String subject, String message)
    {
        try
        {
            if(isSendEmailNotifications())
            {
                if(INFO)
                {
                    log.info("Sending Data Steward notification: \"" + subject + "\", SendEmailNotifications=" + isSendEmailNotifications());
                }
                mailTool.sendMail(getSender(), this.getDataStewardEmail(), subject, message);
            }
            else
            {
                if(INFO)
                {
                    log.info("NOT sending Data Steward notification: \"" + subject + "\", SendEmailNotifications=" + isSendEmailNotifications());
                }
            }
        }
        catch(MessagingException e)
        {
            log.error("Failed to send notification to Data Steward.", e);
            e.printStackTrace();
        }
    }

    public void sendUserNotification(String ecommonsID, String subject, String message)
    {
        try
        {
            if(isSendEmailNotifications())
            {
                String email = this.authFacade.getEmailAddress(ecommonsID);
                if(INFO)
                {
                    log.info("Sending user notification: \"" + subject + "\" to user:" + ecommonsID + " (" + email + "), SendEmailNotifications=" + isSendEmailNotifications());
                }
                mailTool.sendMail(getSender(), email, subject, message);
            }
            else
            {
                if(INFO)
                {
                    log.info("NOT sending user notification: \"" + subject + "\" to user:" + ecommonsID + ", SendEmailNotifications=" + isSendEmailNotifications());
                }
            }
        }
        catch(MessagingException e)
        {
            log.error("Failed to send notification to user: " + String.valueOf(ecommonsID), e);
            e.printStackTrace();
        }
    }

    private String getSender()
    {
        return mailTool.getMailToolConfig().getSender();
    }
}
