package net.shrine.sheriff.model;

import net.shrine.sheriff.controller.SheriffAuthException;
import net.shrine.sheriff.controller.SheriffDAOException;
import org.apache.log4j.Logger;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * @author Andrew McMurry
 */
public class SheriffDAO
{
    //Type-safe query parameterization

    public static enum QueryParam
    {
        QUERYTOPICID("queryTopicID"),
        ROLE("role"),
        ECOMMONSID("eCommonsID"),
        QUERYNAME("queryName"),
        QUERYINTENT("queryIntent"),
        APPROVAL("approval"),
        REQUESTEDTIMESTAMP("requestedTimestamp"),
        APPROVALTIMESTAMP("approvalTimestamp"),
        NOTE("note"),
        PROTOCOLNUMBER("protocolNumber");

        private final String property;

        QueryParam(String propertyName)
        {
            this.property = propertyName;
        }

        public Criterion eq(Object value)
        {
            return Restrictions.eq(property, value);
        }

        public Criterion ne(Object value)
        {
            return Restrictions.ne(property, value);
        }

        public Criterion like(Object value)
        {
            return Restrictions.like(property, value);
        }

        public Order ascending()
        {
            return Order.asc(property);
        }

        public Order descending()
        {
            return Order.desc(property);
        }
    }

    private static final Logger log = Logger.getLogger(SheriffDAO.class);

    private final boolean DEBUG;

    private final boolean INFO;

    private static SessionFactory sessionFactory = null;

    public SheriffDAO()
    {
        DEBUG = log.isDebugEnabled();
        INFO = log.isInfoEnabled();

        if(sessionFactory == null)
        {
            sessionFactory = SheriffHibernateUtil.Instance.getSessionFactory();
        }
    }

    public SheriffEntry createNewPendingEntry(SheriffEntry source) throws SheriffDAOException, SheriffAuthException
    {
        String ecommonsID = source.getECommonsID();
        String email = source.getEmail();
        String queryName = source.getQueryName();
        String queryIntent = source.getQueryIntent();
        String queryReason = source.getQueryReason();
        String queryReasonOther = source.getQueryReasonOther();
        boolean hasNonShrineCollaborator = source.getHasNonShrineCollaborator();
        String nonShrineCollaboratorInfo = source.getNonShrineCollaboratorInfo();

        SheriffEntry clone = createNewPendingEntry(ecommonsID, email, queryName,
                queryIntent, queryReason, queryReasonOther,
                hasNonShrineCollaborator, nonShrineCollaboratorInfo);

        clone.setParentId(source.getQueryTopicID());
        return clone;
    }

    public SheriffEntry createNewPendingEntry(String ecommonsID, String email,
            String queryName, String queryIntent, String queryReason,
            String queryReasonOther, boolean hasNonShrineCollaborator,
            String nonShrineCollaboratorInfo) throws SheriffDAOException,
            SheriffAuthException
    {

        SheriffEntry sheriffEntry = null;
        try
        {
            sheriffEntry = new SheriffEntry();
            //
            sheriffEntry.setECommonsID(ecommonsID);
            sheriffEntry.setEmail(email);
            sheriffEntry.setApproval(ApprovalStatus.Pending);
            sheriffEntry.setRole(RoleLevel.Aggregate);

            sheriffEntry.setQueryName(queryName);
            sheriffEntry.setQueryIntent(queryIntent);

            sheriffEntry.setQueryReason(queryReason);
            sheriffEntry.setQueryReasonOther(queryReasonOther);
            sheriffEntry.setHasNonShrineCollaborator(hasNonShrineCollaborator);
            sheriffEntry.setNonShrineCollaboratorInfo(nonShrineCollaboratorInfo);
            sheriffEntry.setRequestedTimestamp(new Date());
            //
            create(sheriffEntry);
        }
        catch(SheriffException e)
        {
            throw new SheriffDAOException("Failed to write to the sheriff database.", e);
        }
        return sheriffEntry;
    }

    public void create(final SheriffEntry entry) throws SheriffException
    {
        if(DEBUG)
        {
            log.debug("Creating sheriff entry.");
        }
        final Session session = sessionFactory.openSession();
        try
        {

            Calendar cal = Calendar.getInstance();
            cal.setTime(entry.getRequestedTimestamp());
            int year = cal.get(Calendar.YEAR);
            int numberInYear = ProtocolNumberCounter.getNext(session, year);

            session.beginTransaction();
            entry.setProtocolNumber(year, numberInYear);
            if(INFO)
            {
                log.info("Saving entry " + String.valueOf(entry));
            }
            session.save(entry);
            session.getTransaction().commit();

            if(DEBUG)
            {
                log.debug("Created entry " + entry.getProtocolNumber() + " successfully");
            }
        }
        catch(final RuntimeException e)
        {
            session.getTransaction().rollback();
            throw new SheriffDAOException("Could not create entry", e);
        }
        catch(final Error e)
        {
            log.error("Could not create entry: Fatal Error: ", e);
            throw new SheriffDAOException("Holy Fatal error batman!", e);
        }
        finally
        {
            closeSession(session);
        }
    }

    public List<SheriffEntry> readAll() throws SheriffException
    {
        ArrayList<Criterion> clauses = new ArrayList<Criterion>();

        return read(clauses);

    }

    public List<SheriffEntry> readPendingEntries() throws SheriffDAOException
    {
        if(DEBUG)
        {
            log.debug("readPendingEntries() sheriff entries for all users");
        }

        ArrayList<Criterion> clauses = new ArrayList<Criterion>();

        clauses.add(QueryParam.APPROVAL.eq(ApprovalStatus.Pending));

        return read(clauses);
    }

    public List<SheriffEntry> readByRequestorID(final String eCommonsID) throws SheriffException
    {
        ArrayList<Criterion> clauses = new ArrayList<Criterion>();

        clauses.add(QueryParam.ECOMMONSID.eq(eCommonsID));

        return read(clauses);
    }

    public List<SheriffEntry> readApprovedEntries(final String eCommonsID) throws SheriffDAOException
    {
        if(DEBUG)
        {
            log.debug("readApprovedEntries() sheriff entries for user " + eCommonsID);
        }

        ArrayList<Criterion> clauses = new ArrayList<Criterion>();

        clauses.add(QueryParam.ECOMMONSID.eq(eCommonsID));

        clauses.add(QueryParam.APPROVAL.eq(ApprovalStatus.Approved));

        return read(clauses);
    }

    public List<SheriffEntry> readPendingEntries(final String eCommonsID) throws SheriffDAOException
    {
        if(DEBUG)
        {
            log.debug("readPendingEntries() sheriff entries for user " + eCommonsID);
        }

        ArrayList<Criterion> clauses = new ArrayList<Criterion>();

        clauses.add(QueryParam.ECOMMONSID.eq(eCommonsID));

        clauses.add(QueryParam.APPROVAL.eq(ApprovalStatus.Pending));

        return read(clauses);
    }

    public SheriffEntry readById(final Long queryTopicID) throws SheriffDAOException
    {
        if(DEBUG)
        {
            log.debug("readById() for query topic id " + queryTopicID);
        }

        ArrayList<Criterion> clauses = new ArrayList<Criterion>();

        clauses.add(QueryParam.QUERYTOPICID.eq(queryTopicID));

        return read(clauses).get(0);
    }

    public List<SheriffEntry> read(List<Criterion> clauses) throws SheriffDAOException
    {
        return read(clauses, QueryParam.REQUESTEDTIMESTAMP.descending());
    }

    public List<SheriffEntry> read(List<Criterion> clauses, Order... sortCriteria) throws SheriffDAOException
    {
        Session session = null;
        try
        {

            if(DEBUG)
            {
                StringBuilder queryDescription = new StringBuilder("Read: Criteria[");
                for(Criterion clause : clauses)
                {
                    queryDescription.append("{").append(clause.toString()).append("}");
                }

                queryDescription.append("], Order[");
                for(Order order : sortCriteria)
                {
                    queryDescription.append("{").append(order.toString()).append("}");
                }
                queryDescription.append("]");
                log.debug(queryDescription.toString());
            }

            session = sessionFactory.openSession();

            final Criteria criteria = session.createCriteria(SheriffEntry.class);

            for(Criterion clause : clauses)
            {
                criteria.add(clause);
            }

            criteria.add(QueryParam.APPROVAL.ne(ApprovalStatus.Deleted));

            for(Order o : sortCriteria)
            {
                criteria.addOrder(o);
            }

            @SuppressWarnings("unchecked")
            final List<SheriffEntry> results = criteria.list();
            if(results != null)
            {
                return results;
            }
            else
            {
                return new ArrayList<SheriffEntry>();
            }
        }
        catch(Exception e)
        {
            log.error("Failed to read entries.", e);
            throw new SheriffDAOException("Failed to read entries.", e);
        }
        finally
        {
            closeSession(session);
        }
    }

    public void update(Long queryTopicID, ApprovalStatus approval) throws SheriffException
    {
        if(DEBUG)
        {
            log.debug("update() queryTopicID " + queryTopicID + " status " + approval.name());
        }

        try
        {
            SheriffEntry entry = readById(queryTopicID);

            entry.setApproval(approval);
            entry.setApprovalTimestamp(new Date());

            update(entry);
        }
        catch(Exception e)
        {
            log.error("Failed to read entries.", e);
            throw new SheriffException("Failed to set approval status to " + approval.toString() + " for query topic " + queryTopicID);
        }
    }


    public void delete(SheriffEntry entry) throws SheriffException
    {
        if(DEBUG)
        {
            log.debug("delete() queryTopicID " + entry.getProtocolNumber() + " status " + entry.getApproval().name());
        }
        entry.setApproval(ApprovalStatus.Deleted);
        update(entry);
    }

    public void retract(SheriffEntry entry) throws SheriffException
    {
        if(DEBUG)
        {
            log.debug("retract() queryTopicID " + entry.getProtocolNumber() + " status " + entry.getApproval().name());
        }
        if(entry.getApproval() != ApprovalStatus.Pending)
        {
            SheriffException e = new SheriffException("Failed to retract request. Current approval state is: \"" + entry.getApproval().name() + "\"");
            log.error(e);
            throw e;
        }
        entry.setApproval(ApprovalStatus.Retracted);
        update(entry);
    }


    public void resubmit(SheriffEntry entry) throws SheriffException
    {
        if(DEBUG)
        {
            log.debug("resubmit() queryTopicID " + entry.getProtocolNumber() + " status " + entry.getApproval().name());
        }
        if((entry.getApproval() != ApprovalStatus.Denied) && (entry.getApproval() != ApprovalStatus.Retracted))
        {
            SheriffException e = new SheriffException("Failed to re-submit request. Current approval state is: \"" + entry.getApproval().name() + "\"");
            log.error(e);
            throw e;
        }
        SheriffEntry clone = createNewPendingEntry(entry);
        update(clone);
    }

    public void update(final SheriffEntry entry) throws SheriffException
    {
        if(DEBUG)
        {
            log.debug("update() called " + entry);
        }

        Session session = null;

        try
        {
            session = sessionFactory.openSession();
            session.beginTransaction();
            session.update(entry);
            session.getTransaction().commit();
        }
        catch(final RuntimeException e)
        {
            session.getTransaction().rollback();

            throw new SheriffException("Could not update entry", e);
        }
        finally
        {
            closeSession(session);
        }
    }

    public SheriffUser readUser(String ecommonsID) throws SheriffDAOException
    {
        Session session = null;
        try
        {
            session = sessionFactory.openSession();

            final Criteria criteria = session.createCriteria(SheriffUser.class);

            criteria.add(Restrictions.eq("eCommonsID", ecommonsID));

            @SuppressWarnings("unchecked")
            final List<SheriffUser> results = criteria.list();

            return results.size() == 1 ? results.get(0) : null;
        }
        catch(Exception e)
        {
            throw new SheriffDAOException(e);
        }
        finally
        {
            closeSession(session);
        }
    }

    /**
     * Close a Hibernate session, logging exceptions.
     *
     * @param session
     */
    private void closeSession(Session session)
    {
        try
        {
            session.close();
        }
        catch(Exception e)
        {
            log.error("Error while closing session", e);
        }
    }

    private void destroy(SheriffEntry entry) throws SheriffException
    {
        if(DEBUG)
        {
            log.debug("delete() called " + entry);
        }

        final Session session = sessionFactory.openSession();

        try
        {
            session.beginTransaction();
            session.delete(entry);
            session.getTransaction().commit();
        }
        catch(final RuntimeException e)
        {
            session.getTransaction().rollback();

            throw new SheriffDAOException("Could not remove", e);
        }
        finally
        {
            closeSession(session);
        }
    }

    public void truncateSheriffEntries() throws Exception
    {
        log.warn("Truncating SheriffEntry table");

        for(SheriffEntry entry : readAll())
        {
            destroy(entry);
        }
    }


    public static SessionFactory getSessionFactory()
    {
        return sessionFactory;
    }
}
