package net.shrine.sheriff.model;

import org.apache.log4j.Logger;
import org.hibernate.HibernateException;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.AnnotationConfiguration;
import org.hibernate.cfg.Configuration;
import org.spin.tools.config.ConfigTool;

import java.net.URL;
import java.sql.DriverManager;
import java.sql.SQLException;

/**
 * @author Andrew McMurry
 */
public class SheriffHibernateUtil
{
    private static final Logger log = Logger.getLogger(SheriffHibernateUtil.class);

    private static final boolean INFO = log.isInfoEnabled();
    private static final boolean DEBUG = log.isDebugEnabled();

    public static final String hibernateConfigFileName = "hibernate-sheriff.cfg.xml";

    public static final SheriffHibernateUtil Instance = new SheriffHibernateUtil();

    private final SessionFactory sessionFactory;
    private final String dbDriver;
    private final String dbUrl;
    private final String dataSourceName;

    private SheriffHibernateUtil()
    {
        //
        //Shut down the Derby DB when the current JVM instance shuts down.
        addShutdownHook();

        try
        {
            final URL configFile = ConfigTool.getConfigFileAsURL(hibernateConfigFileName);

            if(DEBUG)
            {
                log.debug("Initializing Hibernate with '" + configFile.getFile() + "'");
            }

            final Configuration config = new AnnotationConfiguration()
                    .addAnnotatedClass(ProtocolNumberCounter.class)
                    .addAnnotatedClass(SheriffEntry.class)
                    .configure(configFile); //must be on classpath

            sessionFactory = config.buildSessionFactory();
            dbDriver = config.getProperty("connection.driver_class");
            dbUrl = config.getProperty("connection.url");
            dataSourceName = config.getProperty("connection.datasource");
        }
        catch(Throwable e)
        {
            // Make sure you log the exception, as it might be swallowed
            log.error("Initial SessionFactory creation failed: " + e.getMessage());

            throw new ExceptionInInitializerError(e);
        }
    }

    /**
     * Add a JVM hook to shut down the Derby DB when the current JVM instance shuts down.
     */
    private void addShutdownHook()
    {
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable()
        {
            public void run()
            {
                try
                {
                    shutDown();
                }
                catch(Throwable e)
                {
                    log.error("Failed shutting down Spin query logging - the Node may take longer to start up next time, or the admin database may be corrupted: ", e);
                }
            }
        }));
    }

    public final SessionFactory getSessionFactory()
    {
        return sessionFactory;
    }

    public final String getDataSourceName()
    {
        return dataSourceName;
    }

    public final String getDBUrl()
    {
        return dbUrl;
    }

    public final String getDBDriver()
    {
        return dbDriver;
    }

    public final boolean shutDown()
    {
        shutdownHibernate();

        return shutdownDerby();
    }

    /**
     * NB: This is counterintuitive.  Shutting down Derby SUCCESSFULLY
     * throws an exception.  (The idea is that DriverManager.getConnection()
     * should return a valid connection, or throw, and that retuning a valid
     * Connection is pointless when shutting down the DB.  But still, it's
     * weird.)
     * <p/>
     * "In embedded mode, an application should shut down Derby.
     * If the application fails to shut down Derby explicitly,
     * the Derby does not perform a checkpoint when the JVM shuts down, which means
     * that the next connection will be slower.
     * Explicitly shutting down Derby with the URL is preferred.
     * This style of shutdown will always throw an 'exception'."
     * See:
     * http://www.nabble.com/Shutdown-of-embedded-Derby-server-throws-an-SQLException--td15421009.html
     *
     * @return true if shutdown succeeded, false otherwise
     */
    private boolean shutdownDerby()
    {
        try
        {
            Class.forName(getDBDriver());

            DriverManager.getConnection("jdbc:derby:;shutdown=true");

            log.error("Failed shutting down Spin admin database");

            return false;
        }
        catch(SQLException e)
        {
            if(INFO)
            {
                log.info("Successfully shut down Spin admin database");
            }

            return true;
        }
        catch(Throwable e)
        {
            log.error("Failed shutting down Spin admin database", e);

            return false;
        }
    }

    /**
     * NB: Destroy this SessionFactory and release all resources
     * (caches, connection pools, etc).
     * <p/>
     * **It is the responsibility of the application to ensure
     * that there are no open Sessions before calling close()**.
     */
    private void shutdownHibernate()
    {
        try
        {
            sessionFactory.close();

            if(INFO)
            {
                log.info("Shut down Spin Admin Hibernate SessionFactory");
            }
        }
        catch(HibernateException e)
        {
            log.error("Error shutting down Spin admin Hibernate SessionFactory");

            throw e;
        }
    }
}
