package net.shrine.sheriff.view;

import net.shrine.sheriff.model.ApprovalStatus;
import net.shrine.sheriff.model.SheriffDAO;
import net.shrine.sheriff.model.SheriffDAO.QueryParam;
import net.shrine.sheriff.model.SheriffEntry;
import net.shrine.sheriff.model.SheriffException;
import org.apache.log4j.Logger;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * [ SUMMARY ]
 * <p/>
 * [ Author ]
 * Andrew McMurry
 * Date: Jul 13, 2009
 * <p/>
 * Harvard Medical School Center for BioMedical Informatics
 *
 * @link http://cbmi.med.harvard.edu
 * <p/>
 * [ In partnership with ]
 * @link http://chip.org
 * @link http://lcs.mgh.harvard.edu
 * @link http://www.brighamandwomens.org
 * @link http://bidmc.harvard.edu
 * @link http://dfhcc.harvard.edu
 * @link http://spin.nci.nih.gov/
 * <p/>
 * <p/>
 * -------------------------------------------------
 * [ Licensing ]
 * All works licensed by the Lesser GPL
 * @link http://www.gnu.org/licenses/lgpl.html
 * -------------------------------------------------
 */
public class AdminInterfaceServlet extends HttpServlet
{
    private static final long serialVersionUID = 1L;

    private static final Logger log = Logger.getLogger(AdminInterfaceServlet.class);

    private static final boolean DEBUG = log.isDebugEnabled();

    private static final boolean INFO = log.isInfoEnabled();

    private SheriffDAO sheriff;

    @Override
    public void init() throws ServletException
    {
        super.init();
        if(INFO)
        {
            log.info("Starting up Admin Interface Servlet.");
        }
        if(INFO)
        {
            log.info("Getting handle on DAO.");
        }
        sheriff = new SheriffDAO();
    }

    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        doPost(request, response);
    }

    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        String nextURL = null;

        HttpSession session = request.getSession();

        String ecommonsID = request.getParameter(MVC.Form.eCommonsID.name());
        String email = request.getParameter(MVC.Form.email.name());
        String approvalStatus = request.getParameter(MVC.Form.approval.name());
        String queryName = request.getParameter(MVC.Form.queryName.name());
        String queryIntent = request.getParameter(MVC.Form.queryIntent.name());
        String note = request.getParameter(MVC.Form.note.name());
        String sortBY = request.getParameter(MVC.Form.sortBY.name());
        String sortASC = request.getParameter(MVC.Form.sortASC.name());
        String followUp = request.getParameter(MVC.Form.followUp.name());
        String queryReason = request.getParameter(MVC.Form.queryReason.name());
        String queryReasonOther = request.getParameter(MVC.Form.queryReasonOther.name());
        boolean hasNonShrineCollaborator = false;
        String nonShrineCollaboratorVal = request.getParameter(MVC.Form.hasNonShrineCollaborator.name());
        if(null != nonShrineCollaboratorVal)
        {
            hasNonShrineCollaborator = (nonShrineCollaboratorVal.equalsIgnoreCase("y"));
        }
        String nonShrineCollaboratorInfo = request.getParameter(MVC.Form.nonShrineCollaboratorInfo.name());

        Long queryTopicID = null;

        if(null != request.getParameter(MVC.Form.queryTopicID.name()))
        {
            queryTopicID = Long.parseLong(request.getParameter(MVC.Form.queryTopicID.name()));
        }

        if(DEBUG)
        {
            log.debug("ecommonsID " + ecommonsID);
            log.debug("email " + email);
            log.debug("queryTopicID " + queryTopicID);
            log.debug("approvalStatus " + approvalStatus);
            log.debug("queryName " + queryName);
            log.debug("queryIntent " + queryIntent);
            log.debug("note " + note);
            log.debug("sortBy " + sortBY);
            log.debug("sortASC " + sortASC);
            log.debug("followUp " + followUp);

            log.debug("queryReason " + queryReason);
            log.debug("queryReasonOther " + queryReasonOther);
            log.debug("hasNonShrineCollaborator " + hasNonShrineCollaborator);
            log.debug("nonShrineCollaboratorInfo " + nonShrineCollaboratorInfo);
        }

        try
        {
            switch(MVC.getAction(request))
            {
                case create:

                    SheriffEntry newEntry = sheriff.createNewPendingEntry(
                            ecommonsID, email, queryName, queryIntent, queryReason,
                            queryReasonOther, hasNonShrineCollaborator,
                            nonShrineCollaboratorInfo);
                    if(INFO)
                    {
                        log.info("Created: " + String.valueOf(newEntry));
                    }
                    reCacheEntries(session);
                    nextURL = MVC.nextAdminJsp(MVC.Action.home);
                    break;

                case read:

                    SheriffEntry entry = sheriff.readById(queryTopicID);
                    if(INFO)
                    {
                        log.info("Read: " + String.valueOf(entry));
                    }
                    session.setAttribute(MVC.Session.entry.name(), entry);
                    nextURL = MVC.nextDetailJsp(MVC.Action.home);
                    break;

                case update:

                    SheriffEntry cached = (SheriffEntry) session.getAttribute(MVC.Session.entry.name());
                    if(cached == null)
                    {
                        log.error("Cache entry not found! ");
                    }
                    else
                    {
                        if(DEBUG)
                        {
                            log.debug(String.valueOf(cached));
                        }
                    }

                    if(INFO)
                    {
                        log.info("Admin changed content of the request, updating now.");
                    }

                    SheriffEntry savedCopy = sheriff.readById(queryTopicID);
                    //
                    savedCopy.setQueryIntent(queryIntent);
                    savedCopy.setNote(note);
                    savedCopy.setFollowUp(followUp);
                    savedCopy.setQueryName(queryName);
                    //

                    sheriff.update(savedCopy);
                    if(INFO)
                    {
                        log.info("Admin Updated: " + String.valueOf(savedCopy));
                    }

                    if(!approvalStatus.equalsIgnoreCase(cached.getApproval().name()))
                    {
                        if(INFO)
                        {
                            log.info("Admin changed status of the request, updating now.");
                        }

                        sheriff.update(queryTopicID, ApprovalStatus.valueOf(approvalStatus));
                    }

                    reCacheEntries(session);

                    nextURL = MVC.nextAdminJsp(MVC.Action.home);

                    break;

                case delete:

                    SheriffEntry toDelete = sheriff.readById(queryTopicID);
                    if(INFO)
                    {
                        log.info("Admin Deleting: " + String.valueOf(toDelete));
                    }
                    sheriff.delete(toDelete);
                    reCacheEntries(session);

                case home:
                default:
                    if(INFO)
                    {
                        log.info("User " + ecommonsID + " viewing all open topics");
                    }

                    boolean ready = false;

                    if(sortBY != null && sortASC != null)
                    {
                        try
                        {
                            reCacheEntries(session, sortBY, Boolean.parseBoolean(sortASC));
                            ready = true;
                        }
                        catch(Exception e)
                        {
                            log.error("Could not parse sortBY and sortASC params, reCacheEntries(session) using defaults.", e);
                        }
                    }

                    if(ready == false)
                    {
                        reCacheEntries(session);
                    }

                    nextURL = MVC.nextAdminJsp(MVC.Action.home);
                    break;
            }
        }
        catch(SheriffException e)
        {
            throw new ServletException("Request failed", e);
        }

        request.getRequestDispatcher(nextURL).forward(request, response);
    }

    private void reCacheEntries(HttpSession session) throws SheriffException
    {
        List<Criterion> clauses = new ArrayList<Criterion>();

        clauses.add(QueryParam.APPROVAL.ne(ApprovalStatus.Retracted));

        session.setAttribute(MVC.Session.entries.name(),
                sheriff.read(clauses,
                        SheriffDAO.QueryParam.APPROVAL.ascending(),
                        SheriffDAO.QueryParam.REQUESTEDTIMESTAMP.descending()));
    }

    private void reCacheEntries(HttpSession session, String sortBY, boolean sortAsc) throws SheriffException
    {
        List<Criterion> clauses = new ArrayList<Criterion>();
        clauses.add(QueryParam.APPROVAL.ne(ApprovalStatus.Retracted));

        QueryParam sort = null;
        try
        {
            sort = QueryParam.valueOf(sortBY.toUpperCase());
        }
        catch(Exception e)
        {
            log.error("Unknown sort criteria: " + String.valueOf(sort), e);
        }

        if(sort != null)
        {
            Order sortCriteria = (sortAsc == true) ? sort.ascending() : sort.descending();
            session.setAttribute(MVC.Session.entries.name(), sheriff.read(clauses, sortCriteria));
        }
        else
        {
            reCacheEntries(session);
        }

    }
}