package net.shrine.sheriff.view;

import net.shrine.sheriff.controller.AuthFacade;
import net.shrine.sheriff.controller.SheriffMailUtil;
import org.apache.log4j.Logger;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.text.DateFormat;
import java.util.Date;


/**
 * [ SUMMARY ]
 * <p/>
 * [ Author ]
 * Andrew McMurry
 * Date: Jul 14, 2009
 * <p/>
 * Harvard Medical School Center for BioMedical Informatics
 *
 * @link http://cbmi.med.harvard.edu
 * <p/>
 * [ In partnership with ]
 * @link http://chip.org
 * @link http://lcs.mgh.harvard.edu
 * @link http://www.brighamandwomens.org
 * @link http://bidmc.harvard.edu
 * @link http://dfhcc.harvard.edu
 * @link http://spin.nci.nih.gov/
 * <p/>
 * <p/>
 * -------------------------------------------------
 * [ Licensing ]
 * All works licensed by the Lesser GPL
 * @link http://www.gnu.org/licenses/lgpl.html
 * -------------------------------------------------
 */
public class LogonServlet extends HttpServlet
{
    private static final String SEND_EMAIL_NOTIFICATIONS_PARAM = "SEND_EMAIL_NOTIFICATIONS";
    private static final String DATA_STEWARD_USERNAME_PARAM = "DATA_STEWARD_USERNAME";
    private static final String DATA_STEWARD_PASSWORD_PARAM = "DATA_STEWARD_PASSWORD";
    private static final String DATA_STEWARD_EMAIL_PARAM = "DATA_STEWARD_EMAIL";
    private static final String CHECK_FACULTY_PARAM = "CHECK_FACULTY";

    private static final String AUTH_APPLICATION_PARAM = "AUTH_APPLICATION";
    private static final String AUTH_ISSUER_PARAM = "AUTH_ISSUER";
    private static final String AUTH_PROFILES_URL_PARAM = "AUTH_PROFILES_URL";

    private String dataStewardUserName;
    private String dataStewardPassword;
    private boolean checkFaculty;

    private static final long serialVersionUID = 1L;
    private static final Logger log = Logger.getLogger(LogonServlet.class);

    @Override
    public void init() throws ServletException
    {
        super.init();
        dataStewardUserName = getInitParameter(DATA_STEWARD_USERNAME_PARAM);
        dataStewardPassword = getInitParameter(DATA_STEWARD_PASSWORD_PARAM);

        try
        {
            checkFaculty = Boolean
                    .parseBoolean(getInitParameter(CHECK_FACULTY_PARAM));
            log.info(CHECK_FACULTY_PARAM + " = " + checkFaculty);
        }
        catch(Exception e)
        {
            checkFaculty = true;
            log.error("Failed to parse " + CHECK_FACULTY_PARAM + ", defaulting to TRUE.", e);
        }
        initializeAuthentication();
        initializeMailConfig();

        if(SheriffMailUtil.getInstance().isSendEmailNotifications())
        {
            SheriffMailUtil.getInstance().sendSysAdminNotification(
                    "Sheriff application was restarted",
                    DateFormat.getDateTimeInstance(DateFormat.LONG,
                            DateFormat.LONG).format(new Date()));
        }
    }
    private void initializeAuthentication()
    {
        String application = getInitParameter(AUTH_APPLICATION_PARAM);
        String issuer = getInitParameter(AUTH_ISSUER_PARAM);
        String profilesURL = getInitParameter(AUTH_PROFILES_URL_PARAM);
        log.info("Initializing Authentication...");
        AuthFacade.createInstance(application, issuer, profilesURL);
    }

    private void initializeMailConfig()
    {
        log.info("initializing Mail Config...");

        String dataStewardEmail = getInitParameter(DATA_STEWARD_EMAIL_PARAM);

        boolean sendEmailNotifications;
        try
        {
            sendEmailNotifications = Boolean.parseBoolean(getInitParameter(SEND_EMAIL_NOTIFICATIONS_PARAM));
            log.info(SEND_EMAIL_NOTIFICATIONS_PARAM + " = " + sendEmailNotifications);
        }
        catch(Exception e)
        {
            sendEmailNotifications = false;
            log.error("Failed to parse " + SEND_EMAIL_NOTIFICATIONS_PARAM + ", defaulting to FALSE.", e);
        }

        SheriffMailUtil.createInstance(sendEmailNotifications, dataStewardEmail);
    }

    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        doPost(request, response);
    }

    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        String nextURL = null;

        switch(MVC.getAction(request))
        {
            case login:

                String ecommonsID = request.getParameter(MVC.Form.eCommonsID.name());
                String password = request.getParameter(MVC.Form.password.name());

                //ADMIN CASE
                if(ecommonsID != null && ecommonsID.equalsIgnoreCase(dataStewardUserName))
                {
                    if(password != null && password.equals(dataStewardPassword))
                    {
                        final HttpSession session = request.getSession(true);

                        session.setAttribute(MVC.Session.isAdmin.name(), Boolean.TRUE);

                        nextURL = "admin";
                    }
                    else
                    {
                        nextURL = errorURL("The supplied eCommons username and password were not valid.");
                    }
                }
                else //JOE SMOE USER CASE
                {
                    log.info("Trying to logon user " + ecommonsID);

                    boolean authenticated = false;
                    boolean authorized = false;

                    try
                    {
                        authenticated = AuthFacade.getInstance().authenticateEcommons(ecommonsID, password);

                        if(checkFaculty)
                        {
                            authorized = AuthFacade.getInstance().authorizeFacultyStatus(ecommonsID);
                        }
                        else
                        {
                            authorized = true;
                        }
                    }
                    catch(Exception e)
                    {
                        log.error("Sheriff offline? ", e);

                        nextURL = errorURL("Sorry, this application is currently offline.");
                    }

                    if(!authorized)
                    {
                        nextURL = errorURL("Sorry, you are not approved Harvard faculty.");
                    }
                    else if(!authenticated)
                    {
                        nextURL = errorURL("The supplied eCommons username and password were not valid.");
                    }
                    else
                    {
                        final HttpSession session = request.getSession(true);

                        session.setAttribute(MVC.Session.eCommonsID.name(), ecommonsID);

                        nextURL = "user";
                    }
                }

                break;

            case logout:
                log.info("Logging out user.");

                request.getSession().invalidate();

            case home:

            default:
                nextURL = "jsp/logon.jsp";
                break;
        }

        log.debug("redirecting user to " + nextURL);

        request.getRequestDispatcher(nextURL).forward(request, response);
    }

    private static String errorURL(String error)
    {
        log.error(error);

        return "jsp/logon.jsp?error=" + error;
    }
}


