package net.shrine.sheriff.view;

import net.shrine.sheriff.controller.AuthFacade;
import net.shrine.sheriff.controller.SheriffMailUtil;
import net.shrine.sheriff.model.SheriffDAO;
import net.shrine.sheriff.model.SheriffEntry;
import org.apache.log4j.Logger;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;

/**
 * [ SUMMARY ]
 * <p/>
 * [ Author ]
 * Andrew McMurry
 * Date: Jul 13, 2009
 * <p/>
 * Harvard Medical School Center for BioMedical Informatics
 *
 * @link http://cbmi.med.harvard.edu
 * <p/>
 * [ In partnership with ]
 * @link http://chip.org
 * @link http://lcs.mgh.harvard.edu
 * @link http://www.brighamandwomens.org
 * @link http://bidmc.harvard.edu
 * @link http://dfhcc.harvard.edu
 * @link http://spin.nci.nih.gov/
 * <p/>
 * <p/>
 * -------------------------------------------------
 * [ Licensing ]
 * All works licensed by the Lesser GPL
 * @link http://www.gnu.org/licenses/lgpl.html
 * -------------------------------------------------
 */
public class UserInterfaceServlet extends HttpServlet
{
    private static final long serialVersionUID = 1L;

    private static final Logger log = Logger.getLogger(UserInterfaceServlet.class);

    private static final boolean DEBUG = log.isDebugEnabled();
    private static final boolean INFO = log.isInfoEnabled();

    private SheriffDAO sheriff;

    @Override
    public void init() throws ServletException
    {
        if(INFO)
        {
            log.info("Starting up User Interface Servlet.");
        }
        if(INFO)
        {
            log.info("Getting handle on DAO.");
        }

        try
        {
            sheriff = new SheriffDAO();
        }
        catch(Exception e)
        {
            throw new ServletException("Failed to setup server", e);
        }
    }

    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        doPost(request, response);
    }

    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        //TODO: refactor
        String nextURL = MVC.nextUserJsp(MVC.Action.read);

        HttpSession session = request.getSession();
        String ecommonsID = (String) session.getAttribute(MVC.Session.eCommonsID.name());

        Long entryID = null;
        if(null != request.getParameter(MVC.Form.queryTopicID.name()))
        {
            entryID = Long.parseLong(request.getParameter(MVC.Form.queryTopicID.name()));
        }
        SheriffEntry entry = null;

        String followUp = request.getParameter(MVC.Form.followUp.name());
        String email = request.getParameter(MVC.Form.email.name());

        try
        {
            switch(MVC.getAction(request))
            {
                case create:

                    String queryName = request.getParameter(MVC.Form.queryName.name());
                    if(queryName == null || queryName.length() < 2)
                    {
                        log.error("Invalid queryName: \"" + String.valueOf(queryName) + "\"");
                        nextURL = MVC.nextUserJsp(MVC.Action.read);
                        break;
                    }
                    String queryIntent = request.getParameter(MVC.Form.queryIntent.name());
                    if(queryIntent == null || queryIntent.length() < 2)
                    {
                        log.error("Invalid queryIntent: \"" + String.valueOf(queryIntent) + "\"");
                        nextURL = MVC.nextUserJsp(MVC.Action.read);
                        break;
                    }
                    String queryReason = request.getParameter(MVC.Form.queryReason.name());
                    String queryReasonOther = request.getParameter(MVC.Form.queryReasonOther.name());

                    boolean hasNonShrineCollaborator = false;
                    String nonShrineCollaboratorVal = request.getParameter(MVC.Form.hasNonShrineCollaborator.name());
                    if(null != nonShrineCollaboratorVal)
                    {
                        hasNonShrineCollaborator = (nonShrineCollaboratorVal.equalsIgnoreCase("y"));
                    }
                    String nonShrineCollaboratorInfo = request.getParameter(MVC.Form.nonShrineCollaboratorInfo.name());
                    String name = AuthFacade.getInstance().getName(ecommonsID);
                    String ecommonsEmail = AuthFacade.getInstance().getEmailAddress(ecommonsID);

                    if(DEBUG)
                    {
                        log.debug("createNewPendingEntry: "
                                + " Name \"" + name + "\", "
                                + " User-Entered Email \"" + email + "\", "
                                + " eCommons Email \"" + ecommonsEmail + "\", "
                                + " eCommonsID \"" + ecommonsID + "\", "
                                + " Query Name \"" + queryName + "\", "
                                + " Query Intent \"" + queryIntent + "\", "
                                + " Query Reason \"" + queryReason + "\", "
                                + " Query Reason (Other) \"" + queryReasonOther + "\", "
                                + " Has Non-Shrine Collaborator \"" + nonShrineCollaboratorVal + "\", "
                                + " Non-Shrine Collaborator Info \"" + nonShrineCollaboratorInfo + "\"");
                    }

                    if(INFO)
                    {
                        log.info("Creating new query topic request, User=" + ecommonsID + ",  Name=" + queryName + ": " + queryIntent);
                    }

                    SheriffEntry newEntry = sheriff.createNewPendingEntry(
                            ecommonsID, email, queryName, queryIntent, queryReason,
                            queryReasonOther, hasNonShrineCollaborator,
                            nonShrineCollaboratorInfo);

                    if(INFO)
                    {
                        log.info("Created: " + String.valueOf(newEntry));
                    }

                    if(SheriffMailUtil.getInstance().isSendEmailNotifications())
                    {
                        String nonShrineCollaborator = (nonShrineCollaboratorVal.toLowerCase().equalsIgnoreCase("y")) ? "Yes" : "No";

                        String mailSubject = "SHRINE request for " + name + " (" + ecommonsID + ")";
                        String mailBody = "New SHRINE Request:\n\n"
                                + "Name:           " + name + "\n"
                                + "User-Entered Email: " + email + "\n"
                                + "eCommons Email: " + ecommonsEmail + "\n"
                                + "eCommons ID:    " + ecommonsID + "\n"
                                + "Profiles URL:   " + MVC.getProfileDetailsURL(ecommonsID) + "\n"
                                + "Name:           \"" + queryName + "\"\n"
                                + "Intent:         \"" + queryIntent + "\"\n"
                                + "Reason:         \"" + queryReason + "\"\n"
                                + "Reason (Other): \"" + queryReasonOther + "\"\n"
                                + "Non-Shrine Collaborator: " + nonShrineCollaborator + "\n"
                                + "Non-Shrine Collaborator Info: \"" + nonShrineCollaboratorInfo + "\"";

                        SheriffMailUtil.getInstance().sendDataStewardNotification(mailSubject, mailBody);
                    }

                    cacheEntries(session);

                    nextURL = MVC.nextUserJsp(MVC.Action.read);

                    break;

                case retract:

                    entry = sheriff.readById(entryID);
                    nextURL = MVC.nextUserJsp(MVC.Action.read);
                    if(INFO)
                    {
                        log.info("Retracting: " + String.valueOf(entry));
                    }
                    sheriff.retract(entry);
                    cacheEntries(session);
                    break;

                case resubmit:

                    entry = sheriff.readById(entryID);
                    nextURL = MVC.nextUserJsp(MVC.Action.read);
                    if(INFO)
                    {
                        log.info("Resubmitting: " + String.valueOf(entry));
                    }
                    sheriff.resubmit(entry);
                    cacheEntries(session);
                    break;

                case read:

                    entry = sheriff.readById(entryID);
                    session.setAttribute(MVC.Session.entry.name(), entry);
                    nextURL = MVC.nextUserDetailJsp(MVC.Action.read);
                    break;

                case update:

                    if(INFO)
                    {
                        log.info("Updating: " + String.valueOf(entry));
                    }
                    SheriffEntry savedCopy = sheriff.readById(entryID);
                    savedCopy.setFollowUp(followUp);
                    savedCopy.setEmail(email);
                    sheriff.update(savedCopy);
                    cacheEntries(session);
                    nextURL = MVC.nextUserJsp(MVC.Action.read);
                    break;

                default:
                    if(INFO)
                    {
                        log.info("User " + ecommonsID + " viewing all open topics");
                    }
                    cacheEntries(session);
                    nextURL = MVC.nextUserJsp(MVC.Action.read);
                    break;
            }
        }
        catch(Exception e)
        {
            log.error("Error", e);
            try
            {
                StringWriter sw = new StringWriter();
                PrintWriter pw = new PrintWriter(sw);
                e.printStackTrace(pw);

                SheriffMailUtil.getInstance().sendSysAdminNotification(
                        "Sheriff Error: " + e.getMessage(),
                        "User:      " + ecommonsID + "\n" +
                                "Entry:   	" + String.valueOf(entry) + "\n\n" +

                                "Request:   " + String.valueOf(request) + "\n\n" +

                                "Exception: " + sw.toString());
            }
            finally
            {
                ;
            }
            throw new ServletException("Request failed", e);
        }

        request.getRequestDispatcher(nextURL).forward(request, response);
    }

    private void cacheEntries(HttpSession session)
    {
        try
        {
            String ecommonsID = (String) session.getAttribute(MVC.Session.eCommonsID.name());

            session.setAttribute(MVC.Session.entries.name(), sheriff.readByRequestorID(ecommonsID));

        }
        catch(Exception e)
        {
            log.fatal("Could not read from database", e);
        }
    }
}
