package net.shrine.adapter.query;

import edu.harvard.i2b2.crc.datavo.i2b2message.RequestMessageType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.InstanceRequestType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.MasterDeleteRequestType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.MasterRenameRequestType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.MasterRequestType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.ResultRequestType;
import net.shrine.serializers.crc.CRCRequestType;
import net.shrine.serializers.crc.CRCSerializer;
import org.spin.query.message.serializer.SerializationException;

import static net.shrine.serializers.crc.CRCRequestType.*;
import static org.spin.tools.Util.guardNotNull;

/**
 * @author clint
 *         <p/>
 *         Aug 10, 2010
 *         <p/>
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 */
public class LocalIDInserter
{
    public static final LocalIDInserter Null = new LocalIDInserter();

    private LocalIDInserter()
    {
        super();
    }

    void addMasterID(final String masterID) throws Exception
    {
        //NOOP by default
    }

    void addInstanceID(final String instanceID) throws Exception
    {
        //NOOP by default
    }

    void addResultIDs(final String resultID) throws Exception
    {
        //NOOP by default
    }

    public void addIDs(final String masterID, final String instanceID, final String resultID) throws Exception
    {
        addMasterID(masterID);
        addInstanceID(instanceID);
        addResultIDs(resultID);
    }

    private static final LocalIDInserter forMasterRequest(final RequestMessageType request) throws SerializationException
    {
        final MasterRequestType masterRequest = CRCSerializer.getMasterRequest(request);

        return new LocalIDInserter()
        {
            @Override
            public void addMasterID(final String masterID)
            {
                guardNotNull(masterID, "Unknown master ID");
                masterRequest.setQueryMasterId(masterID);
            }
        };
    }

    private static final LocalIDInserter forInstanceRequest(final RequestMessageType request) throws SerializationException
    {
        final InstanceRequestType instanceRequest = CRCSerializer.getInstanceRequest(request);

        return new LocalIDInserter()
        {
            @Override
            public void addInstanceID(final String instanceID)
            {
                guardNotNull(instanceID, "Unknown instance ID");
                instanceRequest.setQueryInstanceId(instanceID);
            }
        };
    }

    private static final LocalIDInserter forResultRequest(final RequestMessageType request) throws SerializationException
    {
        final ResultRequestType instanceRequest = CRCSerializer.getResultRequest(request);

        return new LocalIDInserter()
        {
            @Override
            public void addResultIDs(final String resultID)
            {
                guardNotNull(resultID, "Unknown result ID");
                instanceRequest.setQueryResultInstanceId(resultID);
            }
        };
    }

    private static LocalIDInserter forMasterDeleteRequest(RequestMessageType request) throws SerializationException
    {
        final MasterDeleteRequestType masterRequest = CRCSerializer.getMasterDeleteRequest(request);

        return new LocalIDInserter()
        {
            @Override
            public void addMasterID(final String masterID)
            {
                guardNotNull(masterID, "Unknown master ID");
                masterRequest.setQueryMasterId(masterID);
            }
        };
    }

    private static LocalIDInserter forMasterRenameRequest(RequestMessageType request) throws SerializationException
    {
        final MasterRenameRequestType masterRequest = CRCSerializer.getMasterRenameRequest(request);

        return new LocalIDInserter()
        {
            @Override
            public void addMasterID(final String masterID)
            {
                guardNotNull(masterID, "Unknown master ID");
                masterRequest.setQueryMasterId(masterID);
            }
        };
    }

    public static final LocalIDInserter forRequest(final RequestMessageType request) throws SerializationException
    {
        guardNotNull(request);

        final CRCRequestType requestType = CRCSerializer.getRequestType(request);

        if(requestType == MasterRequestType)
        {
            return forMasterRequest(request);
        }

        if(requestType == InstanceRequestType)
        {
            return forInstanceRequest(request);
        }

        if(requestType == ResultRequestType)
        {
            return forResultRequest(request);
        }

        if(requestType == MasterDeleteRequestType)
        {
            return forMasterDeleteRequest(request);
        }
        if(requestType == MasterRenameRequestType)
        {
            return forMasterRenameRequest(request);
        }

        return Null;
    }
}
