package net.shrine.adapter.query;

import edu.harvard.i2b2.crc.datavo.i2b2message.RequestMessageType;
import edu.harvard.i2b2.crc.datavo.i2b2message.ResponseMessageType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.*;
import net.shrine.adapter.dao.AdapterDAO;
import net.shrine.serializers.QueryResultIDList;
import net.shrine.serializers.ShrineHeader;
import net.shrine.serializers.ShrineMessage;
import net.shrine.serializers.ShrineMessageConstants;
import net.shrine.serializers.crc.CRCResponseType;
import net.shrine.serializers.crc.CRCSerializer;

import java.util.List;

import static org.spin.tools.Util.makeArrayList;

/**
 * @author clint
 *         <p/>
 *         Aug 23, 2010
 *         <p/>
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 */
public abstract class LocalToNetworkIDMapper
{
    public static final LocalToNetworkIDMapper Null = new LocalToNetworkIDMapper()
    {
        @Override
        public void map() throws Exception
        {
            //NOOP
        }
    };

    private LocalToNetworkIDMapper()
    {

    }

    public abstract void map() throws Exception;

    public static final LocalToNetworkIDMapper forResponse(final ShrineMessage<RequestMessageType> shrineRequest,
            final ResponseMessageType response,
            final AdapterDAO dao)
    {
        final CRCResponseType responseType = CRCSerializer.getResponseType(response);

        if(responseType == CRCResponseType.MasterResponseType)
        {
            return forMasterResponse(response, dao);
        }
        else if(responseType == CRCResponseType.InstanceResponseType)
        {
            return forInstanceResponse(response, dao);
        }
        else if(responseType == CRCResponseType.ResultResponseType)
        {
            return forResultResponse(response, dao);
        }
        else if(responseType == CRCResponseType.MasterInstanceResultResponseType)
        {
            return forRunQueryByDefResponse(shrineRequest, response);
        }

        return Null;
    }

    /**
     * Convert Longs to Strings, passing through nulls
     *
     * @param l
     * @return
     */
    private static final String longToString(final Long l)
    {
        if(l == null)
        {
            return null;
        }

        return String.valueOf(l);
    }

    private static final LocalToNetworkIDMapper forMasterResponse(final ResponseMessageType response, final AdapterDAO dao)
    {
        return new LocalToNetworkIDMapper()
        {
            @Override
            public void map() throws Exception
            {
                final MasterResponseType responsePayload = CRCSerializer.getMasterResponse(response);

                for(final QueryMasterType queryMaster : responsePayload.getQueryMaster())
                {
                    final String localMasterID = queryMaster.getQueryMasterId();

                    queryMaster.setQueryMasterId(longToString(dao.findNetworkMasterID(localMasterID)));
                }
            }
        };
    }

    private static final LocalToNetworkIDMapper forInstanceResponse(final ResponseMessageType response, final AdapterDAO dao)
    {
        return new LocalToNetworkIDMapper()
        {
            @Override
            public void map() throws Exception
            {
                final InstanceResponseType responsePayload = CRCSerializer.getInstanceResponse(response);

                for(final QueryInstanceType queryInstance : responsePayload.getQueryInstance())
                {
                    final String localMasterID = queryInstance.getQueryMasterId();

                    final String localInstanceID = queryInstance.getQueryInstanceId();

                    queryInstance.setQueryMasterId(longToString(dao.findNetworkMasterID(localMasterID)));

                    queryInstance.setQueryInstanceId(longToString(dao.findNetworkInstanceID(localInstanceID)));
                }
            }
        };
    }

    private static final LocalToNetworkIDMapper forResultResponse(final ResponseMessageType response, final AdapterDAO dao)
    {
        return new LocalToNetworkIDMapper()
        {
            @Override
            public void map() throws Exception
            {
                final ResultResponseType responsePayload = CRCSerializer.getResultResponse(response);

                for(final QueryResultInstanceType resultInstance : responsePayload.getQueryResultInstance())
                {
                    final String localInstanceID = resultInstance.getQueryInstanceId();

                    final String localResultID = resultInstance.getResultInstanceId();

                    resultInstance.setQueryInstanceId(longToString(dao.findNetworkInstanceID(localInstanceID)));

                    resultInstance.setResultInstanceId(longToString(dao.findNetworkResultID(localResultID)));
                }
            }
        };
    }

    private static final LocalToNetworkIDMapper forRunQueryByDefResponse(final ShrineMessage<RequestMessageType> shrineRequest, final ResponseMessageType response)
    {
        // TODO: it's silly to have to reparse the headers here.
        // TODO: We should refactor this localToNetwork behavior to not be a shared class
        final ShrineHeader header = shrineRequest.getHeader();
        final Long queryMasterId = (Long)header.getHeader(ShrineMessageConstants.QUERY_MASTER_ID);
        final Long queryInstanceId = (Long)header.getHeader(ShrineMessageConstants.QUERY_INSTANCE_ID);
        final List<Long> queryResultIds = ((QueryResultIDList)header.getHeader(ShrineMessageConstants.QUERY_RESULT_IDS)).resultList;

        //defensive copy
        final List<Long> resultIDsCopy = makeArrayList(queryResultIds);

        return new LocalToNetworkIDMapper()
        {
            @Override
            public void map() throws Exception
            {
                final MasterInstanceResultResponseType masterInstanceResultResponse = CRCSerializer.getQueryMaster(response);

                masterInstanceResultResponse.getQueryMaster().setQueryMasterId(Long.toString(queryMasterId));

                masterInstanceResultResponse.getQueryInstance().setQueryInstanceId(Long.toString(queryInstanceId));

                for(int i = 0; i < resultIDsCopy.size(); ++i)
                {
                    masterInstanceResultResponse.getQueryResultInstance().get(i).setResultInstanceId(Long.toString(resultIDsCopy.get(i)));
                }
            }
        };
    }
}
