package net.shrine.adapter.query;

import edu.harvard.i2b2.crc.datavo.i2b2message.RequestMessageType;
import edu.harvard.i2b2.crc.datavo.i2b2message.ResponseMessageType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.MasterInstanceResultResponseType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.ObjectFactory;
import edu.harvard.i2b2.crc.datavo.setfinder.query.QueryDefinitionRequestType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.QueryDefinitionType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.QueryResultInstanceType;
import net.shrine.adapter.AdapterException;
import net.shrine.adapter.AdapterLockoutException;
import net.shrine.adapter.GaussianObfuscator;
import net.shrine.adapter.dao.AdapterDAO;
import net.shrine.adapter.dao.IDPair;
import net.shrine.adapter.dao.MasterTuple;
import net.shrine.adapter.dao.RequestResponseData;
import net.shrine.adapter.dao.ResultTuple;
import net.shrine.adapter.dao.UserAndMaster;
import net.shrine.config.ShrineConfig;
import net.shrine.dao.DAOException;
import net.shrine.serializers.QueryResultIDList;
import net.shrine.serializers.ShrineHeader;
import net.shrine.serializers.ShrineMessage;
import net.shrine.serializers.ShrineMessageConstants;
import net.shrine.serializers.crc.CRCParsedResponse;
import net.shrine.serializers.crc.CRCQueryStatus;
import net.shrine.serializers.crc.CRCSerializer;
import net.shrine.serializers.hive.HiveCommonSerializer;
import net.shrine.serializers.hive.HiveJaxbContext;
import net.shrine.translators.Translator;
import net.shrine.translators.Translators;
import org.apache.log4j.Logger;
import org.spin.node.QueryContext;
import org.spin.node.actions.QueryException;
import org.spin.query.message.headers.QueryInfo;
import org.spin.query.message.serializer.SerializationException;
import org.spin.tools.NetworkTime;
import org.spin.tools.crypto.signature.Identity;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.datatype.XMLGregorianCalendar;
import java.io.StringWriter;
import java.util.Date;
import java.util.List;

import static org.spin.tools.Util.guardNotNull;
import static org.spin.tools.Util.range;

/**
 * @author Andrew McMurry, MS
 * @date Feb 19, 2010 (REFACTORED 1.6.6)
 * @link http://cbmi.med.harvard.edu
 * @link http://chip.org
 * <p/>
 * NOTICE: This software comes with NO guarantees whatsoever and is
 * licensed as Lgpl Open Source
 * @link http://www.gnu.org/licenses/lgpl.html
 */

public final class RunQueryInstanceFromQueryDefinitionQuery extends AdapterQuery {
    public static final Logger log = Logger.getLogger(RunQueryInstanceFromQueryDefinitionQuery.class);
    public static final boolean DEBUG = log.isDebugEnabled();
    public static final boolean INFO = log.isInfoEnabled();

    private final ShrineConfig config;

    public RunQueryInstanceFromQueryDefinitionQuery(final ShrineConfig config, final Translator<RequestMessageType> requestTranslator, final AdapterDAO dao) {
        super(config.getRealCRCEndpoint(), dao, requestTranslator, Translators.<ResponseMessageType>nullTranslator());

        guardNotNull(config);
        this.config = config;
    }

    /**
     * Test the audit log to see if the requestor is locked out
     *
     * @param identity
     * @return
     */
    private boolean isLockedOut(final Identity identity) {
        if(DEBUG) {
            log.debug("CHECKING FOR LOCKOUT " + identity.getUsername());
        }

        if(config.getAdapterLockoutAttemptsThreshold() == 0) {
            return false;
        }

        return dao.isUserLockedOut(identity, config.getAdapterLockoutAttemptsThreshold());
    }


    private Long getNetworkMasterId(ShrineHeader header) {
        return (Long) header.getHeader(ShrineMessageConstants.QUERY_MASTER_ID);
    }

    private Long getNetworkInstanceId(ShrineHeader header) {
        return (Long) header.getHeader(ShrineMessageConstants.QUERY_INSTANCE_ID);
    }

    private QueryResultIDList getNetworkResultIDList(ShrineHeader header) {
        return (QueryResultIDList) header.getHeader(ShrineMessageConstants.QUERY_RESULT_IDS);
    }

    private void validateRequiredHeaders(ShrineMessage<RequestMessageType> shrineMessage) throws AdapterException {
        if(getNetworkMasterId(shrineMessage.getHeader()) == null ||
                getNetworkInstanceId(shrineMessage.getHeader()) == null ||
                getNetworkResultIDList(shrineMessage.getHeader()) == null) {
            throw new AdapterException("wasn't able to parse required shrine headers " + shrineMessage.getHeader());
        }
    }

    @Override
    protected void beforeQuery(final QueryContext queryContext, final ShrineMessage<RequestMessageType> request) throws Exception {
        super.beforeQuery(queryContext, request);
        validateRequiredHeaders(request);
        doLockoutCheck(queryContext);
    }

    @Override
    protected ResponseMessageType afterQuery(final QueryContext queryContext, final ShrineMessage<RequestMessageType> shrineMessage, final ResponseMessageType response, final long duration) throws Exception {
        responseTranslator.translate(response);

        final CRCParsedResponse parsedCRCResponse = CRCSerializer.getParsedResponse(response);

        guardSameNumberOfResults(shrineMessage, parsedCRCResponse);

        final Identity requestor = queryContext.getQueryInfo().getIdentity();
        RequestMessageType headerRequest = shrineMessage.getPayload();
        String queryDefinitionString = extractQueryDefinitionString(headerRequest);

        logCRCResponseAndCreateIDMappings(queryContext.getQueryInfo().getQueryID(), requestor, shrineMessage.getHeader(), parsedCRCResponse, duration, queryDefinitionString);

        mapLocalIDsToNetworkIDs(shrineMessage, response);

        obfuscateIfNecessary(response);

        return response;
    }

    private String extractQueryDefinitionString(RequestMessageType headerRequest)
            throws SerializationException, JAXBException {
        QueryDefinitionRequestType request = CRCSerializer.getQueryDefinitionRequest(headerRequest);
        QueryDefinitionType definition = request.getQueryDefinition();

        JAXBElement<QueryDefinitionType> definitionEl = new ObjectFactory().createQueryDefinition(definition);
        final Marshaller m = HiveJaxbContext.getInstance().getContext().createMarshaller();
        m.setProperty(Marshaller.JAXB_FRAGMENT, Boolean.TRUE);
        final StringWriter sw = new StringWriter();
        m.marshal(definitionEl, sw);
        String queryDefinitionString = sw.toString();
        return queryDefinitionString;
    }

    void logCRCResponseAndCreateIDMappings(final String spinId, final Identity identity, final ShrineHeader header, final CRCParsedResponse parsedResponse, final long elapsed, String queryDefinitionString) throws DAOException {
        if(DEBUG) {
            log.debug("Logging the parsed response from the CRC for logging, audit, and lockout purposes. ");
        }

        final List<Long> networkResultIds = getNetworkResultIDList(header).resultList;

        final List<String> localResultIds = parsedResponse.getCrcQueryResultIds();

        XMLGregorianCalendar creationTime = determineCreationTime(parsedResponse);
        dao.insertUserAndMasterIDMapping(new UserAndMaster(identity.getDomain(),
                identity.getUsername(),
                getNetworkMasterId(header),
                parsedResponse.getCrcMasterName(),
                new Date(creationTime.toGregorianCalendar().getTimeInMillis())));

        dao.insertMaster(new MasterTuple(IDPair.of(getNetworkMasterId(header), parsedResponse.getCrcMasterQueryID()), queryDefinitionString));

        dao.insertInstanceIDPair(IDPair.of(getNetworkInstanceId(header), parsedResponse.getCrcQueryInstanceId()));

        for(final int i : range(networkResultIds.size())) {
            final RequestResponseData data = new RequestResponseData(identity.getDomain(),
                    identity.getUsername(),
                    getNetworkMasterId(header),
                    getNetworkInstanceId(header),
                    networkResultIds.get(i),
                    parsedResponse.getResultStatus(),
                    parsedResponse.getResultSetSize(),
                    elapsed,
                    spinId,
                    parsedResponse.getResultXML());

            dao.insertRequestResponseData(data);

            dao.insertResultTuple(new ResultTuple(IDPair.of(networkResultIds.get(i), localResultIds.get(i))));
        }
    }

    private XMLGregorianCalendar determineCreationTime(CRCParsedResponse parsedResponse) {

        XMLGregorianCalendar masterCreateDate = parsedResponse.getCrcMasterCreateDate();
        if(masterCreateDate != null) {
            return masterCreateDate;
        }
        return new NetworkTime().getXMLGregorianCalendar();
    }

    private void guardSameNumberOfResults(final ShrineMessage<RequestMessageType> shrineMessage, final CRCParsedResponse parsed) throws QueryException {
        if(mismatchedNumberOfResults(shrineMessage, parsed)) {
            throw new QueryException("There were a different number of results than expected.  " +
                    "Expected:" + parsed.getCrcQueryResultIds().size() + ", Actual:" +
                    getNetworkResultIDList(shrineMessage.getHeader()).resultList.size());
        }
    }

    private boolean mismatchedNumberOfResults(final ShrineMessage<RequestMessageType> shrineMessage, final CRCParsedResponse parsed) {
        return parsed.getCrcQueryResultIds().size() != getNetworkResultIDList(shrineMessage.getHeader()).resultList.size();
    }

    private void obfuscateIfNecessary(final ResponseMessageType response) throws SerializationException, DAOException {
        if(shouldObfuscate(response)) {
            final MasterInstanceResultResponseType result = HiveCommonSerializer.getBodyNode(response, 0, MasterInstanceResultResponseType.class);

            for(final QueryResultInstanceType resultType : result.getQueryResultInstance()) {
                int obfuscationAmount = GaussianObfuscator.obfuscate(resultType);
                dao.updateObfuscationAmount(resultType.getResultInstanceId(), obfuscationAmount);
            }
        }
    }

    private boolean shouldObfuscate(final ResponseMessageType response) {
        return config.isSetSizeObfuscationEnabled() && CRCSerializer.getQueryStatus(response).equals(CRCQueryStatus.COMPLETED);
    }

    private void doLockoutCheck(final QueryContext queryContext) throws AdapterLockoutException {
        final QueryInfo queryInfo = queryContext.getQueryInfo();

        final Identity identity = queryInfo.getIdentity();

        if(isLockedOut(identity)) {
            throw new AdapterLockoutException(identity);
        }
    }
}
