package net.shrine.adapter.translators;

import edu.harvard.i2b2.crc.datavo.i2b2message.RequestMessageType;
import edu.harvard.i2b2.crc.datavo.pdo.query.GetPDOFromInputListRequestType;
import edu.harvard.i2b2.crc.datavo.pdo.query.ItemType;
import edu.harvard.i2b2.crc.datavo.pdo.query.PanelType;
import net.shrine.adapter.dao.AdapterDAO;
import net.shrine.adapter.dao.RequestResponseData;
import net.shrine.config.AdapterMappings;
import net.shrine.dao.DAOException;
import net.shrine.serializers.ShrineJAXBUtils;
import net.shrine.serializers.crc.CRCSerializer;
import net.shrine.translators.TranslationException;
import net.shrine.translators.Translator;
import org.apache.log4j.Logger;
import org.spin.query.message.serializer.SerializationException;
import org.spin.tools.config.ConfigException;
import org.springframework.beans.factory.annotation.Required;

import javax.xml.bind.JAXBException;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

/**
 * This class does all of the request specific translation to support the PDO
 * query.
 *
 * @author Justin Quan
 * @date Jul 26, 2010
 * @link http://cbmi.med.harvard.edu
 * @link http://chip.org
 * <p/>
 * NOTICE: This software comes with NO guarantees whatsoever and is
 * licensed as Lgpl Open Source
 * @link http://www.gnu.org/licenses/lgpl.html
 */
public final class GetPDOFromInputListTranslator implements Translator<RequestMessageType>
{
    private static final Logger log = Logger.getLogger(GetPDOFromInputListTranslator.class);

    private final AdapterMappings adapterMappings;
    private AdapterDAO adapterDAO;

    public GetPDOFromInputListTranslator() throws ConfigException
    {
        adapterMappings = AdapterMappings.getDefaultInstance();
    }

    public GetPDOFromInputListTranslator(final Map<String, List<String>> adaptorMappings)
    {
        adapterMappings = new AdapterMappings();

        for(final String s : adaptorMappings.keySet())
        {
            for(final String s2 : adaptorMappings.get(s))
            {
                adapterMappings.addMapping(s, s2);
            }
        }
    }

    @Override
    public void translate(final RequestMessageType request) throws TranslationException
    {
        try
        {
            final GetPDOFromInputListRequestType pdoQueryDef = CRCSerializer.getPDOInputListRequest(request);

            // replace result_id with its local equivalent
            final long patientSetCollId = Long.parseLong(pdoQueryDef.getInputList().getPatientList().getPatientSetCollId());

            final RequestResponseData previousBroadcast = adapterDAO.findRequestResponseDataByResultID(patientSetCollId);

            if(previousBroadcast == null)
            {
                // we've never seen this broadcast before, which means we have
                // no way to translate it to a local concept.
                log.error("Unrecognized broadcaster patientSetCollId:" + patientSetCollId);

                throw new TranslationException("Unrecognized broadcaster patientSetCollId");
            }

            final String localResultID = adapterDAO.findLocalResultID(previousBroadcast.getBroadcastResultInstanceId());

            pdoQueryDef.getInputList().getPatientList().setPatientSetCollId(localResultID);

            // replace item_key in the panel->item node.
            if(adapterMappings.size() == 0)
            {
                log.debug("No Translations found, performing direct pass through");
            }
            else
            {
                for(final PanelType panel : pdoQueryDef.getFilterList().getPanel())
                {
                    final ListIterator<ItemType> itemIter = panel.getItem().listIterator();

                    while(itemIter.hasNext())
                    {
                        final ItemType item = itemIter.next();
                        itemIter.remove();
                        final List<String> locals = adapterMappings.getMappings(item.getItemKey());
                        for(final String local : locals)
                        {
                            try
                            {
                                ItemType translatedItem = ShrineJAXBUtils.copy(item);
                                translatedItem.setItemKey(local);
                                itemIter.add(translatedItem);
                            }
                            catch(final JAXBException jaxbe)
                            {
                                final String msg = "Translation error- failed to copy construct :" + String.valueOf(item);
                                log.error(msg, jaxbe);
                                throw new TranslationException(msg, jaxbe);
                            }
                        }
                    }
                }
            }

            CRCSerializer.updatePDOQuery(request, pdoQueryDef);

        }
        catch(final DAOException e)
        {
            log.error("Unable to read broadcaster mapping", e);
            throw new TranslationException("Unable to read broadcaster mapping", e);
        }
        catch(final SerializationException e)
        {
            log.error("Problem deserializing pdo xml");
            throw new TranslationException("Problem deserializing pdo xml", e);
        }
    }

    @Required
    public void setAdapterDAO(final AdapterDAO adapterDAO)
    {
        this.adapterDAO = adapterDAO;
    }
}
