package net.shrine.adapter.translators;

import java.util.Map;

import net.shrine.config.ProjectMappings;
import net.shrine.translators.TranslationException;
import net.shrine.translators.Translator;

import org.apache.log4j.Logger;
import org.spin.tools.config.ConfigException;

import edu.harvard.i2b2.crc.datavo.i2b2message.RequestMessageType;

/**
 * @Author David Ortiz
 * @Date June 28, 2010
 *       <p/>
 *       This implementation of Translator is particularly intolerant of missing
 *       mappings in the Adaptor. It is meant for use in the Harvard deployment
 *       where each concept should be mapped and an unmapped concept is an error
 *       condition and we should fail as noisily as possible.
 *       <p/>
 *       Other shrine deployments such as CARRAnet where mapping is less import,
 *       the simple translator is more tolerant of errors while still allowing
 *       for some mapping to take place.
 */
public final class ProjectTranslator implements Translator<RequestMessageType>
{
    public static final Logger log = Logger.getLogger(ProjectTranslator.class);
    public static final boolean DEBUG = log.isDebugEnabled();
    public static final String DEFAULT_MAP_ENTRY = "default";

    protected final ProjectMappings mappings;

    public ProjectTranslator()
    {
        mappings = getMappings();
    }

    // If no mappings found, this shouldn't be an error
    private static ProjectMappings getMappings()
    {
        try
        {
            return ProjectMappings.getDefaultInstance();
        }
        catch(final ConfigException e)
        {
            log.info("Project Mappings disabled, no ProjectMappings.xml found");

            return new ProjectMappings();
        }
    }

    public ProjectTranslator(final ProjectMappings mappings)
    {
        this.mappings = mappings;
    }

    /**
     * This grabs the project mapping
     * 
     * @param original
     * @return
     * @throws Exception
     */
    @Override
    public void translate(final RequestMessageType original) throws TranslationException
    {
        final Map<String, String> projectMap = mappings.getMappings();

        final String projectName = original.getMessageHeader().getProjectId();

        // this is a really bad message
        if(projectName == null)
        {
            throw new TranslationException("This message is missing a project?!?");
        }

        // We found an explicit mapping
        if(projectMap.containsKey(projectName))
        {
            original.getMessageHeader().setProjectId(projectMap.get(projectName));
            return;
        }

        // Either we use explicit mapping, or a default must be set
        final String defaultMapping = projectMap.get(DEFAULT_MAP_ENTRY);
       
        if(defaultMapping != null)
        {
            original.getMessageHeader().setProjectId(defaultMapping);
        }

        log.info("No mapping found for Project: " + projectName);
    }
}
