package net.shrine.monitor.scanner;

import edu.harvard.i2b2.crc.datavo.i2b2message.RequestMessageType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.QueryDefinitionType;
import net.shrine.serializers.crc.CRCSerializer;
import net.shrine.serializers.crc.QueryDefBuilder;
import org.spin.query.message.serializer.SerializationException;
import org.spin.tools.NetworkTime;
import org.spin.tools.PKITool;
import org.spin.tools.Util;
import org.spin.tools.config.ConfigException;

import java.io.IOException;
import java.util.HashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * @author Andrew McMurry, MS
 *         <p/>
 *         With primary support from Children's Hospital Informatics Program @
 *         Harvard-MIT Health Sciences and Technology and
 *         <p/>
 *         Secondary support from the Harvard Medical School
 *         Center for BioMedical Informatics
 *         <p/>
 *         PHD candidate, Boston University Bioinformatics
 *         Member, I2b2 National Center for Biomedical Computing
 *         <p/>
 *         All works licensed under LGPL
 *         <p/>
 *         User: andy
 *         Date: Oct 15, 2010
 *         Time: 4:10:50 PM
 */
class Main
{
    public static void main(String[] args) throws ConfigException
    {
        if(args.length < 1)
        {
            showUsage();
            System.exit(0);
        }
        else
        {
            try
            {
                PKITool.getInstance();

                RuntimeArgs runtime = new RuntimeArgs(args[0]);

                println("Runtime Settings: ");
                println(runtime.toString());

                if(runtime.source == null)
                {
                    println("Source was not set");
                    showUsage();
                    System.exit(0);
                }

                Main main = new Main();
                main.run(runtime);
            }
            catch(ConfigException e)
            {
                println("Config problem?" + e.getMessage());
                System.exit(0);
            }
            catch(Exception e)
            {
                e.printStackTrace();
                println("Exception:" + e.getMessage());
            }
        }
    }

    public void run(RuntimeArgs runtime) throws ConfigException, SerializationException, IOException, InterruptedException, ExecutionException
    {
        int     COUNT_ERROR = 0,
                COUNT_COMPLETE = 0,
                COUNT_INCOMPLETE = 0;

        do //at least one pass through the input source
        {
            QueryDefIterator input = runtime.queryDefIterator();
            ExecutorService executor = Executors.newFixedThreadPool(runtime.threads);

            while(input.hasNext())
            {
                HashMap<Integer, Future<ReqResPair>> futures = Util.makeHashMap();

                for(int t = 1; t <= runtime.threads; t++)
                {
                    QueryDefinitionType definition = input.next();

                    definition.setQueryName("Heartbeat " + System.currentTimeMillis());

                    RequestMessageType request = QueryDefBuilder.getRequestPSM(definition, runtime.getSecurityType(), runtime.project);

                    //Set Project ID
                    request.getMessageHeader().setProjectId(runtime.project);

                    if(runtime.queryTopicID > 0)
                    {
                        CRCSerializer.setQueryTopicId(request, (long) runtime.queryTopicID);
                    }

                    println("Submitting RequestExecutor thread " + t);
                    futures.put(t, executor.submit(
                            new ReqThread(runtime.crc, request, runtime.anonymize))
                    );

                    println("Sleeping for ms: " + runtime.sleep);
                    Thread.sleep(runtime.sleep);
                }

                for(Integer t : futures.keySet())
                {
                    println("Waiting on thread " + t);

                    ReqResPair pair = futures.get(t).get();

                    switch(pair.getStatus())
                    {
                        case COMPLETED:
                            COUNT_COMPLETE++;
                            break;
                        case INCOMPLETE:
                            COUNT_INCOMPLETE++;
                            break;
                        case ERROR:
                            COUNT_ERROR++;
                            break;
                        default:
                            throw new RuntimeException("Unheard of status returned by CRC: " + pair.getStatus().name());
                    }
                }
                println(new NetworkTime().toString());
                println("COMPLETED=" + COUNT_COMPLETE + " ERROR=" + COUNT_ERROR + " INCOMPLETE=" + COUNT_INCOMPLETE);
            }
        }
        while(runtime.repeat); //repeat (such as in "heartbeat" mode)   
    }

    public static void showUsage()
    {
        println("net.shrine.monitor.scanner.Main scanner.properties ");

        println("options: ");

        for(String arg : RuntimeArgs.supported())
        {
            println(arg);
        }
    }

    private static void println(String line)
    {
        System.out.println(line);
    }
}
