package net.shrine.monitor.scanner;

import edu.harvard.i2b2.crc.datavo.setfinder.query.ItemType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.PanelType;
import edu.harvard.i2b2.crc.datavo.setfinder.query.QueryDefinitionType;
import net.shrine.serializers.crc.CRCSerializer;
import org.spin.extension.BatchProcessing;
import org.spin.query.message.serializer.SerializationException;
import org.spin.tools.FileUtils;
import org.spin.tools.StringTools;

import java.io.*;

/**
 * @author Andrew McMurry, MS
 *         <p/>
 *         With primary support from Children's Hospital Informatics Program @
 *         Harvard-MIT Health Sciences and Technology and
 *         <p/>
 *         Secondary support from the Harvard Medical School
 *         Center for BioMedical Informatics
 *         <p/>
 *         PHD candidate, Boston University Bioinformatics
 *         Member, I2b2 National Center for Biomedical Computing
 *         <p/>
 *         All works licensed under LGPL
 *         <p/>
 *         User: andy
 *         Date: Oct 19, 2010
 *         Time: 4:37:04 PM
 */
public class OutputWriter
{
    private File batch;
    private Writer queryWriter, panelWriter, nodeWriter, errorWriter, warningWriter;

    private OutputWriter() throws IOException
    {
        batch = BatchProcessing.getBatchProcessingLabel();
        batch.mkdir();

        System.out.println("BATCH = "+batch.getAbsolutePath());

        queryWriter   = getWriter("query_input.csv");
        panelWriter   = getWriter("panel_input.csv");
        nodeWriter    = getWriter("node_output.csv");
        errorWriter   = getWriter("error.log");
        warningWriter = getWriter("warn.log");

        headers(); 
    }

    private static OutputWriter instance = null;

    public static OutputWriter getInstance() throws IOException
    {
        if(instance==null) instance = new OutputWriter();

        return instance;
    }

    private static String CSV  = ",";
    private static String CRLF = StringTools.crlf();

    public synchronized void write(ReqResPair pair) throws IOException, SerializationException
    {
        xml(pair);

        synchronized (queryWriter)
        {
            queryWriter.write(csvQueryInput(pair));
            queryWriter.write(CRLF);
            queryWriter.flush();
        }

        synchronized (panelWriter)
        {
            panelWriter.write(csvPanelDetail(pair));
            panelWriter.write(CRLF);
            panelWriter.flush();
        }

        synchronized (nodeWriter)
        {
            for(ReqResPair.NodeResponse singleNode : pair.getNodes())
            {
                nodeWriter.write(csvResponse(singleNode));
                nodeWriter.write(CRLF);
                nodeWriter.flush();
            }
        }
    }

    public synchronized void close() throws IOException
    {       
        queryWriter.close();
        panelWriter.close();
        nodeWriter.close();
    }

    public synchronized void warn(String message) throws IOException
    {
        System.out.println("WARN: "+ message);

        warningWriter.write(message + CRLF);
        warningWriter.flush();
    }

    public synchronized void error(String message, Exception e) throws IOException
    {
        System.out.println("ERROR: "+ e.getMessage());

        errorWriter.write(message + ":"+ e.getMessage());
        errorWriter.write(CRLF);
        errorWriter.flush();
    }

    private void headers() throws IOException
    {
        queryWriter.write("-- queryMasterID, status, panels, items");
        queryWriter.write(CRLF);
        queryWriter.flush();
        
        nodeWriter.write("-- queryMasterID, nodeID, setSize, startTime, endTime");
        nodeWriter.write(CRLF);
        nodeWriter.flush();
    }

    private String csvQueryInput(ReqResPair pair)
    {
        StringBuilder csv = new StringBuilder();

        csv .append(pair.getQueryMasterID())
            .append(CSV).append(pair.getStatus().name())
            .append(CSV).append(pair.getPanels())
            .append(CSV).append(pair.getItems());

        return csv.toString();
    }

    private String csvPanelDetail(ReqResPair pair) throws IOException
    {
        StringBuilder csv = new StringBuilder();

        try
        {
            QueryDefinitionType queryDef =
                    CRCSerializer.getQueryDefinitionRequest(
                        pair.getRequest()).
                        getQueryDefinition();
            for(PanelType panel : queryDef.getPanel())
            {
                for(ItemType item : panel.getItem())
                {
                    csv .append(pair.getQueryMasterID())
                        .append(CSV).append(panel.getPanelNumber())
                        .append(CSV).append(replaceBackslashes(item.getItemKey()));
                }
            }
        }
        catch(Exception e)
        {
            error("Could not read query definition?", e);
        }

        return csv.toString();
    }
    
    String replaceBackslashes(String concept_path)
    {
        final String ESCAPE_WINDOWS = "\\\\";
        final String ESCAPE_UNIX = "/";

        return concept_path.replaceAll(ESCAPE_WINDOWS, ESCAPE_UNIX);
    }


    private String csvResponse(ReqResPair.NodeResponse node)
    {
        StringBuilder csv = new StringBuilder();

        csv .append(node.queryMasterID)
            .append(CSV).append(node.nodeID)
            .append(CSV).append(node.setSize)
            .append(CSV).append(node.startTime)
            .append(CSV).append(node.endTime);

        return csv.toString();
    }

    private void xml(ReqResPair pair) throws SerializationException
    {
        File requestFile  = new File(batch + File.separator + "xml" + File.separator + pair.getQueryMasterID()+  "_req.xml");
        File responseFile = new File(batch + File.separator + "xml" + File.separator + pair.getQueryMasterID() + "_res.xml");

        requestFile.getParentFile().mkdir();

        try
        {
            String requestXML  = CRCSerializer.toXMLString(pair.getRequest());
            String responseXML = CRCSerializer.toXMLString(pair.getResponse());

            FileUtils.write(requestFile,  requestXML);
            FileUtils.write(responseFile, responseXML);
        }
        catch (Exception e)
        {
            throw new SerializationException(e);
        }
    }

    private Writer getWriter(String filename) throws IOException
    {
        File path = new File(batch + File.separator + filename);

        return getWriter(path);
    }

    private Writer getWriter(File file) throws IOException
    {
        System.out.println("Buffering to : "+file.getAbsolutePath());

        return new BufferedWriter(new FileWriter(file));
    }

    public File getBatchOutputDir()
    {
        return batch; 
    }
}
