package net.shrine.monitor.scanner;

import edu.harvard.i2b2.crc.datavo.i2b2message.PasswordType;
import edu.harvard.i2b2.crc.datavo.i2b2message.SecurityType;
import org.apache.log4j.Logger;
import org.spin.tools.StringTools;
import org.spin.tools.config.ConfigException;
import org.spin.tools.config.ConfigTool;

import java.io.File;
import java.io.IOException;
import java.util.Properties;

/**
 * @author Andrew McMurry, MS
 *         <p/>
 *         With primary support from Children's Hospital Informatics Program @
 *         Harvard-MIT Health Sciences and Technology and
 *         <p/>
 *         Secondary support from the Harvard Medical School
 *         Center for BioMedical Informatics
 *         <p/>
 *         PHD candidate, Boston University Bioinformatics
 *         Member, I2b2 National Center for Biomedical Computing
 *         <p/>
 *         All works licensed under LGPL
 *         <p/>
 *         User: andy
 *         Date: Oct 19, 2010
 *         Time: 2:33:03 PM
 */
public class RuntimeArgs
{
    public static final Logger log = Logger.getLogger(RuntimeArgs.class);

    //REQUIRED
    public Source source;
    public String file;
    public String crc       = "http://services.i2b2.org/i2b2/rest/QueryToolService/"; //i2b2 CRC or SHRINE BroadcasterAggregator, from the perspective of the client it makes almost no difference.
    public String domain    = "HarvardDemo";
    public String project   = "Demo";
    public String username  = "demo";
    public String password  = "demouser";
    //--

    //OPTIONAL WITH SENSIBLE DEFAULTS
    public int      queryTopicID = 0;     //Default=Do not include queryTopicID          :: send query directly to i2b2 or SHRINE broadcaster aggregator
    public int      threads      = 1;     //Default=1 thread only
    public int      sleep         = 1000;  //Default=Sleep for 1 second in between each query (per thread basis)

    public boolean  translate    = false; //Default=Do not translate query definition    :: send query directly to i2b2 or SHRINE broadcaster aggregator
    public boolean  random       = false; //Default=Do not randomize :: send each query in order.
    public boolean  repeat       = false; //Default=Do not repeat    :: send each query only once.


    /**
     * This feature was added to keep the scanner from accruing large number of patient counts.
     */
    public boolean  anonymize    = false; //Default=Do not anonymize results prior to saving

    //--

    Properties props;

    public enum Source
    {
        AdapterMappings,  //Load from AdapterMappings.xml
        FlatFile,         //Load from flat file, can be used to build complex queries

        SavedQueries,    //Load user saved queries and "replay" each in succession
        OntologyCell     //Load from ontology cell (warning this can be very SLOW....)
    }

    public RuntimeArgs(String file) throws IOException
    {
        this(new File(file));
    }

    public RuntimeArgs(File file) throws IOException
    {
        this(ConfigTool.getProperties(file));
    }

    public RuntimeArgs(Properties properties)
    {
        props = properties;
        source = getSource(getProperty("source"));

        if(notNullKey("crc"))
        {
            crc = getProperty("crc");
        }

        if(notNullKey("domain"))
        {
            domain = getProperty("domain");
        }

        if(notNullKey("project"))
        {
            project = getProperty("project");
        }

        if(notNullKey("username"))
        {
            username = getProperty("username");
        }

        if(notNullKey("password"))
        {
            password = getProperty("password");
        }

        if(notNullKey("queryTopicID"))
        {
            queryTopicID = parseInt("queryTopicID");
        }

        if(notNullKey("threads"))
        {
            threads = parseInt("threads");
        }

        if(notNullKey("file"))
        {
            file = getProperty("file");
        }

        if(notNullKey("random"))
        {
            random = parseBool("random");
        }

        if(notNullKey("translate"))
        {
            translate = parseBool("translate");
        }

        if(notNullKey("repeat"))
        {
            repeat = parseBool("repeat");
        }

        if(notNullKey("sleep"))
        {
            sleep = parseInt("sleep");
        }

        if(notNullKey("anonymize"))
        {
            anonymize  = parseBool("anonymize");
        }
    }

    public boolean notNullKey(String key)
    {
        return props.containsKey(key) && props.getProperty(key).length() > 0;
    }

    public String getProperty(String key)
    {
        return props.getProperty(key);
    }

    public boolean parseBool(String key)
    {
        String value = getProperty(key);

        return value.equalsIgnoreCase("true") || value.equalsIgnoreCase("yes");
    }

    private Integer parseInt(String string)
    {
        if(string == null || string.length() == 0)
        {
            return null;
        }

        try
        {
            return Integer.parseInt(getProperty(string));
        }
        catch(Exception e)
        {
            return null;
        }
    }

    public QueryDefIterator queryDefIterator() throws ConfigException, IOException
    {
        return queryDefIterator(source, translate, file);
    }

    public static QueryDefIterator queryDefIterator(Source source, boolean translate, String argument) throws ConfigException, IOException
    {
        switch(source)
        {
            case AdapterMappings:
                return new AdapterMappingsIterator(argument, translate);

            case FlatFile:
                return new FlatFileIterator(new File(argument));

            default:
                throw new ConfigException("No Impl for source type " + source.name());
        }
    }

    public Source getSource(String source)
    {
        try
        {
            return Source.valueOf(source);
        }
        catch(Exception e)
        {
        }

        log.error("Invalid source " + source);

        return null;
    }

    public SecurityType getSecurityType()
    {
        SecurityType security = new SecurityType();

        security.setDomain(domain);
        security.setUsername(username);

        PasswordType pwd = new PasswordType();

        pwd.setIsToken(false);
        pwd.setValue(password);

        security.setPassword(pwd);

        return security;
    }

    public static String[] supported()
    {
        return new String[]{
                "source",
                "file",
                "crc",
                "domain",
                "project",
                "username",
                "password",
                "queryTopicID",
                "translate",
                "random",
                "threads",
                "sleep"
        };
    }

    public String toString()
    {
        StringBuilder append = new StringBuilder();

        for(String key : supported())
        {
            append.append(key).append("=").append(getProperty(key));
            append.append(StringTools.crlf());
        }

        return append.toString();
    }


}
