/*
 * Decompiled with CFR 0.152.
 */
package org.spin.tools;

import com.sun.net.ssl.internal.ssl.Provider;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.net.MalformedURLException;
import java.net.URI;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import org.apache.commons.codec.binary.Base64;
import org.apache.log4j.Logger;
import org.spin.tools.Util;
import org.spin.tools.config.ConfigException;
import org.spin.tools.config.ConfigTool;
import org.spin.tools.config.KeyStoreConfig;
import org.spin.tools.config.KeyStoreType;
import org.spin.tools.crypto.CryptoException;
import org.spin.tools.crypto.PKCryptor;
import org.spin.tools.crypto.signature.CertData;
import org.spin.tools.crypto.signature.CertID;

public final class PKITool {
    public static final String certificateFormat = "X.509";
    public static final String publicKeyAlgorithm = "RSA";
    public static final String signatureAlgorithm = "SHA1withRSA";
    static final Logger log = Logger.getLogger(PKITool.class);
    static final boolean DEBUG = log.isDebugEnabled();
    static final boolean INFO = log.isInfoEnabled();
    private static final Base64 base64Codec = new Base64();
    private static PKITool instance;
    private final KeyStoreConfig config;
    private KeyStoreType keystoreType;
    private final KeyStore keystore;
    private final CertificateFactory certFactory;
    private final Map<BigInteger, CertData> x509KeyInfoMap = Util.makeHashMap();
    private final CertData myX509;
    private final PrivateKey myPrivateKey;
    private final PublicKey myPublicKey;
    private final Map<Principal, X509Certificate> caCertificates = Util.makeHashMap();

    public static synchronized PKITool getInstance() throws ConfigException {
        if (instance == null) {
            if (DEBUG) {
                log.debug((Object)"instance == null");
            }
            instance = PKITool.makeInstance(ConfigTool.loadKeyStoreConfig());
        }
        return instance;
    }

    public static PKITool makeInstance(KeyStoreConfig keyStoreConfig) throws ConfigException {
        return new PKITool(keyStoreConfig);
    }

    static final URI getKeyStoreURI(KeyStoreConfig keyStoreConfig) throws ConfigException {
        if (keyStoreConfig.getFile() == null) {
            throw new ConfigException("KeyStoreConfig has null File property");
        }
        URI result = ConfigTool.getConfigFileAsURI(keyStoreConfig.getFile().getPath());
        if (INFO) {
            log.info((Object)("KeyStore URI: " + result));
        }
        return result;
    }

    private static final InputStream getKeyStoreStream(KeyStoreConfig keyStoreConfig) throws ConfigException {
        try {
            return PKITool.getKeyStoreURI(keyStoreConfig).toURL().openStream();
        }
        catch (IOException e) {
            throw new ConfigException("Error getting keystore stream", e);
        }
    }

    private PKITool(KeyStoreConfig config) throws ConfigException {
        if (DEBUG) {
            log.debug((Object)"Initializing Keystore");
        }
        Util.guardNotNull(config, "Null KeyStoreConfig");
        Util.guardNotNull((Object)config.getKeystoreType(), "Null KeyStoreType");
        Util.guardNotNull(config.getPasswordAsCharArray(), "Null keystore password");
        try {
            this.config = config;
            this.keystoreType = config.getKeystoreType();
            this.certFactory = CertificateFactory.getInstance(certificateFormat);
            this.keystore = KeyStore.getInstance(this.keystoreType.name());
            this.keystore.load(PKITool.getKeyStoreStream(config), config.getPasswordAsCharArray());
            KeyPair keyPair = PKITool.determineKeyPair(this.keystore, config);
            this.myX509 = keyPair.x509Data;
            this.myPrivateKey = keyPair.privateKey;
            this.myPublicKey = keyPair.publicKey;
            this.fillInCACertMap(config);
            this.fillKeyInfoMap();
            if (config.getSetSystemProperties()) {
                Workarounds.setSystemProperties(config);
            }
            this.debug();
        }
        catch (Exception e) {
            throw new ConfigException("Error initializing PKITool", e);
        }
    }

    private void fillInCACertMap(KeyStoreConfig keystoreConfig) throws KeyStoreException {
        if (keystoreConfig.getCaPublicKeyAliases().size() > 0) {
            for (String certAlias : keystoreConfig.getCaPublicKeyAliases()) {
                X509Certificate tempCert = this.getCertificate(certAlias);
                this.caCertificates.put(tempCert.getIssuerDN(), tempCert);
            }
        }
    }

    public KeyStoreType getKeystoreType() {
        return this.keystoreType;
    }

    private void fillKeyInfoMap() throws CryptoException {
        this.x509KeyInfoMap.clear();
        try {
            for (String alias : Util.iterable(this.keystore.aliases())) {
                X509Certificate cert = this.getCertificate(alias);
                CertData x509Data = PKITool.getX509Data(cert);
                this.x509KeyInfoMap.put(x509Data.getCertID().getSerial(), x509Data);
            }
        }
        catch (CryptoException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CryptoException(e);
        }
    }

    private X509Certificate getCertificate(String alias) throws KeyStoreException {
        return (X509Certificate)this.keystore.getCertificate(alias);
    }

    static final KeyPair determineKeyPair(KeyStore keystore, KeyStoreConfig config) throws CryptoException {
        Util.guardNotNull(keystore);
        Util.guardNotNull(config);
        int numKeyPairs = PKITool.countKeyPairs(keystore);
        if (numKeyPairs == 0) {
            throw new CryptoException("At least one public/private key pair is required.");
        }
        if (numKeyPairs == 1) {
            return PKITool.findKeyPair(keystore, config);
        }
        return PKITool.getKeyPairByKeyAlias(keystore, config);
    }

    static KeyPair findKeyPair(KeyStore keystore, KeyStoreConfig config) throws CryptoException {
        try {
            for (String alias : Util.iterable(keystore.aliases())) {
                if (!keystore.isKeyEntry(alias)) continue;
                return PKITool.getKeyPair(keystore, config, alias);
            }
        }
        catch (CryptoException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CryptoException(e);
        }
        throw new CryptoException("At least one public/private key pair is required.");
    }

    static KeyPair getKeyPairByKeyAlias(KeyStore keystore, KeyStoreConfig config) throws CryptoException {
        if (config.getKeyAlias() == null) {
            throw new CryptoException("There are " + PKITool.countKeyPairs(keystore) + " keypairs in " + config.getFile() + ".  Set the keyAlias property in keystore.xml to specify which keypair to use");
        }
        return PKITool.getKeyPair(keystore, config, config.getKeyAlias());
    }

    private static final int countKeyPairs(KeyStore keystore) throws CryptoException {
        int count = 0;
        try {
            for (String alias : Util.iterable(keystore.aliases())) {
                if (!keystore.isKeyEntry(alias)) continue;
                ++count;
            }
        }
        catch (Exception e) {
            throw new CryptoException(e);
        }
        return count;
    }

    public KeyPair getKeyPair(String alias) throws CryptoException {
        return PKITool.getKeyPair(this.keystore, this.config, alias);
    }

    private static final KeyPair getKeyPair(KeyStore keystore, KeyStoreConfig config, String alias) throws CryptoException {
        try {
            X509Certificate cert = (X509Certificate)keystore.getCertificate(alias);
            CertData myX509 = PKITool.getX509Data(cert);
            PrivateKey myPrivateKey = (PrivateKey)keystore.getKey(alias, config.getPasswordAsCharArray());
            PublicKey myPublicKey = cert.getPublicKey();
            return new KeyPair(myX509, myPrivateKey, myPublicKey);
        }
        catch (CryptoException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CryptoException(e);
        }
    }

    public KeyStoreConfig getConfig() {
        return this.config;
    }

    public PrivateKey getMyPrivateKey() {
        return this.myPrivateKey;
    }

    public PublicKey getMyPublicKey() {
        return this.myPublicKey;
    }

    public CertData getMyX509() {
        return this.myX509;
    }

    public CertID getMyCertID() {
        return this.myX509.getCertID();
    }

    public static CertData getX509Data(X509Certificate cert) throws CryptoException {
        try {
            return new CertData(new CertID(cert), cert.getEncoded());
        }
        catch (CertificateEncodingException e) {
            throw new CryptoException("Could not read encoded bytes for x509 cert", e);
        }
    }

    public Set<BigInteger> getSerialNumbersCopy() {
        return Util.asSet(this.x509KeyInfoMap.keySet());
    }

    public Set<CertData> getImportedCertsCopy() {
        return Util.asSet(this.x509KeyInfoMap.values());
    }

    public boolean containsX509Certificate(CertID serial) {
        return this.containsX509Certificate(serial.getSerial());
    }

    public boolean containsX509Certificate(BigInteger serial) {
        return this.x509KeyInfoMap.containsKey(serial);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addX509Certificate(X509Certificate cert) throws CryptoException {
        if (cert == null) {
            throw new CryptoException("Null certificate passed in");
        }
        Map<BigInteger, CertData> map = this.x509KeyInfoMap;
        synchronized (map) {
            CertData x509 = PKITool.getX509Data(cert);
            this.x509KeyInfoMap.put(x509.getCertID().getSerial(), x509);
        }
    }

    public void addX509Certificate(byte[] encodedInput) throws CryptoException {
        this.addX509Certificate(this.getX509Certificate(encodedInput));
    }

    public void addX509Certificate(InputStream encodedInput) throws CryptoException {
        if (encodedInput == null) {
            throw new CryptoException("Null InputStream passed in");
        }
        try {
            this.addX509Certificate((X509Certificate)this.certFactory.generateCertificate(encodedInput));
        }
        catch (Exception e) {
            throw new CryptoException("Error decoding certificate.  Is it an X.509 cert in the DER encoding?", e);
        }
    }

    public X509Certificate getX509Certificate(CertData x509) throws CryptoException {
        Util.guardNotNull(x509);
        if (x509.getX509Certificate() != null) {
            return this.getX509Certificate(x509.getX509Certificate());
        }
        return this.getX509Certificate(x509.getCertID());
    }

    public X509Certificate getX509Certificate(CertID x509Serial) throws CryptoException {
        Util.guardNotNull(x509Serial);
        return this.getX509Certificate(x509Serial.getSerial());
    }

    public X509Certificate getX509Certificate(BigInteger serial) throws CryptoException {
        Util.guardNotNull(serial);
        if (this.containsX509Certificate(serial)) {
            return this.getX509Certificate(this.x509KeyInfoMap.get(serial).getX509Certificate());
        }
        return null;
    }

    public X509Certificate getX509Certificate(byte[] encoded) throws CryptoException {
        Util.guardNotNull(encoded);
        try {
            return (X509Certificate)this.certFactory.generateCertificate(new ByteArrayInputStream(encoded));
        }
        catch (CertificateException e) {
            throw new CryptoException("Could not get x509 cert from encoded bytes[]", e);
        }
    }

    public List<X509Certificate> getAllCACerts() {
        return Util.makeArrayList(this.caCertificates.values());
    }

    public X509Certificate removeX509Certificate(CertData x509) throws CryptoException {
        Util.guardNotNull(x509);
        return this.removeX509Certificate(x509.getCertID());
    }

    public X509Certificate removeX509Certificate(CertID x509serial) throws CryptoException {
        Util.guardNotNull(x509serial);
        return this.removeX509Certificate(x509serial.getSerial());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public X509Certificate removeX509Certificate(BigInteger x509serial) throws CryptoException {
        CertData removed;
        if (x509serial == null) {
            throw new CryptoException("Null x509Serial passed in.");
        }
        Map<BigInteger, CertData> map = this.x509KeyInfoMap;
        synchronized (map) {
            removed = this.x509KeyInfoMap.remove(x509serial);
        }
        if (removed != null) {
            return this.getX509Certificate(removed.getX509Certificate());
        }
        return null;
    }

    public PublicKey getPublicKey(BigInteger x509Serial) throws CryptoException {
        if (!this.containsX509Certificate(x509Serial)) {
            throw new CryptoException("No entry found for x509Serial " + x509Serial);
        }
        return this.getX509Certificate(this.x509KeyInfoMap.get(x509Serial)).getPublicKey();
    }

    private static final Cipher getPublicKeyCipher() throws NoSuchAlgorithmException, NoSuchPaddingException {
        return Cipher.getInstance(publicKeyAlgorithm);
    }

    private static final Signature getSignerVerifier() throws NoSuchAlgorithmException {
        return Signature.getInstance(signatureAlgorithm);
    }

    public byte[] sign(String challenge) throws SignatureException {
        return this.sign(challenge.getBytes());
    }

    public byte[] sign(byte[] challenge) throws SignatureException {
        try {
            Signature signerVerifier = PKITool.getSignerVerifier();
            signerVerifier.initSign(this.myPrivateKey);
            signerVerifier.update(challenge);
            return signerVerifier.sign();
        }
        catch (Exception e) {
            throw new SignatureException("Invalid key, failed to sign challenge " + new String(challenge), e);
        }
    }

    public boolean verifySignature(String challenge, byte[] signature) throws SignatureException {
        return this.verifySignature(challenge.getBytes(), signature);
    }

    public boolean verifySignature(String challenge, byte[] signature, CertData signerKeyInfo) throws SignatureException {
        return this.verifySignature(challenge, signature, signerKeyInfo.getCertID());
    }

    public boolean verifySignature(String challenge, byte[] signature, CertID x509Serial) throws SignatureException {
        return this.verifySignature(challenge, signature, x509Serial.getSerial());
    }

    public boolean verifySignature(String challenge, byte[] signature, BigInteger signerSerial) throws SignatureException {
        return this.verifySignature(challenge.getBytes(), signature, signerSerial);
    }

    public boolean verifySignature(byte[] challenge, byte[] signature) throws SignatureException {
        return this.verifySignature(challenge, signature, this.myX509.getCertID().getSerial());
    }

    public boolean verifySignature(byte[] challenge, byte[] signature, CertData signerX509) throws SignatureException {
        return this.verifySignature(challenge, signature, signerX509.getCertID().getSerial());
    }

    public boolean verifySignature(byte[] challenge, byte[] signature, CertID signedBy) throws SignatureException {
        return this.verifySignature(challenge, signature, signedBy.getSerial());
    }

    public boolean verifySignature(byte[] challenge, byte[] signature, BigInteger signerSerial) throws SignatureException {
        if (challenge == null || signature == null || signerSerial == null) {
            throw new SignatureException("Invalid input:  Verifying signature, signer: " + signerSerial + " challenge: '" + challenge + "' signature: '" + signature + "'");
        }
        try {
            if (DEBUG) {
                log.debug((Object)("Verifying signature, signer: " + signerSerial + " challenge: '" + challenge + "' signature: '" + signature + "'"));
            }
            if (!this.containsX509Certificate(signerSerial)) {
                log.error((Object)("No certificate found for signer key ID " + signerSerial));
                return false;
            }
            X509Certificate signerCert = this.getX509Certificate(signerSerial);
            if (DEBUG) {
                log.debug((Object)("Signer keyID: " + signerSerial));
                log.debug((Object)("Signer cert: " + signerCert.toString()));
            }
            Signature signerVerifier = PKITool.getSignerVerifier();
            signerVerifier.initVerify(signerCert);
            signerVerifier.update(challenge);
            boolean verified = signerVerifier.verify(signature);
            if (DEBUG) {
                log.debug((Object)("signature verified: " + verified));
            }
            return verified;
        }
        catch (CryptoException e) {
            throw new SignatureException("Failed to retrieve key for signer", e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new SignatureException("Bad signature algorithm, failed to verify challenge " + challenge, e);
        }
        catch (InvalidKeyException e) {
            throw new SignatureException("Invalid key, failed to verify challenge " + challenge, e);
        }
    }

    public byte[] encrypt(byte[] plainText, CertData x509) throws CryptoException {
        X509Certificate cert = this.getX509Certificate(x509.getX509Certificate());
        return this.encrypt(plainText, cert.getPublicKey());
    }

    public byte[] encrypt(byte[] plainText, CertID x509Serial) throws CryptoException {
        return this.encrypt(plainText, x509Serial.getSerial());
    }

    public byte[] encrypt(byte[] plainText, BigInteger x509Serial) throws CryptoException {
        try {
            return this.encrypt(plainText, this.getPublicKey(x509Serial));
        }
        catch (CryptoException e) {
            throw new CryptoException("Error encrypting using the provided key ", e);
        }
    }

    public byte[] encrypt(byte[] inpBytes, PublicKey key) throws CryptoException {
        try {
            Cipher myCipher = Cipher.getInstance(publicKeyAlgorithm);
            myCipher.init(1, key);
            return myCipher.doFinal(inpBytes);
        }
        catch (Exception e) {
            throw new CryptoException("Error encrypting: ", e);
        }
    }

    public byte[] decrypt(byte[] cipherText) throws CryptoException {
        try {
            Cipher myCipher = PKITool.getPublicKeyCipher();
            myCipher.init(2, this.myPrivateKey);
            return myCipher.doFinal(cipherText);
        }
        catch (Exception e) {
            throw new CryptoException("Error decrypting: ", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static final String getString(byte[] bytes) {
        Base64 base64 = base64Codec;
        synchronized (base64) {
            return base64Codec.encodeToString(bytes);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static final byte[] getBytes(String input) {
        Base64 base64 = base64Codec;
        synchronized (base64) {
            return base64Codec.decode(input);
        }
    }

    public final PKCryptor cryptor() {
        return new PKCryptor(this);
    }

    public final void debug() throws KeyStoreException {
        if (INFO) {
            log.info((Object)("Keystore Type: " + this.keystore.getType()));
            log.info((Object)"Public-key crypto Algorithm: RSA");
            log.info((Object)"Signature Algorithm: SHA1withRSA");
            log.info((Object)("Config: " + this.config));
            log.info((Object)"Key Aliases:");
            Enumeration<String> aliases = this.keystore.aliases();
            while (aliases.hasMoreElements()) {
                log.info((Object)("\t" + aliases.nextElement()));
            }
            log.info((Object)"Key IDs:");
            for (CertData x509Serial : this.x509KeyInfoMap.values()) {
                log.info((Object)x509Serial.getCertID());
            }
            if (INFO) {
                log.info((Object)("Private Key ID: " + this.myX509.getCertID()));
            }
        }
    }

    public X509Certificate getCaCertificate(Principal p) {
        return this.caCertificates.get(p);
    }

    @Deprecated
    public static final class Workarounds {
        private static volatile boolean systemPropertiesHaveBeenSet = false;

        private Workarounds() {
        }

        @Deprecated
        public static final synchronized void setSystemProperties() throws ConfigException {
            Workarounds.setSystemProperties(ConfigTool.loadKeyStoreConfig());
        }

        @Deprecated
        public static final synchronized void setSystemProperties(KeyStoreConfig config) {
            if (!systemPropertiesHaveBeenSet) {
                log.warn((Object)"WARNING: Setting System properties; Spin is taking over the JVM-wide keystore. :(");
                try {
                    System.setProperty("javax.net.ssl.trustStore", Workarounds.getKeyStoreFileName(config));
                    if (DEBUG) {
                        log.debug((Object)("Set javax.net.ssl.trustStore to: " + System.getProperty("javax.net.ssl.trustStore")));
                    }
                    System.setProperty("javax.net.ssl.trustStorePassword", config.getPassword());
                    System.setProperty("java.protocol.handler.pkgs", "javax.net.ssl");
                    Security.addProvider((java.security.Provider)new Provider());
                }
                catch (Exception e) {
                    throw new RuntimeException("Error setting up keystore for SSL: ", e);
                }
                systemPropertiesHaveBeenSet = true;
            } else {
                log.warn((Object)"Are you trying to reset system properties?");
            }
        }

        private static final String getKeyStoreFileName(KeyStoreConfig keyStoreConfig) throws ConfigException {
            try {
                return PKITool.getKeyStoreURI(keyStoreConfig).toURL().getFile();
            }
            catch (MalformedURLException e) {
                throw new ConfigException("Couldn't determine Keystore filename", e);
            }
        }
    }

    public static final class KeyPair {
        public final CertData x509Data;
        public final PrivateKey privateKey;
        public final PublicKey publicKey;

        public KeyPair(CertData myX509, PrivateKey myPrivateKey, PublicKey myPublicKey) {
            this.x509Data = myX509;
            this.privateKey = myPrivateKey;
            this.publicKey = myPublicKey;
        }
    }
}

