package org.eaglei.suggest.provider.model.lucene;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eaglei.common.lexical.model.EIModelEntityExtractorFactory;
import org.eaglei.common.lexical.model.ModelEntitySource;
import org.eaglei.model.EagleIOntConstants;
import org.eaglei.suggest.provider.SuggestionProvider;
import org.eaglei.suggest.provider.SuggestionSource;

import com.hp.hpl.jena.vocabulary.RDFS;

/**
 * Factory class for ModelSuggestionProviders that use ModelSuggestionSources created
 * from the eagle-i ontology.

 * @author rfrost
 */
public final class EIModelSuggestionProviderFactory {

	// properties to index
	// TODO embed this in the JenaEIOntModel class
	private static final List<String> props = new ArrayList<String>();
	static {
		props.add(RDFS.label.getURI());
		props.add(EagleIOntConstants.IAO_PREFERRED_TERM_URI);        
		props.add(EagleIOntConstants.IAO_ALTERNATE_TERM_URI);
	}
	
	/**
	 * Creates one ModelSuggestionProvider per top-level resource class in the eagle-i ontology. 
	 * @return Map from resource class URI to SuggestionProvider. 
	 * @throws IOException Thrown if there is an error creating and indexing the suggestion providers
	 */
	public static Map<String, SuggestionProvider> createProviderForEachTopLevel() throws IOException {
		final Map<String, SuggestionProvider> idToProvider = new HashMap<String, SuggestionProvider>();
		final Map<String, ModelEntitySource> uriToSource = EIModelEntityExtractorFactory.createSourcesForEachTopLevel();
		// create one provider per top-level class and one for all
		for (String uri: uriToSource.keySet()) {
			// create a SuggestionProvider for each top-level class
			final SuggestionSource source = new ModelSuggestionSource(uriToSource.get(uri));
			final SuggestionProvider provider = new ModelSuggestionProvider();
			provider.index(source);
			idToProvider.put(uri, provider);
		}
		return idToProvider;
	}

	/**
	 * Creates one ModelSuggestionProvider for all top-level resource classes and non-resource roots in the eagle-i ontology. 
	 * @return Map from resource class URI to SuggestionProvider. 
	 * @throws IOException Thrown if there is an error creating and indexing the suggestion providers
	 */
	public static SuggestionProvider createProviderForAllRoots() throws IOException {
		final SuggestionSource source = new ModelSuggestionSource(EIModelEntityExtractorFactory.createSourceForAllRoots());
		final SuggestionProvider provider = new ModelSuggestionProvider();
		provider.index(source);
		return provider;
	}

}
