/**
 * eagle-i Project
 * Harvard University
 * Apr 23, 2010
 */
package org.eaglei.datatools.jena;

import static org.eaglei.datatools.jena.MetadataConstants.readOnlyLiterals;
import static org.eaglei.datatools.jena.MetadataConstants.readOnlyResources;
import static org.eaglei.datatools.jena.SPARQLConstants.DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_NAME_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.OWNER_NAME_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.OWNER_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.SUBJECT_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.TYPE_VARIABLE;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIDatatypeProperty;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.SimpleSelector;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.vocabulary.RDF;
import com.hp.hpl.jena.vocabulary.RDFS;

/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 */

public class EIInstanceFactory {
	private static final Log logger = LogFactory.getLog(EIInstanceFactory.class);

	private static final boolean isDebugEnabled = logger.isDebugEnabled();

	public static final EIInstanceFactory INSTANCE = new EIInstanceFactory();

	public static EIInstanceFactory getInstance() {
		return INSTANCE;
	}
	
	private static EIOntModel ontModel;
	
	private EIInstanceFactory() {
		// create and configure spring beans
		final ApplicationContext context = new ClassPathXmlApplicationContext(new String[] { "datatools-model-config.xml" });
		// retrieve configured instance
		ontModel = context.getBean(EIOntModel.class);
	}

	

	/**
	 * Serialize an EInstance into an RDF language
	 * 
	 * @param instance
	 * @param lang
	 *            - one of the RDF languages supported by Jena; predefined values are "RDF/XML", "RDF/XML-ABBREV", "N-TRIPLE", "TURTLE", (and "TTL") and "N3". The default value, represented by null is "RDF/XML".
	 * @return - the serialized EIInstance as a STring
	 */

	public String serialize(final EIInstance instance, final String lang) {
		if (instance == null) {
			return null;
		}

		final Model model = convertToJenaModel(instance);
		// Serialize model to String in specified format
		final StringWriter sw = new StringWriter();
		try {
			// TODO validate format
			model.write(sw, lang);
			final String s = sw.toString();
			sw.flush();
			return s;
		} finally {
			try {
				if (sw != null) {
					sw.close();
				}
			} catch (final IOException e) {/* not much to do */
			}
		}

	}

	public Model convertToJenaModel(final EIInstance instance) {
		if (instance == null) {
			return null;
		}
		final Model model = ModelFactory.createDefaultModel();
		// Create the resource that will be the subject of all statements
		final Resource resourceInstance = model.createResource(instance.getInstanceURI().toString());

		// Set its type and label
		final Resource eiType = model.createResource(instance.getInstanceType().getURI().toString());
		model.add(model.createStatement(resourceInstance, RDF.type, eiType));
		model.add(model.createStatement(resourceInstance, RDFS.label, instance.getInstanceLabel()));

		// Set other types
		final List<EIEntity> otherEITypes = instance.getOtherEITypes();
		if (otherEITypes != null) {
			for (final EIEntity t : otherEITypes) {
				final Resource typeResource = model.createResource(t.getURI().toString());
				model.add(model.createStatement(resourceInstance, RDF.type, typeResource));
			}
		}
		// Set all other properties
		final Map<EIEntity, Set<String>> dataProps = instance.getDatatypeProperties();
		final Map<EIEntity, Set<EIEntity>> objectProps = instance.getObjectProperties();
		final Map<EIEntity, Set<String>> literalProps = instance.getNonOntologyLiteralProperties();
		final Map<EIEntity, Set<EIEntity>> resourceProps = instance.getNonOntologyResourceProperties();
		final Map<EIEntity, EIEntity> roResourceProps = instance.getReadOnlyResourceProperties();
		final Map<EIEntity, String> roLiteralProps = instance.getReadOnlyLiteralProperties();

		addLiteralTriples(model, resourceInstance, dataProps);

		addResourceTriples(model, resourceInstance, objectProps);

		addLiteralTriples(model, resourceInstance, literalProps);

		addResourceTriples(model, resourceInstance, resourceProps);

		for (final Map.Entry<EIEntity, String> entry : roLiteralProps.entrySet()) {
			final Property p = model.createProperty(entry.getKey().getURI().toString());
			if (entry.getValue() != null && entry.getValue().length() > 0) {
				model.add(model.createStatement(resourceInstance, p, entry.getValue()));
			}
		}

		for (final Map.Entry<EIEntity, EIEntity> entry : roResourceProps.entrySet()) {
			final Property p = model.createProperty(entry.getKey().getURI().toString());
			if (entry.getValue() != null && entry.getValue().toString().length() > 0) {
				final Resource value = model.createResource(entry.getValue().getURI().toString());
				model.add(model.createStatement(resourceInstance, p, value));
			}
		}
		return model;
	}

	/**
	 * @param model
	 * @param resourceInstance
	 * @param resources
	 */
	private void addResourceTriples(final Model model, final Resource resourceInstance, final Map<EIEntity, Set<EIEntity>> resources) {
		for (final Map.Entry<EIEntity, Set<EIEntity>> entry : resources.entrySet()) {
			final Property p = model.createProperty(entry.getKey().getURI().toString());
			final Set<EIEntity> values = entry.getValue();
			if (values != null) {
				for (final EIEntity value : values) {
					if (value != null && value.toString().length() > 0) {
						final Resource valueResource = model.createResource(value.getURI().toString());
						model.add(model.createStatement(resourceInstance, p, valueResource));
					}
				}
			}
		}
	}

	/**
	 * @param model
	 * @param resourceInstance
	 * @param literals
	 */
	private void addLiteralTriples(final Model model, final Resource resourceInstance, final Map<EIEntity, Set<String>> literals) {
		for (final Map.Entry<EIEntity, Set<String>> entry : literals.entrySet()) {
			final Property p = model.createProperty(entry.getKey().getURI().toString());
			final Set<String> values = entry.getValue();
			for (final String value : values) {
				if (value != null && value.length() > 0) {
					model.add(model.createStatement(resourceInstance, p, value));
				}
			}
		}
	}

	/**
	 * Create an EIInstance from an RDF String serialization in one of the RDF languages supported by Jena
	 * 
	 * @param instanceUri
	 *            - the instance uri (all the triples in the rdf parameter should have it as subject)
	 * @param rdf
	 * @param lang
	 * @return
	 */
	// TODO deal with cases where the RDF does not represent an EIInstance

	public EIInstance create(final EIURI instanceUri, final String rdf, final String lang) {
		if (instanceUri == null || instanceUri.toString().length() == 0 || rdf == null || rdf.length() == 0) {
			return null;
		}
		final Model model = ModelFactory.createDefaultModel();
		model.read(new StringReader(rdf), null, lang);
		return create(instanceUri, model);
	}

	/**
	 * Create an EIInstance with no properties other than an eiType (at least one type must be set)
	 * 
	 * @param typeUri
	 *            - type of the EInstance; if more than one, add can be used
	 * @param instanceEntity
	 *            - entity containing the instance Uri and its label
	 * @return null if the typeUri is not an EI Class
	 */
	public EIInstance createEmpty(final EIURI typeUri, final EIEntity instanceEntity) {
		if (instanceEntity == null) {
			return null;
		}
		final EIClass instanceClass = safeGetClass(typeUri);
		if (instanceClass == null) {
			return null;
		}
		final String label = instanceClass.getEntity().getLabel();
		// FIXME temporary fix to lack of capitalization
		//instanceClass.getEntity().setLabel(capitalizeFirst(label));
		instanceClass.getEntity().setLabel(label);
		final EIInstance ei = EIInstance.create(instanceClass.getEntity(), instanceEntity);
		// FIXME we want to go back to using EIClass in EIInstance but can't fix allover for now
		ei.setInstanceClass(instanceClass);
		return ei;
	}

	// TODO refactor these two
	public EIInstanceMinimal createEmptyMinimal(final EIURI typeUri, final EIEntity instanceEntity) {
		if (instanceEntity == null) {
			return null;
		}
		final EIClass instanceClass = safeGetClass(typeUri);
		if (instanceClass == null) {
			return null;
		}
		final String label = instanceClass.getEntity().getLabel();
		// FIXME temporary fix to lack of capitalization
		// instanceClass.getEntity().setLabel(capitalizeFirst(label));
		instanceClass.getEntity().setLabel(label);
		final EIInstanceMinimal ei = EIInstanceMinimal.create(instanceClass.getEntity(), instanceEntity);
		// FIXME we want to go back to using EIClass in EIInstance but can't fix allover for now
		// ei.setInstanceClass(instanceClass);
		return ei;
	}

	/**
	 * Create an EIInstance from a Jena model, with a given instanceUri. Label and type must be present in model
	 * 
	 * @param instanceUri
	 *            - the instance uri (all the triples in the model should have it as subject)
	 * @param model
	 * @return
	 */
	//FIXME uncomment null instance code when ready
	
	public EIInstance create(final EIURI instanceUri, final Model model) {	
		final Resource subject = getInstanceSubject(instanceUri, model);
		if(subject == null) {
			return null;
			//return EIInstance.NULL_INSTANCE;
		}
		final List<EIClass> eiClasses = getInstanceClasses(model, subject);
		if(eiClasses == null || eiClasses.isEmpty()) {
			return null;
			//return EIInstance.NULL_INSTANCE;
		}

		// FIXME for now set instance type to first in list;
		// The most common case is only one ei type; we can't really know what to put there otherwise
		// Ideally the most specific type but that's hard to know
		final EIClass instanceClass = eiClasses.get(0);
		if(instanceClass == null) {
			return null;
			//return EIInstance.NULL_INSTANCE;
		}
		eiClasses.remove(0);

		final String instanceClassUri = instanceClass.getEntity().getURI().toString();
		final String instanceLabel = getInstanceLabel(subject);
		if (instanceLabel == null) {
			return null;
			//return EIInstance.NULL_INSTANCE;
		}
		
		if (isDebugEnabled) {
			logger.debug("Creating an instance of class: " + instanceClass.getEntity().toString() + " with URI: " + instanceUri + " and label: " + instanceLabel);
		}

		// create IEInstance
		final EIInstance ei = EIInstance.create(instanceClass,  EIEntity.create(instanceUri, instanceLabel));
		setInstanceTypes(eiClasses, ei);
		
		// query the model for all statements about this subject
		final StmtIterator iterator = model.listStatements(subject, null, (RDFNode) null);
		if (isDebugEnabled) {
			logger.debug("Searching for statements with subject: " + instanceUri.toString() + " of class: " + instanceClass.getEntity() + " and label: " + instanceLabel);
		}

		// Iterate to add properties in the different maps
		while (iterator.hasNext()) {
			final Statement statement = iterator.nextStatement();
			final Property predicate = statement.getPredicate();
			// No need to process label, as it was done previously
			if (predicate.equals(RDFS.label)) {
				continue;
			}
			final EIURI propertyUri = EIURI.create(predicate.getURI());
			final RDFNode o = statement.getObject();

			// No need to process type if it is instanceClass
			if (predicate.equals(RDF.type) && o.isResource() && instanceClassUri.equals(((Resource) o).getURI())) {
				continue;
			}

			// No need to process types already in eiClasses; other types will end up in resource map
			if (predicate.equals(RDF.type) && o.isResource() && containsUri(eiClasses, ((Resource) o).getURI())) {
				continue;
			}

			// Check if property is in ontology and add accordingly
			final EIProperty p = getEIOntProperty(instanceClass.getEntity().getURI(), propertyUri);
			// Get the label from model or from ontology if object property
			String objectLabel = null;
			if (o.isResource()) {
				objectLabel = getObjectLabel(o);
			}
			
			if (p != null) {
				addEIProperty(ei, p, propertyUri, o, objectLabel);
			} else {
				addNonEIProperty(ei, predicate, propertyUri, o, objectLabel);
			}
		}
		return ei;
	}

	private String getObjectLabel(final RDFNode o) {
		String objectLabel;
		final EIClass classForObject = safeGetClass(EIURI.create(((Resource) o).getURI()));
		if (classForObject != null) {
			// TODO reuse this entity, duh
			objectLabel = classForObject.getEntity().getLabel();
		} else { // object is an eagle-i resource {
			final Statement objectLabelStatement = ((Resource) o).getProperty(RDFS.label);
			if (objectLabelStatement != null) {
				objectLabel = objectLabelStatement.getLiteral().toString();
			} else {
				objectLabel = ((Resource) o).getURI().toString();
			}
		}
		return objectLabel;
	}
	
	

	private void setInstanceTypes(final List<EIClass> eiClasses, final EIInstance ei) {
		// If there were more than one ei types found previously
		if (!eiClasses.isEmpty()) {
			final List<EIEntity> eiTypes = new ArrayList<EIEntity>(eiClasses.size());
			for (final EIClass c : eiClasses) {
				eiTypes.add(c.getEntity());
			}
			ei.setOtherEITypes(eiTypes);
		}
	}

	private String getInstanceLabel(final Resource subject) {
		final Statement labelStatement = subject.getProperty(RDFS.label);
		if (labelStatement == null) {
			if (isDebugEnabled) {
				logger.debug("RDFS Label is not set for instance");
			}
			return null;
			//return EIInstance.NULL_INSTANCE;
		}
		final String instanceLabel = labelStatement.getString();
		return instanceLabel;
	}

	private List<EIClass> getInstanceClasses(final Model model, final Resource subject) {
		// Instances can have multiple types
		final List<EIClass> eiClasses = getEIClasses(model, subject);

		// Nothing in the received model matches an EIClass
		if (eiClasses == null) {
			if (isDebugEnabled) {
				logger.debug("No eagle-i types found in the model");
			}
			return null;
		}
		return eiClasses;
	}

	private Resource getInstanceSubject(final EIURI instanceUri, final Model model) {
		if (instanceUri == null || model == null) {
			if (isDebugEnabled) {
				logger.debug("null parameters; no EIInstance was created");
			}
			return null;
		}
		final Resource subject = model.getResource(instanceUri.toString());
		// Get resource will create a resource if one doens't exist already, so need to check
		if (!model.contains(subject, null, (RDFNode) null)) {
			if (isDebugEnabled) {
				logger.debug("model doesn't contain subject; no EIInstance was created");
			}
			return null;
		}
		return subject;
	}
	
	

	/**
	 * Batch creation of a list of EIInstances from a model
	 * 
	 * @param model
	 * @return
	 */
	public List<EIInstance> create(final Model model) {
		if (model == null) {
			return Collections.EMPTY_LIST;
		}
		final List<EIInstance> instances = new ArrayList<EIInstance>();
		final Set<Resource> subjects = model.listSubjects().toSet();
		// create an EIInstance per subject
		for (final Resource r : subjects) {
			final Model results = model.query(new SimpleSelector(r, null, (RDFNode) null));
			final EIInstance ei = create(EIURI.create(r.getURI()), results);
			if (ei != null) {
				instances.add(ei);
			}
		}
		return instances;
	}

	// TODO replace with SearchResult
	/**
	 * Create method for abridged EIInstances (label, type, owner, WFState, creation date)
	 * 
	 * @param resultSet
	 * @return
	 */

	// TODO the query could contain labels of owner and state as well
	public List<EIInstanceMinimal> create(final ResultSet resultSet) {
		if (resultSet == null) {
			return Collections.EMPTY_LIST;
		}
		final Map<EIURI, EIInstanceMinimal> instances = new HashMap<EIURI, EIInstanceMinimal>();
		// Need to preserve ordering to avoid a sort operation; this list will contain live elements
		final List<EIInstanceMinimal> instanceList = new ArrayList<EIInstanceMinimal>();
		while (resultSet.hasNext()) {
			final QuerySolution solution = resultSet.next();
			// Only type and label are mandatory
			if (solution.contains(SUBJECT_VARIABLE) && solution.contains(TYPE_VARIABLE) && solution.contains(LABEL_VARIABLE)) {
				final EIURI instanceUri = getUriFromSolution(solution, SUBJECT_VARIABLE);
				final EIURI instanceType = getUriFromSolution(solution, TYPE_VARIABLE);
				final String label = getStringFromSolution(solution, LABEL_VARIABLE);
				// type is not an EIClass ; we can skip solution
				if (safeGetClass(instanceType) == null) {
					continue;
				}

				EIInstanceMinimal ei;
				if (!instances.containsKey(instanceUri)) {
					ei = createEmptyMinimal(instanceType, EIEntity.create(instanceUri, label));
					instances.put(instanceUri, ei);
					instanceList.add(ei);
					if (isDebugEnabled) {
						logger.debug("Creating new EIInstance: " + ei);
					}

				} else {
					ei = instances.get(instanceUri);
					ei.addEIType(EIEntity.create(instanceType, ""));
					if (isDebugEnabled) {
						logger.debug("Adding type: " + instanceType + " to existing EIInstance :" + ei);
					}
				}
				// Continue with setting optional parts
				ei.setWFOwner(getEntityFromSolution(solution, OWNER_VARIABLE, OWNER_NAME_VARIABLE));
				ei.setWFState(getEntityFromSolution(solution, STATE_VARIABLE, ""));
				ei.setCreationDate(getStringFromSolution(solution, DATE_VARIABLE));
				ei.setLab(getEntityFromSolution(solution, LAB_VARIABLE, LAB_NAME_VARIABLE));
			} else {
				continue;
			}
		}
		// This adds way too much time
		// Collections.sort(instanceList);
		return instanceList;
	}

	private EIEntity getEntityFromSolution(final QuerySolution solution, final String uriVariable, final String labelVariable) {
		if (solution.contains(uriVariable)) {
			final EIURI uri = EIURI.create(solution.getResource(uriVariable).getURI());
			if (STATE_VARIABLE.equals(uriVariable)) {
				return DataToolsOntConstants.getStatusEntity(uri);
			}
			String label = "<none>";
			if (solution.contains(labelVariable)) {
				label = solution.getLiteral(labelVariable).getString();
			}
			return EIEntity.create(uri, label);
		} else {
			return EIEntity.NULL_ENTITY;
		}
	}

	private EIURI getUriFromSolution(final QuerySolution solution, final String variable) {
		if (solution.contains(variable)) {
			return EIURI.create(solution.getResource(variable).getURI());
		} else {
			return EIURI.NULL_EIURI;
		}
	}

	private String getStringFromSolution(final QuerySolution solution, final String variable) {
		if (solution.contains(variable)) {
			return solution.getLiteral(variable).getString();
		} else {
			return "";
		}
	}

	/**
	 * Find a property for a class in the eagle-i ontology.
	 * 
	 * @param instanceClassUri
	 * @param propertyUri
	 * @return the property if found, null otherwise
	 */
	private EIProperty getEIOntProperty(final EIURI instanceClassUri, final EIURI propertyUri) {
		if (instanceClassUri == null || propertyUri == null) {
			return null;
		}

		final List<EIProperty> properties = ontModel.getProperties(instanceClassUri);
		if (properties == null || properties.isEmpty()) {
			return null;
		}
		for (final EIProperty p : properties) {
			final EIURI propUri = p.getEntity().getURI();
			if (propUri.equals(propertyUri)) {
				return p;
			}
		}
		// If property wasn't found in domain
		return null;
	}

	/**
	 * 
	 * @param model
	 * @param subject
	 * @return a list of EIEntities if the model has EITypes null if no EITypes were found in model
	 */

	private List<EIClass> getEIClasses(final Model model, final Resource subject) {
		final List<Statement> typeStatements = model.listStatements(subject, RDF.type, (Resource) null).toList();
		if (typeStatements == null || typeStatements.isEmpty()) {
			return null;
		}
		final List<EIClass> eiClasses = new ArrayList<EIClass>();
		for (final Statement st : typeStatements) {
			final Resource type = (Resource) st.getObject();
			final EIURI typeUri = EIURI.create(type.getURI());
			// logger.debug("Trying to match URI to EICLass: " + typeUri);
			final EIClass eiClass = safeGetClass(typeUri);
			if (eiClass != null) {
				eiClasses.add(eiClass);
			}
		}
		return eiClasses.isEmpty() ? null : eiClasses;
	}

	private void addEIProperty(final EIInstance ei, final EIProperty predicate, final EIURI propertyUri, final RDFNode o, final String label) {

		final String preferredLabel = ontModel.getPreferredLabel(propertyUri);
		//final EIEntity propEntity = EIEntity.create(propertyUri, capitalizeFirst(preferredLabel));
		final EIEntity propEntity = EIEntity.create(propertyUri, preferredLabel);
		if (predicate instanceof EIDatatypeProperty) {
			if (o.isLiteral()) {
				final String value = o.toString();
				if (isDebugEnabled) {
					logger.debug("adding property: [" + propertyUri + "] with literal value : [" + value + "] ");
				}
				ei.addDatattypeProperty(propEntity, value);
			} else {
				final EIEntity entity = EIEntity.create(EIURI.create(((Resource) o).getURI()), label);
				if (isDebugEnabled) {
					logger.debug("property: [" + propertyUri + "] found in ontology but value is not literal; adding as non-ontology property ");
				}
				ei.addNonOntologyResourceProperty(propEntity, entity);
			}
		} else if (predicate instanceof EIObjectProperty) {
			if (o.isResource()) {
				final EIEntity entity = EIEntity.create(EIURI.create(((Resource) o).getURI()), label);
				if (isDebugEnabled) {
					logger.debug("adding property: [" + propertyUri + "] with object value : [" + entity + "] ");
				}
				ei.addObjectProperty(propEntity, entity);
			} else {
				if (isDebugEnabled) {
					logger.debug("property: [" + propertyUri + "] found in ontology but value is not resource; adding as non-ontology property ");
				}
				ei.addNonOntologyLiteralProperty(propEntity, o.toString());
			}
		}
	}

	private void addNonEIProperty(final EIInstance ei, final Property predicate, final EIURI propertyUri, final RDFNode o, final String label) {
		if (isDebugEnabled) {
			logger.debug("Property not found in ontology; we'll still add it: " + propertyUri);
		}
		EIEntity propEntity;
		if (readOnlyResources.containsKey(propertyUri)) {
			propEntity = readOnlyResources.get(propertyUri);
		} else if (readOnlyLiterals.containsKey(propertyUri)) {
			propEntity = readOnlyLiterals.get(propertyUri);
		} else {
			// FIXME
			final String predicateLabel = predicate.getLocalName();
			propEntity = EIEntity.create(propertyUri, predicateLabel);
		}
		if (o.isLiteral()) {
			if (readOnlyLiterals.containsKey(propertyUri)) {
				ei.setReadOnlyLiteralProperty(propEntity, o.toString());
			} else {
				ei.addNonOntologyLiteralProperty(propEntity, o.toString());
			}
		} else if (o.isResource()) {
			final EIEntity entity = EIEntity.create(EIURI.create(((Resource) o).getURI()), label);
			if (readOnlyResources.containsKey(propertyUri)) {
				ei.setReadOnlyResourceProperty(propEntity, entity);
			} else {
				ei.addNonOntologyResourceProperty(propEntity, entity);
			}
		} else {
			if (isDebugEnabled) {
				logger.debug("Could not add property " + propertyUri);
			}
		}
	}

	private boolean containsUri(final List<EIClass> eiClasses, final String uri) {
		if (eiClasses == null || uri == null) {
			return false;
		}
		for (final EIClass eiClass : eiClasses) {
			if (uri.equalsIgnoreCase(eiClass.getEntity().getURI().toString())) {
				return true;
			}
		}
		return false;
	}

	private String capitalizeFirst(final String s) {
		final char first = s.charAt(0);
		return Character.toUpperCase(first) + s.substring(1);
	}

	private EIClass safeGetClass(final EIURI typeUri) {
		if (ontModel.isModelClassURI(typeUri.toString())) {
			return ontModel.getClass(typeUri);
		} else {
			return null;
		}
	}
}
