/**
 * The eagle-i consortium
 * Harvard University
 * Nov 3, 2010
 */
package org.eaglei.datatools.jena;

import static org.eaglei.datatools.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_NAME_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.OWNER_NAME_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.OWNER_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.SUBJECT_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.TYPE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.CREATION_DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.MODIFIED_DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.IS_STUB_VARIABLE;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;

/**
 * @author Daniela Bourges-Waldegg
 * 
 */
public class EIInstanceMinimalFactory {

	private static EIOntModel ontModel;

	private static final Log logger = LogFactory.getLog( EIInstanceMinimalFactory.class );

	private static final boolean isDebugEnabled = logger.isDebugEnabled();

	private static final EIInstanceMinimalFactory INSTANCE = new EIInstanceMinimalFactory();

	public static EIInstanceMinimalFactory getInstance() {
		return INSTANCE;
	}

	private EIInstanceMinimalFactory() {
		// create and configure spring beans
		final ApplicationContext context = new ClassPathXmlApplicationContext( new String[] { "datatools-model-config.xml" } );
		// retrieve configured instance
		ontModel = context.getBean( EIOntModel.class );
	}

	public EIInstanceMinimal createEmptyMinimal(final EIURI typeUri, final EIEntity instanceEntity) {
		if ( instanceEntity == null ) {
			return null;
		}
		final EIClass instanceClass = safeGetClass( typeUri );
		if ( instanceClass == null ) {
			return null;
		}
		final String label = instanceClass.getEntity().getLabel();
		instanceClass.getEntity().setLabel( label );
		final EIInstanceMinimal ei = EIInstanceMinimal.create( instanceClass.getEntity(), instanceEntity );
		return ei;
	}

	// TODO replace with SearchResult
	/**
	 * Create method for abridged EIInstances (label, type, owner, WFState, creation date)
	 * 
	 * @param resultSet
	 * @return
	 */

	// TODO the query could contain labels of owner and state as well
	public List<EIInstanceMinimal> create(final ResultSet resultSet) {
		if ( resultSet == null ) {
			return Collections.EMPTY_LIST;
		}
		final Map<EIURI, EIInstanceMinimal> instances = new HashMap<EIURI, EIInstanceMinimal>();
		// Need to preserve ordering to avoid a sort operation; this list will contain live elements
		final List<EIInstanceMinimal> instanceList = new ArrayList<EIInstanceMinimal>();
		while ( resultSet.hasNext() ) {
			final QuerySolution solution = resultSet.next();
			// Only type and label are mandatory
			if ( solution.contains( SUBJECT_VARIABLE ) && solution.contains( TYPE_VARIABLE ) && solution.contains( LABEL_VARIABLE ) ) {
				final EIURI instanceUri = getUriFromSolution( solution, SUBJECT_VARIABLE );
				final EIURI instanceType = getUriFromSolution( solution, TYPE_VARIABLE );
				final String label = getStringFromSolution( solution, LABEL_VARIABLE );
				// type is not an EIClass ; we can skip solution
				if ( safeGetClass( instanceType ) == null ) {
					if ( isDebugEnabled ) {
						logger.debug( "Query solution without EIClass: " + solution );
					}
					continue;
				}

				EIInstanceMinimal ei;
				if ( !instances.containsKey( instanceUri ) ) {
					ei = createEmptyMinimal( instanceType, EIEntity.create( instanceUri, label ) );
					instances.put( instanceUri, ei );
					instanceList.add( ei );

				} else {
					ei = instances.get( instanceUri );
					if ( isDebugEnabled ) {
						logger.debug( "Processing query solution for an already processed instance: " + ei );
					}
					ei.addEIType( EIEntity.create( instanceType, "" ) );
					// FIXME process additional labs here
				}
				// Continue with setting optional parts
				ei.setWFOwner( getEntityFromSolution( solution, OWNER_VARIABLE, OWNER_NAME_VARIABLE ) );
				ei.setWFState( getEntityFromSolution( solution, STATE_VARIABLE, "" ) );
				ei.setCreationDate( getStringFromSolution( solution, CREATION_DATE_VARIABLE ) );
				ei.setModifiedDate( getStringFromSolution( solution, MODIFIED_DATE_VARIABLE ) );
				ei.setLab( getEntityFromSolution( solution, LAB_VARIABLE, LAB_NAME_VARIABLE ) );
				ei.setIsStub( getBooleanFromSolution( solution, IS_STUB_VARIABLE ) );
			} else {
				if ( isDebugEnabled ) {
					logger.debug( "Query solution without type, label or subject: " + solution );
				}
				continue;
			}
		}
		// This adds way too much time
		// Collections.sort(instanceList);
		return instanceList;
	}

	private EIEntity getEntityFromSolution(final QuerySolution solution, final String uriVariable, final String labelVariable) {
		if ( solution.contains( uriVariable ) ) {
			final EIURI uri = EIURI.create( solution.getResource( uriVariable ).getURI() );
			if ( STATE_VARIABLE.equals( uriVariable ) ) {
				return DataToolsOntConstants.getStatusEntity( uri );
			}
			String label = "<none>";
			if ( solution.contains( labelVariable ) ) {
				label = solution.getLiteral( labelVariable ).getString();
			}
			return EIEntity.create( uri, label );
		} else {
			return EIEntity.NULL_ENTITY;
		}
	}

	private EIURI getUriFromSolution(final QuerySolution solution, final String variable) {
		if ( solution.contains( variable ) ) {
			return EIURI.create( solution.getResource( variable ).getURI() );
		} else {
			return EIURI.NULL_EIURI;
		}
	}

	private String getStringFromSolution(final QuerySolution solution, final String variable) {
		if ( solution.contains( variable ) ) {
			return solution.getLiteral( variable ).getString();
		} else {
			return "";
		}
	}

	private boolean getBooleanFromSolution(final QuerySolution solution, final String variable) {
		if ( solution.contains( variable ) ) {
			return solution.getLiteral( variable ).getString().equals( "True" );
			//|| solution.getLiteral( variable ).getBoolean(); // TODO: use this once type is fixed
		} else {
			return false;
		}
	}

	private EIClass safeGetClass(final EIURI typeUri) {
		if ( ontModel.isModelClassURI( typeUri.toString() ) ) {
			return ontModel.getClass( typeUri );
		} else {
			return null;
		}
	}

}
