/**
 * The eagle-i consortium
 * Harvard University
 * Nov 5, 2010
 */
package org.eaglei.datatools.jena;

import static org.eaglei.datatools.provider.RepositoryProviderMessages.NOT_AUTHORIZED_MESSAGE;
import static org.eaglei.datatools.provider.RepositoryProviderMessages.NO_SESSION_MESSAGE;
import static org.eaglei.datatools.provider.RepositoryProviderMessages.UNAVAILABLE_MESSAGE;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.provider.EIDataToolsProviderException;
import org.eaglei.datatools.provider.RepositoryProviderMessages;
import org.eaglei.security.Session;

/**
 * @author Daniela Bourges-Waldegg
 * 
 */
public class ProviderUtils {
	private static final Log logger = LogFactory.getLog( ProviderUtils.class );

	public static String getStringFromInputStream(final InputStream in) throws IOException {
		if ( in == null ) {
			return null;
		}
		final StringWriter writer = new StringWriter();
		// encoding needs to be explicitly set
		IOUtils.copy( in, writer, "UTF-8" );
		return writer.toString();
	}

	public static void validateSession(final Session session) throws EIDataToolsProviderException {
		if ( !Session.isValid( session ) ) {
			logger.error( "Invalid Session - request cannot be completed" );
			throw new EIDataToolsProviderException( NO_SESSION_MESSAGE );
		}
	}

	/**
	 * Convenience method for dealing with the most general case of http request/response expected status code is 200 and we can give a generic error message
	 * 
	 * @param client
	 * @param method
	 * @return the string returned by the repo
	 * @throws HttpException
	 * @throws IOException
	 * @throws EIDataToolsProviderException
	 */
	public static String getHttpResponse(final HttpClient client, final HttpMethod method) throws HttpException, IOException, EIDataToolsProviderException {
		int status = 0;
		String responseBody;
		try {
			status = client.executeMethod( method );
			responseBody = ProviderUtils.getStringFromInputStream( method.getResponseBodyAsStream() );
		} catch (final Exception e) {
			throw new EIDataToolsProviderException( RepositoryProviderMessages.getFailedMessage( "", status ) );
		} finally {
			method.releaseConnection();
		}

		if ( status == HttpStatus.SC_OK || status == HttpStatus.SC_CREATED ) {
			logger.info( "Request succeded with status: " + status);
			//+ " response: " + responseBody );
			return responseBody;
		}
		// Send the appropriate error message
		if ( status == HttpStatus.SC_NOT_FOUND ) {
			logger.error( "Repo unavailable" );
			throw new EIDataToolsProviderException( UNAVAILABLE_MESSAGE );
		} else if ( status == HttpStatus.SC_UNAUTHORIZED ) {
			logger.error( "Not authorized to get user information (login/whoami)" );
			throw new EIDataToolsProviderException( NOT_AUTHORIZED_MESSAGE );
		} else if ( status == HttpStatus.SC_CONFLICT ) {
			logger.error( "Stale token. Update instance failed with status: " + status + " response: " + responseBody );
			throw new EIDataToolsProviderException( RepositoryProviderMessages.STALE_TOKEN_MESSAGE );
		} else {
			logger.error( "Operation failed with status: " + status + " response: " + responseBody );
			throw new EIDataToolsProviderException( RepositoryProviderMessages.getFailedMessage( "", status ) );
		}
	}

}
