package org.eaglei.datatools.jena;


import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.commons.lang.CharEncoding;
import org.eaglei.datatools.SortByProperties;
import org.eaglei.datatools.provider.DatatoolsSecurityProvider;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIInstanceFactory;
import org.eaglei.search.provider.AuthSearchRequest;
import org.eaglei.security.Session;
import org.eaglei.services.repository.AbstractRepositoryProvider;
import org.eaglei.services.repository.ProviderUtils;
import org.eaglei.services.repository.RepositoryProviderException;
import org.eaglei.services.repository.SecurityProvider;
import org.eaglei.model.jena.SPARQLQueryUtil;
import org.eaglei.model.jena.EIInstanceMinimalFactory;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Resource;

import static org.eaglei.model.jena.SPARQLConstants.CREATION_DATE_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.IS_STUB_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.MODIFIED_DATE_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.PROVIDER_NAME_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.PROVIDER_VARIABLE;

/**
 * 
 * @author Ricardo De Lima
 * @author Lucy Hadden
 * @author Daniela Bourges-Waldegg
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * 
 */
public final class RESTRepositoryProvider extends AbstractRepositoryProvider implements RepositoryProvider {

	private final JenaEIInstanceFactory instanceFactory;

	private final FooRepositorySecurityProvider securityProvider;

	private static final String RDF_FORMAT = "RDF/XML";

	private static final String READ_VIEW = "user";

	public RESTRepositoryProvider(final JenaEIInstanceFactory instanceFactory, final SecurityProvider generalSecurityProvider) {
		this.instanceFactory = instanceFactory;
		securityProvider = new FooRepositorySecurityProvider(generalSecurityProvider);
	}

	//FIXME understand if this belongs here
	public FooRepositorySecurityProvider getDatatoolsSecurityProvider() {
		return securityProvider;
	}
	
	// CRUD
	public List<EIURI> getNewInstanceID(final Session session, final int count) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );

		final PostMethod method = new PostMethod( RestCommands.GetNewInstanceIDs.getURL() );
		method.setParameter( "accept", "application/sparql-results+xml" );
		method.setParameter( "count", Integer.toString( count ) );
		log.info( "getting " + count + " instance IDs" );
		final String responseBody = ProviderUtils.getHttpResponse( securityProvider.getHttpClient(session), method );
		final ResultSet results = ResultSetFactory.fromXML( responseBody );
		final List<EIURI> instances = new ArrayList<EIURI>( count );
		while ( results.hasNext() ) {
			final QuerySolution soln = results.nextSolution();
			final Resource r = soln.getResource( "new" );
			final String id = r.getURI();
			instances.add( EIURI.create( id ) );
			log.info( "Created new instance id: " + id );
		}
		if ( isDebugEnabled ) {
			log.debug( "Created " + instances.size() + " ids" );
		}
		if ( instances.isEmpty() ) {
			throw new RepositoryProviderException( "unable to get requested instance IDs" );
		}
		return instances;
	}

	public EIInstance getEmptyEIInstance(final Session session, final EIURI classUri, final EIEntity instanceEntity) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		EIInstance instance = instanceFactory.createEmpty( classUri, instanceEntity );
		if ( isNull( instance ) ) {
			throw new RepositoryProviderException( "Unable to create empty instance of class: " + classUri + " and entity " + instanceEntity );
		}
		return instance;
	}

	public EIInstance getEmptyEIInstance(final Session session, final EIURI classUri) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );

		final List<EIURI> ids = getNewInstanceID( session, 1 );
		if ( isDebugEnabled ) {
			log.debug( "New instance ID is: " + ids.get( 0 ) );
		}
		final EIInstance instance = instanceFactory.createEmpty( classUri, EIEntity.create( ids.get( 0 ), "" ) );
		if ( isNull( instance ) ) {
			throw new RepositoryProviderException( "Unable to create empty instance of class: " + classUri );
		}
		return instance;
	}

	// For delete we only need the token
	private String getToken(final Session session, final String instanceUri) throws RepositoryProviderException {
		return updateAndGetToken( session, null, instanceUri );
	}

	public String getToken(final Session session, final EIInstance instance) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );

		if ( isNull( instance ) ) {
			log.warn( "trying to get token to update invalid instance" );
			return null;
		}

		return getToken( session, instance.getInstanceURI().toString() );
	}

	public Map<EIInstance, String> getTokens(final Session session, final List<EIInstance> instances) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		if ( instances == null ) {
			log.warn( "trying to get token to update invalid instance list" );
			return null;
		}

		final Map<EIInstance, String> result = new HashMap<EIInstance, String>();
		for (final EIInstance instance : instances) {
			result.put( instance, getToken( session, instance ) );
		}

		return result;
	}

	@Override
	public void updateInstance(final Session session, final EIInstance instance, final String token) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		if ( isNull( instance ) ) {
			log.warn( "Trying to update instance with null instance or null instance URI" );
			return;
		}
		// clear read-only properties so we don't screw up the metadata
		// repo-side
		if ( instance.isExtendedInstance() ) {
			instance.setReadOnlyLiteralProperties( new HashMap<EIEntity, String>() );
		}
		final List<EIURI> mainAndEmbeddedUris = new ArrayList<EIURI>();
		mainAndEmbeddedUris.add( instance.getInstanceURI() );
		// FIXME this sucks - need to get the instance to compare which EIs existed before and which are new
		if ( !instance.isStub() ) {
			final EIInstance existingInstance = getOneInstance( session, instance.getInstanceURI(), true );
			if ( existingInstance.hasEmbeddedInstances() ) {
				mainAndEmbeddedUris.addAll( existingInstance.getEmbeddedInstanceUriList() );
			}

			if ( instance.hasEmbeddedInstances() ) {
				removeEmptyEmbeddedInstances( instance );
			}
		}
		final String instanceAsString = instanceFactory.serialize( instance, RDF_FORMAT );
		final String instanceUri = instance.getInstanceURI().toString();
		updateOneInstance( session, instanceAsString, instanceUri, mainAndEmbeddedUris, token );
	}

	// FIXME the String version of update *does not* update embedded instances (this is used for ETL)
	public String updateInstance(final Session session, final String rdfString, final String instanceUri, final String token) throws RepositoryProviderException {
		return updateOneInstance( session, rdfString, instanceUri, Arrays.asList( EIURI.create( instanceUri ) ), token );
	}

	public void updateInstances(final Session session, final Map<EIInstance, String> instancesWithTokens) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		if ( instancesWithTokens == null ) {
			log.warn( "trying to update invalid instance list" );
			return;
		}

		for (final EIInstance instance : instancesWithTokens.keySet()) {
			log.debug( "Updating one instance in list: " + instance );
			updateInstance( session, instance, instancesWithTokens.get( instance ) );
		}
	}

	// TODO: put in a Provider & tag override
	public void createInstances(final Session session, final List<EIInstance> instances, final EIEntity workspaceEntity) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		if ( instances == null ) {
			log.warn( "trying to create instances with null list" );
			return;
		}

		if ( isDebugEnabled ) {
			log.debug( "bulk-creating " + instances.size() + " instances" );
		}

		for (final EIInstance instance : instances) {
			if ( isDebugEnabled ) {
				log.debug( "bulk-creating instance " + instance.getInstanceLabel() );
			}
			createInstance( session, instance, workspaceEntity );
		}
	}

	@Override
	public void createInstance(final Session session, final EIInstance instance, final EIEntity workspaceEntity) throws RepositoryProviderException {
		log.info( "Instance label is   " + instance.getInstanceLabel().toString() );
		securityProvider.isValid( session.getSessionId(), true );
		if ( isNull( instance ) ) {
			log.warn( "Trying to create (duplicate) instance with null instance or null instance URI" );
			return;
		}
		if ( instance.hasEmbeddedInstances() ) {
			removeEmptyEmbeddedInstances( instance );
		}

		final String instanceUri = instance.getInstanceURI().toString();
		final String instanceAsString = instanceFactory.serialize( instance, RDF_FORMAT );
		createInstance( session, instanceAsString, instanceUri, workspaceEntity );
	}

	public void createInstance(final Session session, final String rdfString, final String instanceUri, final EIEntity workspaceEntity) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		if ( rdfString == null ) {
			log.warn( "Trying to create (duplicate) instance with null instance or null instance URI" );
			return;
		}
		final PostMethod method = new PostMethod( RestCommands.UpdateInstance.getURL() );
		final Part idPart = new StringPart( "uri", instanceUri, "utf-8" );

		Part workspacePart = null;
		if ( workspaceEntity != null && !workspaceEntity.equals( EIEntity.NULL_ENTITY ) ) {
			workspacePart = new StringPart( "workspace", workspaceEntity.getURI().toString(), CharEncoding.UTF_8 );
		} else {
			workspacePart = new StringPart( "workspace", DEFAULT_WORKSPACE_ENTITY.getURI().toString(), CharEncoding.UTF_8 );
		}

		final Part actionPart = new StringPart( "action", "create" );
		final Part insertPart = new XmlStringPart( "insert", rdfString, CharEncoding.UTF_8 );

		final Part[] parts = { workspacePart, actionPart, insertPart, idPart };
		method.setRequestEntity( new MultipartRequestEntity( parts, method.getParams() ) );
		if ( isDebugEnabled ) {
			log.debug( "Trying to create instance at " + instanceUri );
		}

		ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
	}

	public EIInstance deepCopy(final Session session, final EIURI originalUri) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );

		final EIInstance original = getOneInstance( session, originalUri, true );
		if ( isDebugEnabled ) {
			log.debug( "got original instance" );
		}
		EIInstance instance = duplicateInstance( session, original );
		if ( isNull( instance ) ) {
			throw new RepositoryProviderException( "result of deepcopy is null for instance: " + originalUri );
		}
		return instance;
	}

	public void deleteInstances(final Session session, final List<EIURI> instanceUris) throws RepositoryProviderException {
		for (final EIURI instanceUri : instanceUris) {
			deleteInstance( session, instanceUri );
		}
	}

	@Override
	public void deleteInstance(final Session session, final EIURI instanceUri) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		if ( instanceUri == null || instanceUri.toString() == null || instanceUri.toString().length() == 0 ) {
			return;
		}
		deleteOneInstance( session, instanceUri );
	}

	public Map<EIURI, String> getModifiedDates(final Session session, final List<EIURI> uris) throws RepositoryProviderException {
		final Map<EIURI, String> uriToModifiedDate = new HashMap<EIURI, String>();

		for (final EIURI uri : uris) {
			final String date = getModifiedDate( session, uri );
			if ( isDebugEnabled ) {
				log.debug( date );
			}
			uriToModifiedDate.put( uri, date );
		}

		return uriToModifiedDate;
	}

	private void deleteOneInstance(final Session session, final EIURI instanceUri) throws RepositoryProviderException {
		final List<EIURI> urisToDelete = new ArrayList<EIURI>();
		urisToDelete.add( instanceUri );
		// FIXME this sucks, we don't have the instance hence we don't have the embedded instances; need to do a get first
		final EIInstance outerInstance = getOneInstance( session, instanceUri, true );
		if ( outerInstance.hasEmbeddedInstances() ) {
			urisToDelete.addAll( outerInstance.getEmbeddedInstanceUriList() );
		}

		final String token = getToken( session, instanceUri.toString() );
		final PostMethod method = new PostMethod( RestCommands.UpdateInstance.getURL() );
		setWriteParameters( method, instanceUri.toString() );
		method.setParameter( "action", "update" );
		method.setParameter( "token", token );
		method.setParameter( "delete", createDeleteStatement( urisToDelete ) );
		if ( isDebugEnabled ) {
			log.debug( "Trying to delete instance at " + instanceUri.toString() );
		}
		ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
	}

	private String getModifiedDate(final Session session, final EIURI uri) throws RepositoryProviderException {
		final String query = SPARQLQueryUtil.getInstance().getModifiedDateQuery( uri );

		final String result = postQuery( session, query, READ_VIEW );

		final ResultSet resultSet = ResultSetFactory.fromXML( result );
		String modifiedDate = "";
		while ( resultSet.hasNext() ) {
			final QuerySolution qs = resultSet.next();
			if ( qs.contains( MODIFIED_DATE_VARIABLE ) ) {
				modifiedDate = qs.getLiteral( MODIFIED_DATE_VARIABLE ).getString();
			}
		}
		return modifiedDate;
	}

	// End CRUD

	// DatatoolsMetaDataProvider

	// TODO: put in metadata interface, get out of here(?)
	// WARNING: BROKEN.
	@Deprecated
	public boolean addLabToUser(final Session session, final EIURI labUri) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );

		if ( labUri == null ) {
			log.warn( "Trying to associate with null lab" );
			return false;
		}

		log.info( "addLabToUser : " + labUri );
		final String content = "<" + session.getUserURI() + "> <" + DatatoolsMetadataConstants.IsWatchingLabUri.toString() + "> <" + labUri + "> .";
		log.info( " rdf user lab string: " + content );
		final PostMethod method = new PostMethod( RestCommands.Graph.getURL() );
		method.setParameter( "format", "text/plain" );
		method.setParameter( "name", DatatoolsMetadataConstants.DatatoolsMetadataWorkspace );
		method.setParameter( "type", "workspace" );
		method.setParameter( "action", "add" );
		method.setParameter( "content", content );

		try {
			ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
			return true;
		} catch (final RepositoryProviderException e) {
			log.error( e );
			return false;
		}
	}

	// End DatatoolsMetaDataProvider

	// QueryProvider
	public List<EIInstanceMinimal> EIQuery(final Session session, final String sparql) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		if ( isDebugEnabled ) {
			log.debug( "Sparql: " + sparql );
		}
		final ResultSet resultSet = ResultSetFactory.fromXML( postQuery( session, sparql ) );
		return EIInstanceMinimalFactory.getInstance().create( resultSet );
	}

	public List<EIInstanceMinimal> listResourcesForObjectPropertyValue(final Session session, final EIURI classUri, final EIURI provider, final EIURI state, final boolean onlyProvider) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		final String query = SPARQLQueryUtil.getInstance().getResourcesForObjectPropertyValuesQuery( classUri, provider, state, onlyProvider );
		return EIQuery( session, query );
	}

	// The return format is hardcoded to RDF/XML
	// FIXME should we allow the format to be specified?
	public String query(final Session session, final String sparql) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );

		return postQuery( session, sparql );
	}

	public String retrieveLabel(final Session session, final EIURI uri) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		final String q = SPARQLQueryUtil.getInstance().getRetrieveLabelQuery( uri );
		if ( isDebugEnabled ) {
			log.debug( "retrieve label query: " + q );
		}
		final String result = postQuery( session, q, READ_VIEW );
		if ( isDebugEnabled ) {
			log.debug( "result " + result );
		}
		final ResultSet resultSet = ResultSetFactory.fromXML( result );
		final List<String> labels = new ArrayList<String>();
		while ( resultSet.hasNext() ) {
			final QuerySolution qs = resultSet.next();
			if ( qs.contains( LABEL_VARIABLE ) ) {
				labels.add( qs.getLiteral( LABEL_VARIABLE ).getString() );
			}
		}
		return labels.isEmpty() ? "" : labels.get( 0 );

	}

	// End QueryProvider

	// InstanceProvider
	@Override
	public EIInstance setReferencingResources(Session session, EIInstance instance) throws RepositoryProviderException {
		final ResultSet results = ResultSetFactory.fromXML( postQuery( session, SPARQLQueryUtil.getInstance().getReferencedByQuery( instance.getInstanceURI() ) ) );
		EIInstanceMinimalFactory.getInstance().setReferencingInstances( instance, results );
		return instance;
	}

	public EIInstance getOneInstance(final Session session, final EIURI instanceUri, final boolean isExtended) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		if ( instanceUri == null || instanceUri.toString() == null || EIURI.NULL_EIURI.equals( instanceUri ) ) {
			log.warn( "Trying to get instance for null EIURI" );
			throw new RepositoryProviderException( "Trying to get instance for null EIURI" );
		}
		final PostMethod method = new PostMethod( RestCommands.GetInstance.getURL() );
		setReadParameters( method );
		method.setParameter( "uri", instanceUri.toString() );
		if ( isDebugEnabled ) {
			log.debug( "Trying to get instance " + instanceUri.toString() );
		}
		final String responseBody = ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
		EIInstance eiInstance;
		eiInstance = instanceFactory.createExtended( instanceUri, responseBody, RDF_FORMAT );

		if ( isNull( eiInstance ) ) {
			throw new RepositoryProviderException( "Unable to create EIInstance for instanceUri: " + instanceUri );
		}
		eiInstance.setStubs( getAllStubEntities( session, eiInstance ) );
		return eiInstance;
	}

	// End InstanceProvider

	// WorkflowProvider
	@Override
	public List<EIInstanceMinimal> listResources(final Session session, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean isAscending, final boolean strictOwnerFilter) throws RepositoryProviderException {
		return listResources( session, queryRequest, orderBy, isAscending, strictOwnerFilter, false );
	}

	@Override
	public List<EIInstanceMinimal> listResources(final Session session, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean isAscending, final boolean strictOwnerFilter, final boolean stubsOnly)
			throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );

		final PostMethod method = new PostMethod( RestCommands.ListResources.getURL() );
		// State restriction
		final EIURI state = queryRequest.getWFState();

		if ( state != null && !state.equals( EIURI.NULL_EIURI ) ) {
			method.setParameter( "state", state.toString() );
		} else {
			method.setParameter( "state", "all" );
		}
		final StringBuilder sparql = new StringBuilder();

		// Type restriction
		final EIURI classUri = queryRequest.getType();
		if ( classUri != null && !classUri.equals( EIURI.NULL_EIURI ) && !SPARQLQueryUtil.getInstance().isClassGroup( classUri ) ) {
			method.setParameter( "type", classUri.toString() );
		} else if ( SPARQLQueryUtil.getInstance().isClassGroup( classUri ) ) {
			sparql.append( SPARQLQueryUtil.getInstance().getClassGroupPattern( classUri ) );
		} else { // restrict to eagle-i resources
			sparql.append( SPARQLQueryUtil.getInstance().allTypesPattern( !stubsOnly ) );
		}

		// Resource provider restriction
		final EIURI provider = queryRequest.getLab();
		if ( provider != null && !provider.equals( EIURI.NULL_EIURI ) ) {
			sparql.append( SPARQLQueryUtil.getInstance().providerRestrictionPattern( provider ) );
		} else {
			sparql.append( SPARQLQueryUtil.getInstance().providerRestrictionPattern( EIURI.NULL_EIURI ) );
		}

		sparql.append( SPARQLQueryUtil.getInstance().getStubPattern( !stubsOnly ) );
		String sortOrder = "";
		if ( !isAscending ) {
			sortOrder = "DESC";
		}
		commonListResourcesMethodSetup( EIURI.create( session.getUserURI() ), queryRequest, orderBy, sortOrder, strictOwnerFilter, sparql, method );

		final ResultSet resultSet = ResultSetFactory.fromXML( ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method ) );
		final List<EIInstanceMinimal> results = EIInstanceMinimalFactory.getInstance().create( resultSet );
		if ( isDebugEnabled ) {
			log.debug( "number of resources retrieved: " + results.size() );
		}
		return results;
	}

	@Override
	public List<EIInstanceMinimal> listReferencingResources(final Session session, final EIURI resourceUri, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		final PostMethod method = new PostMethod( RestCommands.ListResources.getURL() );
		method.setParameter( "state", "all" );
		final StringBuilder sparql = new StringBuilder();
		// Restrict to eagle-i resources
		sparql.append( SPARQLQueryUtil.getInstance().allTypesPattern( false ) );
		// Include lab info
		sparql.append( SPARQLQueryUtil.getInstance().providerRestrictionPattern( EIURI.NULL_EIURI ) );
		// Restrict to referenced resources
		sparql.append( SPARQLQueryUtil.getInstance().referencedByPattern( resourceUri ) );

		commonListResourcesMethodSetup( EIURI.create( session.getUserURI() ), queryRequest, orderBy, "", strictOwnerFilter, sparql, method );

		final ResultSet resultSet = ResultSetFactory.fromXML( ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method ) );
		final List<EIInstanceMinimal> results = EIInstanceMinimalFactory.getInstance().create( resultSet );
		return results;
	}

	@Override
	public List<EIURI> claim(final Session session, final List<EIURI> uris) throws RepositoryProviderException {
		final List<EIURI> successes = new ArrayList<EIURI>();
		securityProvider.isValid( session.getSessionId(), true );
		// String currentModifiedDate = getModifiedDate(session, uri)

		final PostMethod method = new PostMethod( RestCommands.Claim.getURL() );

		for (final EIURI uri : uris) {
			method.setParameter( "uri", uri.toString() );
			try {
				ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
				successes.add( uri );
			} catch (final RepositoryProviderException e) {
				log.warn( "could not claim " + uri );
				// failures.add(uri);
			}
		}
		return successes;
	}

	@Override
	public List<EIURI> release(final Session session, final List<EIURI> uris) throws RepositoryProviderException {
		log.info( "releasing " + uris.size() + " uris" );
		final List<EIURI> successes = new ArrayList<EIURI>();
		securityProvider.isValid( session.getSessionId(), true );
		final PostMethod method = new PostMethod( RestCommands.Release.getURL() );

		for (final EIURI uri : uris) {
			method.setParameter( "uri", uri.toString() );
			try {
				ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
				successes.add( uri );
			} catch (final RepositoryProviderException e) {
				log.warn( "could not release " + uri );
				// failures.add(uri);
			}
		}
		return successes;
	}

	@Override
	public List<EIURI> transition(final Session session, final List<EIURI> uris, final EIEntity transitionEntity) throws RepositoryProviderException {
		final List<EIURI> successes = new ArrayList<EIURI>();
		securityProvider.isValid( session.getSessionId(), true );

		for (final EIURI uri : uris) {
			final PostMethod method = new PostMethod( RestCommands.Transition.getURL() );
			method.setParameter( "uri", uri.toString() );
			method.setParameter( "transition", transitionEntity.getURI().toString() );
			try {
				ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
				successes.add( uri );
			} catch (final RepositoryProviderException e) {
				log.warn( "could not transition " + uri );
				// failures.add(uri);
			}
		}

		return successes;
	}

	// End WorkflowProvider

	// Misc.

	protected void setWriteParameters(final PostMethod method, final String id) {
		method.setParameter( "uri", id );
		method.setParameter( "format", FORMAT_VALUE );
		// As per conversation w Larry, repository/update doesn't need a
		// view/graph except for create
		// and removing it when unneeded avoids mismatch of views
		// will add it only in createInstance
		// method.setParameter("workspace",
		// "http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace");
		method.setRequestHeader( "charset", "UTF-8" );
	}

	protected void setReadParameters(final PostMethod method) {
		method.setParameter( "format", FORMAT_VALUE );
		method.setParameter( "view", READ_VIEW );
		method.setParameter( "noinferred", "true" );
		method.setRequestHeader( "charset", "UTF-8" );
	}

	private void removeEmptyEmbeddedInstances(final EIInstance instance) {
		for (final EIInstance embedded : instance.getEmbeddedInstanceList()) {
			if ( embedded.getInstanceLabel().length() == 0 ) {
				instance.removeEmbeddedInstance( embedded.getEntity() );
			}
		}

	}

	private boolean isNull(final EIInstance instance) {
		return instance == null || instance.getInstanceURI() == null || EIInstance.NULL_INSTANCE.equals( instance );
	}

	private String encodeToUTF8(final String rdfString) throws UnsupportedEncodingException {
		return new String( rdfString.getBytes( "UTF-8" ) );
	}

	// FIXME - move duplicate instance either to EIInstance or to EIInstanceFactory?
	public EIInstance duplicateInstance(final Session session, final EIInstance original) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );

		final EIInstance duplicate = getEmptyEIInstance( session, original.getInstanceType().getURI() );

		if ( duplicate == null ) {
			throw new RepositoryProviderException( "unable to duplicate instance: " + original );
		}

		for (final EIEntity property : original.getDatatypeProperties().keySet()) {
			for (final String value : original.getDatatypeProperty( property )) {
				duplicate.addDatattypeProperty( property, value );
			}
		}

		final boolean checkForEmbedded = original.hasEmbeddedInstances();
		for (final EIEntity property : original.getObjectProperties().keySet()) {
			for (final EIEntity value : original.getObjectProperty( property )) {
				if ( checkForEmbedded && original.getEmbeddedInstances().keySet().contains( value ) ) {
					final EIInstance duplicateEmbedded = duplicateInstance( session, original.getEmbeddedInstance( value ) );
					duplicate.addEmbeddedInstance( property, duplicateEmbedded );
					duplicateEmbedded.setInstanceLabel( original.getEmbeddedInstance( value ).getInstanceLabel() );
					// this is not done by addEmbeddedInstance
					duplicate.addObjectProperty( property, duplicateEmbedded.getEntity() );
				} else {
					duplicate.addObjectProperty( property, value );
				}
			}
		}

		for (final EIEntity property : original.getNonOntologyLiteralProperties().keySet()) {
			for (final String value : original.getNonOntologyLiteralProperty( property )) {
				duplicate.addNonOntologyLiteralProperty( property, value );
			}
		}

		for (final EIEntity property : original.getNonOntologyResourceProperties().keySet()) {
			for (final EIEntity value : original.getNonOntologyResourceProperty( property )) {
				duplicate.addNonOntologyResourceProperty( property, value );
			}
		}

		if ( isDebugEnabled ) {
			log.debug( "finished making new copy" );
		}
		return duplicate;
	}

	private String updateOneInstance(final Session session, final String rdfString, final String instanceUri, final List<EIURI> mainAndEmbeddedUris, final String token) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		if ( rdfString == null || rdfString.length() == 0 ) {
			log.warn( "Trying to create (duplicate) instance with null instance or null instance URI" );
			return null;
		}
		if ( token == null ) {
			return updateAndGetToken( session, rdfString, instanceUri );
		}

		final PostMethod method = new PostMethod( RestCommands.UpdateInstance.getURL() );
		final Part idPart = new StringPart( "uri", instanceUri, CharEncoding.UTF_8 );
		final Part actionPart = new StringPart( "action", "update" );
		final Part insertPart = new XmlStringPart( "insert", rdfString, CharEncoding.UTF_8 );
		final Part deletePart = new XmlStringPart( "delete", createDeleteStatement( mainAndEmbeddedUris ), CharEncoding.UTF_8 );
		final Part tokenPart = new StringPart( "token", token );

		final Part[] parts = { idPart, actionPart, insertPart, deletePart, tokenPart };
		method.setRequestEntity( new MultipartRequestEntity( parts, method.getParams() ) );
		if ( isDebugEnabled ) {
			log.debug( "Trying to update instance at " + instanceUri );
		}
		return ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
	}

	private String createDeleteStatement(final List<EIURI> instanceUris) {
		String s = null;
		final StringBuilder wildcard = new StringBuilder();
		for (final EIURI instanceUri : instanceUris) {
			wildcard.append( createDeleteWildcard( instanceUri ) );
			wildcard.append( "\n" );
		}
		log.info( "creating a wildcard graph to delete all the statements: " + wildcard.toString() );
		final Model m = ModelFactory.createDefaultModel();
		final StringReader st = new StringReader( wildcard.toString() );
		m.read( st, null, "N-TRIPLE" );
		final StringWriter sw = new StringWriter();
		try {
			// TODO validate format
			m.write( sw, RDF_FORMAT );
			s = sw.toString();
			sw.flush();
		} finally {
			try {
				if ( sw != null ) {
					sw.close();
				}
			} catch (final IOException e) {
				log.warn( "createDelete statement: " + e );
			}
		}
		return s;
	}

	private String createDeleteWildcard(final EIURI instanceUri) {
		final String wildcard = "<" + instanceUri.toString() + "> <http://eagle-i.org/ont/repo/1.0/MatchAnything> <http://eagle-i.org/ont/repo/1.0/MatchAnything> .";
		return wildcard;
	}

	private String updateAndGetToken(final Session session, final String rdfString, final String instanceUri) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );
		final List<Part> partsList = new ArrayList<Part>();
		final PostMethod method = new PostMethod( RestCommands.UpdateInstance.getURL() );
		final Part idPart = new StringPart( "uri", instanceUri, CharEncoding.UTF_8 );
		final Part actionPart = new StringPart( "action", "gettoken", CharEncoding.UTF_8 );
		partsList.add( idPart );
		partsList.add( actionPart );

		if ( rdfString != null && rdfString.length() > 0 ) {
			final Part insertPart = new XmlStringPart( "insert", rdfString, CharEncoding.UTF_8 );
			partsList.add( insertPart );

		}
		method.setRequestEntity( new MultipartRequestEntity( partsList.toArray( new Part[partsList.size()] ), method.getParams() ) );
		final String resp = ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
		return parseResponseToken( resp );
	}

	private String parseResponseToken(final String resp) {
		final ResultSet results = ResultSetFactory.fromXML( resp );
		final QuerySolution soln = results.nextSolution();
		final Resource r = soln.getResource( "token" );
		final String tokenid = r.getURI();
		log.info( "Retrieved new token " + tokenid );
		return tokenid;
	}

	protected Set<EIEntity> getAllStubEntities(final Session session, final EIInstance instance) throws RepositoryProviderException {
		final Set<EIEntity> stubs = getStubEntities( session, instance.getInstanceURI() );
		if ( instance.hasEmbeddedInstances() ) {
			for (EIURI embeddedUri : instance.getEmbeddedInstanceUriList()) {
				stubs.addAll( getStubEntities( session, embeddedUri ) );
			}
		}

		return stubs;
	}

	private Set<EIEntity> getStubEntities(final Session session, final EIURI instanceUri) throws RepositoryProviderException {
		final ResultSet results = ResultSetFactory.fromXML( postQuery( session, SPARQLQueryUtil.getInstance().getRetrieveStubsQuery( instanceUri ) ) );
		final Set<EIEntity> stubs = new HashSet<EIEntity>();
		while ( results.hasNext() ) {
			final QuerySolution solution = results.nextSolution();
			if ( solution.contains( SPARQLConstants.SUBJECT_VARIABLE ) && solution.contains( SPARQLConstants.LABEL_VARIABLE ) ) {
				final EIURI uri = EIURI.create( solution.getResource( SPARQLConstants.SUBJECT_VARIABLE ).getURI() );
				final String label = solution.getLiteral( LABEL_VARIABLE ).getString();
				stubs.add( EIEntity.create( uri, label ) );
			} else {
				continue;
			}
		}
		return stubs;
	}

	private synchronized String postQuery(final Session session, final String q) throws RepositoryProviderException {
		return postQuery( session, q, READ_VIEW );
	}

	private synchronized String postQuery(final Session session, final String q, final String view) throws RepositoryProviderException {
		securityProvider.isValid( session.getSessionId(), true );

		if ( q == null ) {
			log.warn( "Null query!" );
			return null;
		}
		final PostMethod method = new PostMethod( RestCommands.Query.getURL() );
		if ( isDebugEnabled ) {
			log.debug( "Trying to query: " + RestCommands.Query.getURL() );
		}
		setReadParameters( method );
		method.setParameter( "view", view );
		try {
			method.setParameter( "query", encodeToUTF8( q ) );
		} catch (final UnsupportedEncodingException e) {
			log.warn( "could not encode to utf-8: " + e.getMessage() );
			log.warn( "will send unencoded query string: " + q );
			method.setParameter( "query", q );
		}
		return ProviderUtils.getHttpResponse( securityProvider.getHttpClient( session ), method );
	}

	private void commonListResourcesMethodSetup(final EIURI user, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final String sortOrder, final boolean strictOwnerFilter, final StringBuilder sparql, final PostMethod method) {
		method.setParameter( "detail", "full" );
		method.setParameter( "format", FORMAT_VALUE );

		// TODO see at the UI level if other combinations are needed (e.g. owner=self + unclaimed=true for all resources available to user)
		if ( strictOwnerFilter ) {
			method.setParameter( "owner", "self" );
			method.setParameter( "unclaimed", "false" );
		} else {
			method.setParameter( "owner", "all" );
			method.setParameter( "unclaimed", "true" );
		}

		sparql.append( SPARQLQueryUtil.getInstance().modifiedDatePattern() );
		// include additional orderBy variables only if necessary
		if ( orderBy == SortByProperties.status || orderBy == SortByProperties.type ) {
			sparql.append( SPARQLQueryUtil.getInstance().additionalLabelsPattern() );
		}

		// FIXME this is a fix for the erroneously inserted dcterms:created triples
		// should be removed once the data is cleaned up
		sparql.append( " filter(!regex(str(?" ).append( CREATION_DATE_VARIABLE ).append( "), \"http://www.w3.org/2001/XMLSchema#dateTime\"))" );
		// FIXME fix for resources with no labels (needs cleanup)
		sparql.append( " filter(bound(?" ).append( LABEL_VARIABLE ).append( "))" );

		if ( isDebugEnabled ) {
			log.debug( "Sparql pattern sent to workflow/resources is: " + sparql.toString() );

		}

		try {
			method.setParameter( "addPattern", encodeToUTF8( sparql.toString() ) );
		} catch (UnsupportedEncodingException e) {
			log.warn( "could not encode to utf-8: " + e.getMessage() );
			log.warn( "will send unencoded query string pattern: " + sparql.toString() );
			method.setParameter( "addPattern", sparql.toString() );
		}
		method.setParameter( "addResults", "?" + PROVIDER_VARIABLE + " ?" + PROVIDER_NAME_VARIABLE + " ?" + MODIFIED_DATE_VARIABLE + " ?" + IS_STUB_VARIABLE );
		final StringBuilder modifiers = new StringBuilder();
		modifiers.append( " ORDER BY " + sortOrder + "( <http://eagle-i.org/ont/repo/1.0/upperCaseStr>(?" ).append( orderBy.getVariable() ).append( ") )" );

		// FIXME verify that there are no fence post errors
		if ( queryRequest.isPaginated() && queryRequest.getMaxResults() > 0 && queryRequest.getStartIndex() >= 0 ) {
			modifiers.append( "LIMIT " ).append( String.valueOf( queryRequest.getMaxResults() ) ).append( " OFFSET " ).append( String.valueOf( queryRequest.getStartIndex() ) );
		}
		method.setParameter( "addModifiers", modifiers.toString() );

		if ( isDebugEnabled ) {
			log.debug( "modifiers sent to workflow/resources are: " + modifiers.toString() );

		}
	}
}
