package org.eaglei.datatools.jena;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.jena.QueryBuilder.RDFobjectType;
import org.eaglei.datatools.provider.DataManagmentService;
import org.eaglei.model.EIURI;
import org.eaglei.security.Session;
import org.eaglei.services.repository.AbstractRepositoryProvider;
import org.eaglei.services.repository.RepositoryProviderException;
import org.eaglei.services.repository.SecurityProvider;

import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;

public class DataManagmentProvider extends AbstractRepositoryProvider implements DataManagmentService {

	//THis doesn' seem to be used (dbw)
	//private final FooRepositorySecurityProvider securityProvider;
	public static final String[] graphs = new String[] { DatatoolsMetadataConstants.DatatoolsPublished, DatatoolsMetadataConstants.DatatoolsWithdrawn, DatatoolsMetadataConstants.DatatoolsMetadataWorkspace };
	protected static final Log activityLogger = LogFactory.getLog( "bulk curation activity:" );
	private QueryBuilder queryBuilder;
	TriplesCRUDService triplesService;
	private final String repositoryURL;

	public DataManagmentProvider(final SecurityProvider generalSecurityProvider, final String repositoryURL) {
		this.repositoryURL=repositoryURL;
		//securityProvider = new FooRepositorySecurityProvider( generalSecurityProvider );
		queryBuilder = BulkCurationQueryBuilder.getInstance();
		triplesService = new TriplesCRUDService( generalSecurityProvider, repositoryURL );
	}

	
	public List<BulkCurationTriple> getResourcesToRepair(Session session, EIURI typeURI, EIURI predicateURI, String objectString, Boolean isObjectLiteral) throws RepositoryProviderException {
		List<BulkCurationTriple> instanceList = new ArrayList<BulkCurationTriple>();

		for (String graph : graphs) {

			String query = queryBuilder.diagnoseQuery( typeURI, predicateURI, objectString, getObjectTypeByBoolean( isObjectLiteral ), graph );
			if ( log.isDebugEnabled() ) {
				log.debug( "the query to get resource is " + query );
			}
			instanceList.addAll( getResourcesToRepair( session, query ) );
		}
		return instanceList;
	}

	
	public List<BulkCurationTriple> getResourcesToRepairByRegex(Session session, EIURI typeURI, EIURI predicateURI, String regex) throws RepositoryProviderException {
		List<BulkCurationTriple> instanceList = new ArrayList<BulkCurationTriple>();
		for (String graph : graphs) {
			List<BulkCurationTriple> resources = getResourcesToRepairByRegexOneGraph( session, typeURI, predicateURI, regex, graph );
			instanceList.addAll( resources );
		}
		return instanceList;
	}
	
	public List<BulkCurationTriple> getMetadataResourcesToRepairByRegex(Session session, EIURI typeURI, EIURI predicateURI, String regex) throws RepositoryProviderException {
		String query = queryBuilder.diagnoseQueryByRegexForMetadata(predicateURI, regex, DatatoolsMetadataConstants.Metadata);
		if ( log.isDebugEnabled() ) {
			log.debug( "The query to get resources is " + query );
		}
		return getResourcesToRepair( session, query );
	}

	//Private methods for common operations
	private List<BulkCurationTriple> getResourcesToRepair(Session session, String query) throws RepositoryProviderException {
		String graphContent = triplesService.getTriplesToDisplay( session, query );
		final ResultSet resultSet = ResultSetFactory.fromXML( graphContent );
		final List<BulkCurationTriple> listOfTriples = BulkCurationTripleFactory.getBulkCurationTriples( resultSet );
		if ( log.isDebugEnabled() ) {
			log.debug( "Returning " + listOfTriples.size() + " triples" );
		}
		return listOfTriples;
	}
	private List<BulkCurationTriple> getResourcesToRepairByRegexOneGraph(Session session, EIURI typeURI, EIURI predicateURI, String regex, String graph) throws RepositoryProviderException {
		String query = queryBuilder.diagnoseQueryByRegex( typeURI, predicateURI, regex, graph );
		if ( log.isDebugEnabled() ) {
			log.debug( "The query to get resource is " + query );
		}
		return getResourcesToRepair( session, query );
	}
	

	@Override
	public Integer replaceObject(Session session, EIURI typeURI, EIURI predicateURI, String oldObjectString, Boolean isOldObjectLiteral, String newObjectString, Boolean isNewObjectLiteral) throws RepositoryProviderException {

		Integer before = SanityCheck( session, typeURI, predicateURI, predicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ), newObjectString, getObjectTypeByBoolean( isNewObjectLiteral ) );
		Integer toReplace = triplesService.getCountOfTriples( session, BulkCurationQueryBuilder.getInstance().getSanityCheckQuery( typeURI, predicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ) ) );
		for (String graph : graphs) {
			String addNewObjectquery = queryBuilder.getQueryToAddObject( typeURI, predicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ), newObjectString, getObjectTypeByBoolean( isNewObjectLiteral ), graph );
			String deleteOldObjectquery = queryBuilder.getQueryToDeleteTriple( typeURI, predicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ), graph );
			if ( log.isDebugEnabled() ) {
				log.debug( "query to add new object triple is " + addNewObjectquery );
				log.debug( "query to delete old object triple is " + deleteOldObjectquery );
			}

			replaceResource( session, addNewObjectquery, deleteOldObjectquery, graph );
		}
		Integer after = SanityCheck( session, typeURI, predicateURI, predicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ), newObjectString, getObjectTypeByBoolean( isNewObjectLiteral ) );

		if ( after - before != 0 ) {
			throw new RepositoryProviderException( "Sanity check failed" );
		}
		return new Integer( toReplace );

	}

	@Override
	public Integer replacePredicate(Session session, EIURI typeURI, EIURI oldPredicateURI, EIURI newPredicateURI, String objectString, Boolean isObjectLiteral) throws RepositoryProviderException {
		Integer before = SanityCheck( session, typeURI, oldPredicateURI, newPredicateURI, objectString, getObjectTypeByBoolean( isObjectLiteral ), objectString, getObjectTypeByBoolean( isObjectLiteral ) );
		Integer toReplace = triplesService.getCountOfTriples( session, BulkCurationQueryBuilder.getInstance().getSanityCheckQuery( typeURI, oldPredicateURI, objectString, getObjectTypeByBoolean( isObjectLiteral ) ) );
		
		for (String graph : graphs) {
			String addNewPredicatequery = queryBuilder.getQueryToAddPredicate( typeURI, oldPredicateURI, newPredicateURI, objectString, getObjectTypeByBoolean( isObjectLiteral ), graph );
			String deleteOldObjectquery = queryBuilder.getQueryToDeleteTriple( typeURI, oldPredicateURI, objectString, getObjectTypeByBoolean( isObjectLiteral ), graph );
			if ( log.isDebugEnabled() ) {
				log.debug( "query to add new predicate triple is " + addNewPredicatequery );
				log.debug( "query to delete old predicate triple is " + deleteOldObjectquery );
			}
			replaceResource( session, addNewPredicatequery, deleteOldObjectquery, graph );
		}
		Integer after = SanityCheck( session, typeURI, oldPredicateURI, newPredicateURI, objectString, getObjectTypeByBoolean( isObjectLiteral ), objectString, getObjectTypeByBoolean( isObjectLiteral ) );
		if ( after - before != 0 ) {
			throw new RepositoryProviderException( "Sanity check failed" );
		}
		return new Integer( toReplace );

	}

	@Override
	public Integer replaceObjectAndPredicate(Session session, EIURI typeURI, EIURI oldPredicateURI, EIURI newPredicateURI, String oldObjectString, Boolean isOldObjectLiteral, String newObjectString, Boolean isNewObjectLiteral)
			throws RepositoryProviderException {
		Integer before = SanityCheck( session, typeURI, oldPredicateURI, newPredicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ), newObjectString, getObjectTypeByBoolean( isNewObjectLiteral ) );
		Integer toReplace = triplesService.getCountOfTriples( session, BulkCurationQueryBuilder.getInstance().getSanityCheckQuery( typeURI, oldPredicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ) ) );
		
		for (String graph : graphs) {
			String addNewObjectAndPredicatequery = queryBuilder.getQueryToAddPredicateAndObject( typeURI, oldPredicateURI, newPredicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ), newObjectString,
					getObjectTypeByBoolean( isNewObjectLiteral ), graph );
			String deleteOldObjectquery = queryBuilder.getQueryToDeleteTriple( typeURI, oldPredicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ), graph );
			if ( log.isDebugEnabled() ) {
				log.debug( "query to add new predicate and object triple is " + addNewObjectAndPredicatequery );
				log.debug( "query to delete old  triple is " + deleteOldObjectquery );
			}

			replaceResource( session, addNewObjectAndPredicatequery, deleteOldObjectquery, graph );
		}
		Integer after = SanityCheck( session, typeURI, oldPredicateURI, newPredicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ), newObjectString, getObjectTypeByBoolean( isNewObjectLiteral ) );
		if ( after - before != 0 ) {
			throw new RepositoryProviderException( "Sanity check failed" );
		}
		return new Integer( toReplace );

	}

	//TODO need sanity check
	//e.g. query for triples before, query again afterwards - after query should return 0
	@Override
	public Integer replaceObjectByRegex(Session session, EIURI typeURI, EIURI predicateURI, String regex, String replaceRegex) throws RepositoryProviderException {
		for (String graph : graphs) {
			replaceObjectByRegexForOneGraph( session, typeURI, predicateURI, regex, replaceRegex, graph );
		}
		return new Integer( 1 );
	}

	private void replaceObjectByRegexForOneGraph(Session session, EIURI typeURI, EIURI predicateURI, String regex, String replaceRegex, String graph) throws RepositoryProviderException {
		String constructQryByRegex = queryBuilder.getConstructQueryByRegex( typeURI, predicateURI, regex, graph );
		String graphContent = triplesService.getTriplesToRepair( session, constructQryByRegex );
		String regexReplacedGraphContent = doRegexOperationOnGraph( graphContent, regex, replaceRegex );
		triplesService.addNewTriples( session, regexReplacedGraphContent, graph );
		triplesService.deleteOldTriple( session, graphContent, graph );
	}

	//TODO need sanity check
	//e.g. query for triples before, query again afterwards - after query should return 0
	//**CAUTION**
	//This is only for command lines; we should not offer in GUI (hence it's not in the interface)
	public Integer deleteMetadataObjectByRegex(Session session, EIURI typeURI, EIURI predicateURI, String regex) throws RepositoryProviderException {
		String query = queryBuilder.getQueryToDeleteTriplesByRegexForMetadata( predicateURI, regex, DatatoolsMetadataConstants.Metadata );
		String graphContent = triplesService.getTriplesToRepair( session, query );
		triplesService.deleteOldTriple( session, graphContent, DatatoolsMetadataConstants.Metadata );
		return new Integer( 1 );
	}
	
	
	//TODO sanity check
	//query before and after; after should return 0
	@Override
	public Integer deleteTriple(Session session, EIURI typeURI, EIURI predicateURI, String objectString, Boolean isObjectLiteral) throws RepositoryProviderException {
		for (String graph : graphs) {
			String removeTriplesQuery = queryBuilder.getQueryToDeleteTriple( typeURI, predicateURI, objectString, getObjectTypeByBoolean( isObjectLiteral ), graph );
			String graphContent = triplesService.getTriplesToRepair( session, removeTriplesQuery );
			triplesService.deleteOldTriple( session, graphContent, graph );
			logUserActivity( graph, graphContent, "" );
		}

		return null;
	}

	//TODO sanity check
	@Override
	public Integer addTriple(Session session, EIURI typeURI, EIURI oldPredicateURI, EIURI newPredicateURI, String oldObjectString, Boolean isOldObjectLiteral, String newObjectString, Boolean isNewObjectLiteral) throws RepositoryProviderException {
		for (String graph : graphs) {
			String addNewObjectAndPredicatequery = queryBuilder.getQueryToAddPredicateAndObject( typeURI, oldPredicateURI, newPredicateURI, oldObjectString, getObjectTypeByBoolean( isOldObjectLiteral ), newObjectString,
					getObjectTypeByBoolean( isNewObjectLiteral ), graph );
			String graphContent = triplesService.getTriplesToRepair( session, addNewObjectAndPredicatequery );
			triplesService.addNewTriples( session, graphContent, graph );
			logUserActivity( graph, "", addNewObjectAndPredicatequery );
		}
		return null;
	}

	private void replaceResource(Session session, String addNewObjectQuery, String deleteOldObjectQuery, String graph) throws RepositoryProviderException {
		String addTripleGraphContent = triplesService.getTriplesToRepair( session, addNewObjectQuery );
		String removeTripleGraphContent = triplesService.getTriplesToRepair( session, deleteOldObjectQuery );
		triplesService.addNewTriples( session, addTripleGraphContent, graph );
		triplesService.deleteOldTriple( session, removeTripleGraphContent, graph );
		logUserActivity( graph, removeTripleGraphContent, addTripleGraphContent );
	}

	private String doRegexOperationOnGraph(final String graphContent, final String regex, final String replaceRegex) {
		java.util.regex.Pattern instancePattern = java.util.regex.Pattern.compile( regex, Pattern.CASE_INSENSITIVE | Pattern.MULTILINE );
		java.util.regex.Matcher match = instancePattern.matcher( graphContent );
		String resultGraph = match.replaceAll( replaceRegex );
		return resultGraph;
	}

	private RDFobjectType getObjectTypeByBoolean(Boolean isObjectLiteral) {
		if ( isObjectLiteral ) {
			return RDFobjectType.objectIsLiteral;
		} else {
			return RDFobjectType.objectIsResource;
		}
	}

	public Integer SanityCheck(Session session, EIURI type, EIURI oldPredicate, EIURI newPredicate, String oldObject, RDFobjectType oldObjectType, String newObject, RDFobjectType newObjectType) throws RepositoryProviderException {
		final String queryForOld = BulkCurationQueryBuilder.getInstance().getSanityCheckQuery( type, oldPredicate, oldObject, oldObjectType );
		final String queryForNew = BulkCurationQueryBuilder.getInstance().getSanityCheckQuery( type, newPredicate, newObject, newObjectType );
		final Integer oldCount = triplesService.getCountOfTriples( session, queryForOld );
		final Integer newCount = triplesService.getCountOfTriples( session, queryForNew );
		return oldCount + newCount;

	}

	private void logUserActivity(String namedGraph, String oldTripleGraph, String newTripleGraph) {
		activityLogger.info( "Graph name: " + namedGraph );
		activityLogger.info( "---------------------Old Triples-----------" );
		activityLogger.info( oldTripleGraph );
		activityLogger.info( "---------------------new Triples-----------" );
		activityLogger.info( newTripleGraph );
	}

	public String getDefaultRepositoryProviderURI() {
		return this.DEFAULT_REPOSITORY_URL;
	}

}
