package org.eaglei.datatools.jena;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import javax.servlet.ServletContext;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.User;
import org.eaglei.datatools.WorkFlowTransition;
import org.eaglei.datatools.Workspace;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.datatools.provider.DatatoolsSecurityProvider;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.MetadataConstants;
import org.eaglei.security.Session;
import org.eaglei.services.repository.AbstractRepositoryProvider;
import org.eaglei.services.repository.RepositoryProviderException;
import org.eaglei.services.repository.SecurityProvider;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

/** Class for handling authentication and login calls to the repository. 
 * 
 * @author Jason Barreto
 *
 */
public class FooRepositorySecurityProvider extends AbstractRepositoryProvider implements DatatoolsSecurityProvider {
	
	/** The logger for this class. */
	private static final Log log = LogFactory.getLog( FooRepositorySecurityProvider.class );
	
	private SecurityProvider securityProvider;
	

	public FooRepositorySecurityProvider(SecurityProvider generalSecurityProvider) {
		securityProvider = generalSecurityProvider;
		try {
			DatatoolsConfiguration config = getConfiguration(null);
			DEFAULT_REPOSITORY_URL = config.getDatatoolsRepositoryURL();
			DEFAULT_REPOSITORY_URI = config.getDatatoolsRepositoryURI();
		}
		catch(RepositoryProviderException e) {
			//Presumably this situation can't be handled
			throw new RuntimeException("Error generating Datatools client configuration",e);
		}
	}
	


	@Override
	public User login(final String userName, final String password) throws RepositoryProviderException {
		Session session = securityProvider.logIn(DEFAULT_REPOSITORY_URI, userName, password);

		return getUserInformation(session);

	}
	

	public User login(final String DEFAULT_REPOSITORY_URI,final String userName, final String password) throws RepositoryProviderException {
		Session session = securityProvider.logIn(DEFAULT_REPOSITORY_URI, userName, password);
		return getUserInformation(session);

	}

	@Override
	public void logout(final Session session) throws RepositoryProviderException {
		securityProvider.logOut(session.getSessionId());
	}
	
	

	@Override
	public boolean isOnline() {
		return securityProvider.isOnline(DEFAULT_REPOSITORY_URI);
	}

	@Override
	public List<Workspace> getWorkspaces(final Session session) throws RepositoryProviderException {
		List<HashMap<String, String>> workspaceMapList = securityProvider.getWorkspaces(DEFAULT_REPOSITORY_URI, session.getSessionId());
		List<Workspace> workspaceList = new ArrayList<Workspace>(workspaceMapList.size());
		for (HashMap<String, String> workspaceMap : workspaceMapList) {
			final String workspaceName = workspaceMap.get(MetadataConstants.WORKSPACE_NAMED_GRAPH_LABEL);
			final String workspaceURI = workspaceMap.get( MetadataConstants.WORKSPACE_NAMED_GRAPH_URI );
			final String typeURI = workspaceMap.get( MetadataConstants.WORKSPACE_TYPE );
			final boolean canUserAdd = Boolean.valueOf(workspaceMap.get( MetadataConstants.WORKSPACE_ADD ));
			final boolean canUserDelete = Boolean.valueOf(workspaceMap.get( MetadataConstants.WORKSPACE_REMOVE ));
			Workspace newWorkspace = new Workspace( workspaceName, EIURI.create( workspaceURI ), EIURI.create( typeURI ), canUserAdd, canUserDelete );
			workspaceList.add(newWorkspace);
		}
		
		return workspaceList;
	}

	/** Loads the DataTools configuration from a property file and returns it
	 * as a DatatoolsConfiguration object. The property file location can either be passed
	 * in or is determined by querying for a system property. If the property does not 
	 * exist then a default configuration is used.
	 * 
	 * @param propFile The path to the desired property file. If null is passed in then the
	 * system properties are queried.
	 * @return A DatatoolsConfiguration object representing the configuration present
	 * in the property file.
	 * @throws RepositoryProviderException If loading of the configuration file fails.
	 */
	public DatatoolsConfiguration getConfiguration(String propFile) throws RepositoryProviderException {
		if (propFile == null) {
			// check if the system property is specified
			propFile = System.getProperty( DataToolsOntConstants.DATATOOLS_CONFIG );
			if ( propFile == null ) {
				// use the default
				propFile = DataToolsOntConstants.DEFAULT_DATATOOLS_CONFIG;
			}
		}
		log.debug( "Using properties file " + propFile );

		final URL configURL = this.getClass().getClassLoader().getResource( propFile );
		if ( configURL == null ) {
			log.error( "Could not locate " + propFile + " on classpath " );
			throw new RepositoryProviderException("Unable to locate property configuration file.");
		}
		try {
			final File configFile = new File( configURL.toURI() );
			return new DatatoolsConfiguration( configFile );
		} catch (IOException e) {
			log.error( "Error loading configuration from " + configURL + " " + e );
			throw new RepositoryProviderException("IO Error loading configuration property file.", e);
		} catch (URISyntaxException e) {
			log.error( "Error getting name of configuration file: " + configURL + " " + e);
			throw new RepositoryProviderException("Error parsing URI for configuration file.", e);
		}
	}
	
	private User getUserInformation(Session session) throws RepositoryProviderException {
		
		List<WorkFlowTransition> states = new ArrayList<WorkFlowTransition>();
		states = listWorkFlowTransitions( session, null );
		log.info( "has number of wfsStates " + states.size() );

		List<Workspace> workspaceList = Collections.emptyList();
		return new User( session.getUserName(), EIURI.create( session.getUserURI() ), states, workspaceList, session );
	}
	
	/** Sets the base URL used to define URLs going to the repository.
	 * 
	 * @param repoURL A string representing the url for the repository.
	 */
	public void setDefaultRepositoryURL(String repoURL) {
		DEFAULT_REPOSITORY_URL = repoURL;
	}
	
	/** Sets the base URI used to define URLs going to the repository.
	 * 
	 * @param repoURI A string representing the uri for the repository.
	 */
	public void setDefaultRepositoryURI(String repoURI) {
		DEFAULT_REPOSITORY_URI = repoURI;
	}
	
	@Override
	public List<WorkFlowTransition> listWorkFlowTransitions(final Session session, final EIEntity workspaceEntity) throws RepositoryProviderException {

		List<HashMap<String, EIEntity>> workFlowTransitionMapList = securityProvider.listWorkFlowTransitions(DEFAULT_REPOSITORY_URI, session.getSessionId(), workspaceEntity);
		List<WorkFlowTransition> transitionList = new ArrayList<WorkFlowTransition>(workFlowTransitionMapList.size());
		for (HashMap<String, EIEntity> workFlowTransitionMap : workFlowTransitionMapList) {
			final EIEntity transitionEntity = workFlowTransitionMap.get(WORKFLOW_TRANSITION_SUBJECT);
			final EIEntity fromStateEntity = workFlowTransitionMap.get(WORKFLOW_TRANSITION_FROM);
			final EIEntity toStateEntity = workFlowTransitionMap.get(WORKFLOW_TRANSITION_TO);
			final WorkFlowTransition workFlowTransition = new WorkFlowTransition( transitionEntity, fromStateEntity, toStateEntity, true );
			transitionList.add( workFlowTransition );
		}
		//TODO: Throw if list size is zero (unauthorized user)
		log.info( transitionList.size() + " transitions were returned" );
		return transitionList;
	}

	@Override
	public User whoami(Session session) throws RepositoryProviderException {
		Session newSession = securityProvider.whoami(session.getSessionId());
		return getUserInformation(newSession);
	}
	
	public boolean isValid(String sessionId, boolean shouldThrow) throws RepositoryProviderException {
		return securityProvider.isValid(sessionId, shouldThrow);
	}
	
	public Session getSession(String sessionId) {
		return securityProvider.getSession(sessionId);
	}
	
	public HttpClient getHttpClient(Session session) {
		return securityProvider.getHttpClient(session);
	}
}
