package org.eaglei.datatools.logging;

import java.util.Date;

/**
 * 
 * @author Clint Gilbert
 * 
 */

public final class JSLogger {
	public static enum Level {
		DEBUG(0) {
			@Override
			public void output(final String message) {
				JSLogger.nativeDebug( message );
			}
		},

		INFO(1) {
			@Override
			public void output(final String message) {
				JSLogger.nativeInfo( message );
			}
		},

		WARN(2) {
			@Override
			public void output(final String message) {
				JSLogger.nativeWarn( message );
			}
		},

		ERROR(3) {
			@Override
			public void output(final String message) {
				JSLogger.nativeError( message );
			}
		};

		private final int value;

		private Level(final int value) {
			this.value = value;
		}

		protected final int getValue() {
			return value;
		}

		protected abstract void output(final String message);

		protected final boolean isHigherThanOrEqualTo(final Level level) {
			return value >= level.value;
		}
	}

	private final String name;

	private final Level threshold;

	private final boolean isEnabled;

	private static Level defaultThreshold = Level.INFO;

	public static final Level getDefaultThreshold() {
		return defaultThreshold;
	}

	public static final void setDefaultThreshold(final Level defaultThreshold) {
		JSLogger.defaultThreshold = defaultThreshold != null ? defaultThreshold : Level.INFO;
	}

	private JSLogger(final String name, final Level threshold) {
		super();

		this.name = name;

		this.threshold = threshold;

		isEnabled = isEnabled();
	}

	public static final JSLogger getLogger(final String name, final Level threshold) {
		return new JSLogger( name, threshold );
	}

	public static final JSLogger getLogger(final String name) {
		return getLogger( name, defaultThreshold );
	}

	public final void debug(final String message) {
		log( Level.DEBUG, message );
	}

	public final void info(final String message) {
		log( Level.INFO, message );
	}

	public final void warn(final String message) {
		log( Level.WARN, message );
	}

	public final void error(final String message) {
		log( Level.ERROR, message );
	}

	private final void log(final Level level, final String message) {
		if ( isEnabled ) {
			if ( level.isHigherThanOrEqualTo( threshold ) ) {
				level.output( "[" + new Date() + "][" + level.name() + "][" + name + "][" + message + "]" );
			}
		}
	}

	public final boolean isDebugEnabled() {
		return Level.DEBUG.isHigherThanOrEqualTo( threshold );
	}

	public final boolean isInfoEnabled() {
		return Level.INFO.isHigherThanOrEqualTo( threshold );
	}

	public final boolean isWarnEnabled() {
		return Level.WARN.isHigherThanOrEqualTo( threshold );
	}

	public final boolean isErrorEnabled() {
		return Level.ERROR.isHigherThanOrEqualTo( threshold );
	}

	private static final native boolean isEnabled()/*-{
													try
													{
													return (($wnd.console !== undefined) && ($wnd.console.debug !== undefined ) && ($wnd.console.info !== undefined ) && ($wnd.console.warn !== undefined ) && ($wnd.console.error !== undefined ));
													}
													catch(e)
													{
													return false;
													}
													}-*/;

	private static native final void nativeDebug(final String message)/*-{
																		$wnd.console.debug(message);
																		}-*/;

	private static native final void nativeInfo(final String message)/*-{
																		$wnd.console.info(message);
																		}-*/;

	private static native final void nativeWarn(final String message)/*-{
																		$wnd.console.warn(message);
																		}-*/;

	private static native final void nativeError(final String message)/*-{
																		$wnd.console.error(message);
																		}-*/;
}
