// The MIT License
//
// Copyright (c) 2005 Michael Grove
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.

package org.eaglei.datatools.etl.utils;

import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.ResIterator;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Seq;
import com.hp.hpl.jena.rdf.model.NodeIterator;



import com.hp.hpl.jena.vocabulary.RDF;
import com.hp.hpl.jena.vocabulary.RDFS;
import com.hp.hpl.jena.vocabulary.OWL;
import com.hp.hpl.jena.vocabulary.DAML_OIL;

import java.util.TreeSet;
import java.util.Collections;
import java.util.Set;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;
import java.util.HashSet;
import java.util.LinkedHashSet;

/**
 *
 * <p>Title: Jena Utils</p>
 *
 * <p>Description: A collection of functions for doing common operations using the Jena library</p>
 *
 * <p>Copyright: Copyright (c) 2006</p>
 *
 * <p>Company: Mindswap (http://www.mindswap.org)</p>
 *
 * @author Michael Grove
 * @version 1.0
 */
public class JenaUtils {

    /**
     * XML Scheme URI
     */
    public static final String XML_URI = "http://www.w3.org/2000/10/XMLSchema#";

    /**
     * Array of the URI's of all the XML datatypes
     */
    public static final String[] XML_DATATYPES = new String[] {
        XML_URI+"string", XML_URI+"boolean", XML_URI+"float", XML_URI+"double", XML_URI+"decimal",
        XML_URI+"timeDuration", XML_URI+"binary", XML_URI+"uriReference", XML_URI+"integer", XML_URI+"nonPositiveInteger",
        XML_URI+"negativeInteger", XML_URI+"long", XML_URI+"int", XML_URI+"short", XML_URI+"byte",
        XML_URI+"nonNegativeInteger", XML_URI+"unsignedLong", XML_URI+"unsignedInt", XML_URI+"unsignedShort", XML_URI+"unsignedByte",
        XML_URI+"positiveInteger", XML_URI+"time", XML_URI+"date", XML_URI+"anyURI"
    };

    private static final Model XML_MODEL = ModelFactory.createDefaultModel();
    private static final String NEW_XML_URI = "http://www.w3.org/2001/XMLSchema#";

    public static final Resource XML_STRING = XML_MODEL.createResource(NEW_XML_URI+"string");
    public static final Resource XML_BOOLEAN = XML_MODEL.createResource(NEW_XML_URI+"boolean");
    public static final Resource XML_FLOAT = XML_MODEL.createResource(NEW_XML_URI+"float");
    public static final Resource XML_DOUBLE = XML_MODEL.createResource(NEW_XML_URI+"double");
    public static final Resource XML_DECIMAL = XML_MODEL.createResource(NEW_XML_URI+"decimal");

    public static final Resource XML_INTEGER = XML_MODEL.createResource(NEW_XML_URI+"integer");
    public static final Resource XML_NONPOSITIVE_INTEGER = XML_MODEL.createResource(NEW_XML_URI+"nonPositiveInteger");
    public static final Resource XML_NEGATIVE_INTEGER = XML_MODEL.createResource(NEW_XML_URI+"negativeInteger");
    public static final Resource XML_LONG = XML_MODEL.createResource(NEW_XML_URI+"long");
    public static final Resource XML_INT = XML_MODEL.createResource(NEW_XML_URI+"int");
    public static final Resource XML_SHORT = XML_MODEL.createResource(NEW_XML_URI+"short");
    public static final Resource XML_NONNEGATIVE_INTEGER = XML_MODEL.createResource(NEW_XML_URI+"nonNegativeInteger");
    public static final Resource XML_UNSIGNED_LONG = XML_MODEL.createResource(NEW_XML_URI+"unsignedLong");
    public static final Resource XML_UNSIGNED_INT = XML_MODEL.createResource(NEW_XML_URI+"unsignedInt");
    public static final Resource XML_UNSIGNED_SHORT = XML_MODEL.createResource(NEW_XML_URI+"unsignedShort");
    public static final Resource XML_POSITIVE_INTEGER = XML_MODEL.createResource(NEW_XML_URI+"positiveInteger");
    public static final Resource XML_TIME = XML_MODEL.createResource(NEW_XML_URI+"time");
    public static final Resource XML_DATE = XML_MODEL.createResource(NEW_XML_URI+"date");

    private static Map mNsMap;

    /**
     * Returns whether or not the specified datatype is a known XSD datatype
     * @param s String the datatype
     * @return boolean true of the datatype is an XSD type, false otherwise
     */
    public static boolean isXMLDatatype(String s)
    {
        for (int i = 0; i < XML_DATATYPES.length; i++)
            if (s.equals(XML_DATATYPES[i]))
                return true;
        return false;
    } // isXMLDatatype

    /**
     * Returns the min cardinality of a property
     * @param p Property the property
     * @return int the mincardinality for the given property, -1 defines no set cardinality
     */
    public static int findMinCardinality(Property p)
    {
        // TODO: should take into account restrictions placed on classes
        if (p.listProperties(OWL.minCardinality).hasNext())
            return p.listProperties(OWL.minCardinality).nextStatement().getInt();
        else return -1;
    } // findMinCardinality

    /**
     * Returns the max cardinality of a property
     * @param p Property the property
     * @return int the maxcardinality for the given property, -1 defines no set cardinality
     */
    public static int findMaxCardinality(Property p)
    {
        // TODO: should take into account restrictions placed on classes
        if (p.listProperties(OWL.maxCardinality).hasNext())
            return p.listProperties(OWL.maxCardinality).nextStatement().getInt();
        else return -1;
    } // findMaxCardinality

    /**
     * Returns whether or not the given resouce is a Class
     * @param aRes Resource the resource
     * @return boolean true if the resource is a class, false otherwise
     */
    public static boolean isClass(Resource aRes) {
        return isClass(aRes,aRes.getModel());
    } // isClass

    /**
     * Returns whether or not the given resouce is a Class
     * @param aRes Resource the resource
     * @param theModel Model the kb
     * @return boolean true if the resource is a class, false otherwise
     */
    public static boolean isClass(Resource aRes, Model theModel)
    {
        boolean isclass = theModel.contains(aRes,RDF.type,RDFS.Class) || theModel.contains(aRes,RDF.type,OWL.Class) || theModel.contains(aRes,RDF.type,DAML_OIL.Class);
        if (!isclass)
        {
            StmtIterator sIter = theModel.listStatements(aRes, RDFS.subClassOf, (RDFNode)null);
            while (sIter.hasNext())
            {
                Resource aSubClass = sIter.nextStatement().getResource();
                if (!aRes.equals(aSubClass) && isClass(aSubClass, theModel))
                {
                    isclass = true;
                    break;
                } // if
            } // while
            sIter.close();
        } // if
        return isclass;
    } // isClass

    /**
     * Returns whether or not the given resource is a Property
     * @param aRes Resource the resource
     * @return boolean true if the resource is a proeprty, false otherwise
     */
    public static boolean isProperty(Resource aRes)
    {
        return isRDFProperty(aRes) || isDatatypeProperty(aRes) || isObjectProperty(aRes) || aRes.hasProperty(RDF.type,OWL.AnnotationProperty);
    } // isProperty

    /**
     * Returns whether or not the given resource is defined as an rdf:Property
     * @param aRes Resource the resource
     * @return boolean true if its an rdf:Property, false otherwise
     */
    public static boolean isRDFProperty(Resource aRes)
    {
        boolean isprop = aRes.hasProperty(RDF.type,RDF.Property);
        if (!isprop)
        {
            StmtIterator sIter = aRes.listProperties(RDFS.subPropertyOf);
            while (sIter.hasNext())
            {
                Resource aSubProp = sIter.nextStatement().getResource();
                if (!aRes.equals(aSubProp) && isProperty(aSubProp))
                {
                    isprop = true;
                    break;
                } // if
            } // while
            sIter.close();
        } // if
        return isprop;
    } // isRDFProperty

    /**
     * Returns whether or not the given resource is defined as a datatype property
     * @param aRes Resource the resource
     * @return boolean true if its a datatype property, false otherwise
     */
    public static boolean isDatatypeProperty(Resource aRes)
    {
        return isDatatypeProperty(aRes,aRes.getModel());
    }

    /**
     * Returns whether or not the given resource is defined as a datatype property
     * @param theRes Resource the resource
     * @param theModel Model the kb
     * @return boolean true if its a datatype property, false otherwise
     */
    public static boolean isDatatypeProperty(Resource theRes, Model theModel)
    {
        boolean isprop = theModel.contains(theRes, RDF.type, DAML_OIL.DatatypeProperty) || theModel.contains(theRes, RDF.type, OWL.DatatypeProperty);
        if (!isprop)
        {
            StmtIterator sIter = theModel.listStatements(theRes, RDFS.subPropertyOf, (RDFNode)null);
            while (sIter.hasNext())
            {
                Resource aSubProp = sIter.nextStatement().getResource();
                if (!theRes.equals(aSubProp) && isDatatypeProperty(aSubProp,theModel))
                {
                    isprop = true;
                    break;
                } // if
            } // while
            sIter.close();
        } // if

        if (!isprop)
        {
            Property aProp = (Property)theRes.as(Property.class);
            Iterator rIter = getDeclaredRanges(aProp,theModel).iterator();
            while (rIter.hasNext())
            {
                Resource aRange = (Resource)rIter.next();
                if (!aRange.isAnon() && (isXMLDatatype(aRange.getURI()) || aRange.getURI().equals(RDFS.Literal.getURI()))) {
                    isprop = true;
                    break;
                }
            }
        }

        return isprop;
    } // isDatatypeProperty

    /**
     * Returns whether or not the given resource is defined as an object property
     * @param theRes Resource the resource
     * @return boolean true if its an object property, false otherwise
     */
    public static boolean isObjectProperty(Resource theRes)
    {
        return isObjectProperty(theRes,theRes.getModel());
    }

    /**
     * Returns whether or not the given resource is defined as an object property
     * @param theRes Resource the resource
     * @param theModel Model the kb
     * @return boolean true if its an object property, false otherwise
     */
    public static boolean isObjectProperty(Resource theRes, Model theModel)
    {
        if (theRes.equals(RDF.Property))
            return false;

        boolean isprop = theModel.contains(theRes, RDF.type, DAML_OIL.ObjectProperty) || theModel.contains(theRes, RDF.type, OWL.ObjectProperty);
        if (!isprop)
        {
            StmtIterator sIter = theModel.listStatements(theRes, RDFS.subPropertyOf, (RDFNode)null);
            while (sIter.hasNext())
            {
                Resource aSubProp = sIter.nextStatement().getResource();
                if (!theRes.equals(aSubProp) && isObjectProperty(aSubProp,theModel))
                {
                    isprop = true;
                    break;
                } // if
            } // while
            sIter.close();
        } // if

        if (!isprop)
        {
            Property aProp = (Property)theRes.as(Property.class);
            Iterator rIter = getDeclaredRanges(aProp,theModel).iterator();
            while (rIter.hasNext())
            {
                Resource aRange = (Resource)rIter.next();
                if (isClass(aRange)) {
                    isprop = true;
                    break;
                }
                else continue;
            }
        }

        return isprop;
    } // isObjectProperty

    /**
     * Returns whether or not the given resource is defined as an owl:Ontology
     * @param theRes Resource the resource
     * @return boolean true if its an owl:Ontology, false otherwise
     */
    public static boolean isOntology(Resource theRes) {
        return isOntology(theRes,theRes.getModel());
    }

    /**
     * Returns whether or not the given resource is defined as an owl:Ontology
     * @param theRes Resource the resource
     * @param theModel Model the kb
     * @return boolean true if its an owl:Ontology, false otherwise
     */
    public static boolean isOntology(Resource theRes, Model theModel) {
        return theModel.contains(theRes, RDF.type,OWL.Ontology);
    } // isOntology

    /**
     * Returns all the subclasses of a given resource
     *
     * @param r Resource the resource
     * @param direct boolean if true, return only direct subclasses, otherwise return all
     * @return Set the set of all the subclasses
     */
    public static Set getAllSubClassesOf(Resource r, boolean direct)
    {
        return getAllSubClassesOf(r,r.getModel(),direct);
    }

    /**
     * Returns all the subclasses of a given resource
     * @param r Resource the resource
     * @param m Model the model to use
     * @param direct boolean if true, return only direct subclasses, otherwise return all
     * @return Set the set of all subclasses
     */
    public static Set getAllSubClassesOf(Resource r, Model m, boolean direct)
    {
        Set sc = new HashSet();
        ResIterator rIter = m.listSubjectsWithProperty(RDFS.subClassOf,r);
        while (rIter.hasNext())
        {
            Resource aRes = rIter.nextResource();

             if (aRes.isAnon())
                 continue;

            // skip over rdf/rdfs/owl constructs that are declared as superclasses
            if (aRes.getURI().startsWith(RDF.getURI()) || aRes.getURI().startsWith(RDFS.getURI()) || aRes.getURI().startsWith(OWL.getURI()))
                continue;
            else if (r.equals(aRes))
                continue;

            sc.add(aRes);

            if (!direct)
            {
                Set s = getAllSubClassesOf(aRes,m,direct);
                sc.addAll(s);
            }
        } // while
        rIter.close();
        return sc;
    } // getAllSuperClassesOf


    /**
     * Returns all the superclasses of a given resource
     * @param r Resource the resource
     * @return boolean the set of all the superclasses
     */
    public static Set getAllSuperClassesOf(Resource r, boolean direct)
    {
        return getAllSuperClassesOf(r,r.getModel(),direct);
    }

    /**
     * Returns all the superclasses of the given resource
     * @param r Resource the resource
     * @param m Model the model to search in
     * @return Set the set of all superclasses
     */
    public static Set getAllSuperPropertiesOf(Resource r, Model m)
    {
        Set sc = new HashSet();
        StmtIterator sIter = r.listProperties(RDFS.subPropertyOf);
        while (sIter.hasNext())
        {
            Statement stmt = sIter.nextStatement();
            Resource aRes = stmt.getResource();

             if (aRes.isAnon())
                 continue;

            // skip over rdf/rdfs/owl constructs that are declared as superclasses
            // do we want to do this?
            if (aRes.getURI().startsWith(RDF.getURI()) || aRes.getURI().startsWith(RDFS.getURI()) || aRes.getURI().startsWith(OWL.getURI()))
                continue;
            else if (r.equals(aRes))
                continue;

            sc.add(aRes);

            if (!isLanguageTerm(aRes))
            {
                Set s = getAllSuperPropertiesOf(aRes,m);
                sc.addAll(s);
            }
        }
        sIter.close();
        return sc;
    }

    /**
     * Returns all the superclasses of the given resource
     * @param r Resource the resource
     * @param m Model the model to search in
     * @return Set the set of all superclasses
     */
    public static Set getAllSuperClassesOf(Resource r, Model m, boolean direct)
    {
        Set sc = new HashSet();
        StmtIterator sIter = r.listProperties(RDFS.subClassOf);
        while (sIter.hasNext())
        {
            Statement stmt = sIter.nextStatement();
            Resource aRes = stmt.getResource();

             if (aRes.isAnon())
                 continue;

            // skip over rdf/rdfs/owl constructs that are declared as superclasses
            // do we want to do this?
            if (aRes.getURI().startsWith(RDF.getURI()) || aRes.getURI().startsWith(RDFS.getURI()) || aRes.getURI().startsWith(OWL.getURI()))
                continue;
            else if (r.equals(aRes))
                continue;

            sc.add(aRes);

            if (!isLanguageTerm(aRes))
            {
                if (!direct)
                {
                    Set s = getAllSuperClassesOf(aRes,m,direct);
                    sc.addAll(s);
                }
            }
        }
        sIter.close();
        return sc;
    }

    /**
     * Returns all the subproperties of the given resource
     * @param r Resource the resource
     * @return Set the set of all subproperties of the resource
     */
    public static Set getAllSubPropertiesOf(Resource r) {
        return getAllSubPropertiesOf(r,r.getModel());
    }

    /**
     * Returns all the subproperties of the given resource
     * @param r Resource the property resource
     * @param m Model the model to use
     * @return Set the set of all subproperties of the property resource
     */
    public static Set getAllSubPropertiesOf(Resource r, Model m)
    {
        Set sc = new HashSet();
        ResIterator rIter = m.listSubjectsWithProperty(RDFS.subPropertyOf,r);
        while (rIter.hasNext())
        {
            Resource aRes = rIter.nextResource();

            if (aRes.isAnon())
                continue;

            sc.add(aRes);

            Set s = getAllSubPropertiesOf(aRes);
            sc.addAll(s);
        } // while
        rIter.close();
        return sc;
    } // getAllSubPropertiesOf

    /**
     * Returns the most specific type of an instance
     * @param theRes Resource the instance
     * @param filter boolean if true, filter out Anon types as well as OWL/DAML/RDF/RDFS types.
     * @return Resource the first declared type, or null if no type declared
     */
    public static Resource getType(Resource theRes, boolean filter)
    {
        return getType(theRes, theRes.getModel(), filter);
    }

    /**
     * Returns the most specific type of an instance
     * @param theRes Resource the inst
     * @param theModel Model the kb
     * @param filter boolean if true, filter out Anon types as well as OWL/DAML/RDF/RDFS types.
     * @return Resource the first declared type, or null if no type declared
     */
    public static Resource getType(Resource theRes, Model theModel, boolean filter)
    {
        Set types = getTypes(theRes,theModel, filter);
        Vector v = new Vector(types);
        Collections.sort(v,new SubclassSortResourceComparator());
        if (v.size() > 0)
            return (Resource)v.lastElement();
        else return null;
    } // getType

    /**
     * Returns the set of all stated types for the given instance
     * @param theRes Resource the instance
     * @param filter boolean if true, filter out Anon types as well as OWL/DAML/RDF/RDFS types.
     * @return Vector the list of all types
     */
    public static Set getTypes(Resource theRes, boolean filter)
    {
        return getTypes(theRes, theRes.getModel(), filter);
    }

    public static Set getTypes(Resource theRes, Model theModel, boolean filter)
    {
        Set s = new LinkedHashSet();

        StmtIterator sIter = theModel.listStatements(theRes, RDF.type, (RDFNode)null);
        while (sIter.hasNext())
        {
            Resource type = sIter.nextStatement().getResource();
            if (filter && (type.isAnon() || type.getURI().startsWith(OWL.getURI()) || type.getURI().startsWith(RDFS.getURI()) || type.getURI().startsWith(RDF.getURI())))
                continue;
            else {
                s.add(type);
            }
        } // while
        sIter.close();
        return s;
    }

    /**
     * Return all types of an instance
     * @param theRes Resource the instance resource
     * @param filter boolean if true, filter out OWL/RDF/RDFS/DAML types
     * @return Set the set of all types for the instance
     */
    public static Set getAllTypes(Resource theRes, boolean filter)
    {
        return getAllTypes(theRes, theRes.getModel(), filter);
    }

    /**
     * Return all types of an instance
     * @param theRes Resource the inst
     * @param theModel Model the kb
     * @param filter boolean if true, filter out OWL/RDF/RDFS/DAML types
     * @return Set the set of all types for the instance
     */
    public static Set getAllTypes(Resource theRes, Model theModel, boolean filter)
    {
        Set types = new TreeSet(new SubclassSortResourceComparator());

        Set s = getTypes(theRes,theModel,filter);

        Iterator aIter = s.iterator();
        while (aIter.hasNext())
        {
            Resource aType = (Resource)aIter.next();

            types.add(aType);

            types.addAll(getAllSuperClassesOf(aType,theModel,false));
        }

        return types;
    }

    /**
     * Returns all the ranges for a property with respect to a given class
     * @param aProp Property the property
     * @param theClass Resource the class the property belongs to
     * @return Set the set of all ranges for the property on the specified class
     */
    public static Set getRanges(Property aProp, Resource theClass) {
        return getRanges(aProp,theClass, theClass == null? aProp.getModel() : theClass.getModel());
    }

    /**
     * Returns all the ranges for a property with respect to a given class
     * @param aProp Property the property
     * @param theClass Resource the class the property belonds tot
     * @param theModel Model the model to use
     * @return Set the set of all ranges for the property on the specified class
     */
    public static Set getRanges(Property aProp, Resource theClass, Model theModel) {
        Set ranges = new HashSet();
        ranges.addAll(getDeclaredRanges(aProp,theModel));

        if (aProp.hasProperty(RDFS.subPropertyOf))
        {
            StmtIterator sIter = aProp.listProperties(RDFS.subPropertyOf);
            while (sIter.hasNext())
            {
                Statement stmt = sIter.nextStatement();

                Property p = null;
                if (stmt.getResource() instanceof Property)
                    p = (Property)stmt.getResource();
                else p = (Property)stmt.getResource().as(Property.class);

                if (!p.isAnon() && !p.equals(aProp))
                {
                    Set s = getRanges(p,theClass);
                    ranges.addAll(s);
                } // if
            } // while
            sIter.close();
        } // if

        // now that we've collected all the ranges
        // lets get the subclasses of all the ranges
        Vector newRanges = new Vector();
        Iterator iter = ranges.iterator();
        while (iter.hasNext())
        {
            Resource r = (Resource)iter.next();
            Set sc = getAllSubClassesOf(r,false);
            newRanges.addAll(sc);
        } // for
        ranges.addAll(newRanges);

        if (theClass != null)
            ranges.addAll(getRestrictionRanges(aProp,theClass));

        return ranges;
    }

    /**
     * Returns all the explictly declared ranges for the property
     * @param theProp Property the property
     * @return Set the set of all ranges for the property
     */
    public static Set getDeclaredRanges(Property theProp) {
        return getDeclaredRanges(theProp,theProp.getModel());
    }

    /**
     * Returns all the explicitly declared ranges for the property
     * @param prop Property the property
     * @param theModel Model the model to use
     * @return Set the set of all ranges for the property
     */
    public static Set getDeclaredRanges(Property theProp, Model theModel) {
        Set ranges = new HashSet();
        StmtIterator si = theModel.listStatements(theProp,RDFS.range,(RDFNode)null);
        while (si.hasNext())
        {
          Statement aStmt = si.nextStatement();
          ranges.add(aStmt.getObject());
        } // while
        si.close();
        return ranges;
    }

    private static Set getRestrictionRanges(Property theProp, Resource theClass)
    {
        Set ranges = new HashSet();

        StmtIterator sIter = theClass.listProperties(RDFS.subClassOf);
        while (sIter.hasNext())
        {
            Statement stmt = sIter.nextStatement();
            if (stmt.getObject() instanceof Resource && !theClass.equals(stmt.getResource()) && !stmt.getResource().isAnon()
                    && !stmt.getResource().toString().startsWith(RDFS.getURI()) && !stmt.getResource().toString().startsWith(OWL.getURI()) && !stmt.getResource().toString().startsWith(RDF.getURI()))
                {
                    Set s = getRestrictionRanges(theProp,stmt.getResource());
                    ranges.addAll(s);
                }
            else if (stmt.getObject() instanceof Resource && stmt.getResource().hasProperty(RDF.type,OWL.Restriction))
            {
                Resource restriction = stmt.getResource();

                if (restriction.getProperty(OWL.onProperty).getResource().equals(theProp))
                {
                    // we'll treat some and all values from as the same thing in the code
                    // photostuff doesnt need to make the distinction, at least not yet
                    // doubtful we'll want error checking that sophisticated
                    if (restriction.hasProperty(OWL.allValuesFrom))
                        ranges.add(restriction.getProperty(OWL.allValuesFrom).getObject());
                    else if (restriction.hasProperty(OWL.someValuesFrom))
                        ranges.add(restriction.getProperty(OWL.someValuesFrom).getObject());
                }
            }
        }
        sIter.close();

        return ranges;
    }

    /**
     * Given a Jena model, return the pretty rdf/xml representation of the model
     * NOTE: This function is not complete, dont use, use the jena writer pretty print feature
     * @param m Model the model
     * @return String the string rdf/xml representation of the given model
     */
    public static String getPrettyRDF(Model m)
    {
        mNsMap = m.getNsPrefixMap();

        StringBuffer rdf = new StringBuffer("<"+encode(RDF.getURI()+"RDF")+"\n");
        Set aSet = mNsMap.keySet();
        Iterator iter = aSet.iterator();
        while (iter.hasNext())
        {
            String key = iter.next().toString();
            String value = mNsMap.get(key).toString();
            if (key.equals(""))
                rdf.append("");
            else rdf.append("\txmlns:"+key+"=\""+value+"\"\n");
        } // while

        // print namespaces
        rdf.append(">\n\n");
        Hashtable seen = new Hashtable();

        ResIterator rIter = m.listSubjects();
        while (rIter.hasNext())
        {
            Resource r = rIter.nextResource();

            // skip anon's, they'll be used elsewhere
            // also skip anything we've already seen/used
            if (r.isAnon() || seen.containsKey(r.toString()))
                continue;

            rdf.append(toRDF(r,m,0)+"\n\n");
        } // while
        rIter.close();

        rdf.append("</"+encode(RDF.getURI()+"RDF")+">");
        return rdf.toString();
    } // getPrettyRDF

    /**
     * Helper function for the getPrettyRDF function that actually generates the rdf for a term
     * @param r Resource the term to generate the rdf for
     * @param m Model the parent model
     * @param indent int indent size
     * @return String the string representation of the resource in rdf/xml
     */
    private static String toRDF(Resource r, Model m, int indent)
    {
        String rdfAbout = encode(RDF.getURI()+"about");
        String rdfResource = encode(RDF.getURI()+"resource");

        String ind = "";
        for (int i = 0; i < indent; i++, ind+=" ");
        StringBuffer rdf = new StringBuffer();

        Resource type = r.getProperty(RDF.type).getResource();
        String encType = encode(type.toString());
        if (r.isAnon())
            rdf.append(ind+"<"+encType+">\n");
        else rdf.append(ind+"<"+encType+" "+rdfAbout+"=\""+r.toString()+"\">\n");
        StmtIterator sIter = r.listProperties();
        while (sIter.hasNext())
        {
            Statement stmt = sIter.nextStatement();
            // already processed it, skip
            if (stmt.getPredicate().equals(RDF.type))
                continue;

            String name = encode(stmt.getPredicate().toString());
            if (stmt.getObject() instanceof Literal)
                rdf.append(ind+"  <"+name+">"+stmt.getObject().toString()+"</"+name+">\n");
            else if (stmt.getResource().isAnon())
                rdf.append(ind+"  <"+name+">\n"+toRDF(stmt.getResource(),m,indent+4)+"\n"+ind+"  </"+name+">\n");
            else rdf.append(ind+"  <"+name+" "+rdfResource+"=\""+stmt.getObject().toString()+"\"/>\n");
        } // while
        sIter.close();
        rdf.append(ind+"</"+encType+">");

        return rdf.toString();
    } // toRDF

    /**
     * Helper function for the getPrettyRDF method that encodes the URL of a resource to its qname
     * @param uri String the url of the resource
     * @return String the qname for the resource
     */
    private static String encode(String uri)
    {
        Set aSet = mNsMap.keySet();
        Iterator iter = aSet.iterator();
        String uriPart = uri.substring(0,uri.lastIndexOf("#")+1);

        String prefix = null;
        while (iter.hasNext())
        {
            String key = iter.next().toString();
            String value = mNsMap.get(key).toString();
            if (value.equals(uriPart))
                prefix = key;
        } // while

        if (prefix == null)
            return uri;
        return prefix+":"+uri.substring(uri.lastIndexOf("#")+1);
    } // encode

    /**
     * Returns whether or not the specified string is a valid RDF id
     * @param theId String the id
     * @return boolean returns true if valid id, false otherwise
     */
    public static boolean isValidRDFId(String theId)
    {
    	return true;
    	/*
        boolean valid = true;

        if (theId.length() <= 0)
            return false;

        // this is a full url, lets just check the local name for validity!
        if (BasicUtils.isValidURL(theId))
            theId = getLocalName(theId);

        if (Character.isDigit(theId.charAt(0)) || !(Character.isLetter(theId.charAt(0)) || theId.charAt(0) == '_'))
            valid = false;
        else if (theId.indexOf(":") != -1)
            valid = false;
        else if (theId.indexOf("/") != -1)
            valid = false;
        //else if (theId.indexOf(" ") != -1)
        //    valid = false;
        else if (theId.equals(""))
            valid = false;
        return valid;
        */
    } // isValidRDFId

    /**
     * Returns all the properties that are declared for a given resource
     * @param aRes Resource the resource
     * @return Set the set of all declared properties for the resource
     */
    public static Set getDeclaredProps(Resource aRes) {
        return getDeclaredProps(aRes,aRes.getModel());
    }

    /**
     * Returns all the declared properties for a given resource
     * @param aRes Resource the resource
     * @return Set the set of all declared properties for the given resource
     */
    public static Set getDeclaredProps(Resource aRes, Model theModel)
    {
        HashSet s = new HashSet();
//        if (aRes instanceof OntClass)
//        {
//            OntClass oc = (OntClass)aRes;
//            ExtendedIterator rIter = oc.listDeclaredProperties(true);
//            while (rIter.hasNext())
//                s.add(rIter.next());
//            rIter.close();
//        } // if
//        else if (aRes instanceof DAMLClass)
//        {
//            DAMLClass dc = (DAMLClass)aRes;
//            ExtendedIterator rIter = dc.listDeclaredProperties(true);
//            while (rIter.hasNext())
//                s.add(rIter.next());
//            rIter.close();
//        } // else if
//        else // regular RDF resource - just look at declared domains, super classes
        {

            // gather all the props that say they have this class in their domain
            aRes = theModel.getResource(aRes.toString());

            ResIterator rIter = theModel.listSubjectsWithProperty(RDFS.domain,aRes);
            while (rIter.hasNext())
                s.add(rIter.nextResource());
            rIter.close();

            // see if we're a subclass of a restriction
            // then check that restriction to see if its on a property
            StmtIterator sIter = aRes.listProperties(RDFS.subClassOf);;
            while (sIter.hasNext())
            {
                Statement stmt = sIter.nextStatement();
                if (stmt.getResource().hasProperty(RDF.type,OWL.Restriction))
                {
                    Resource restriction = stmt.getResource();
                    if (restriction.hasProperty(OWL.onProperty))
                    {
                        // i think all that matters is that there is an onProperty restriction
                        // for the prop, i dont think it matters if its accompanied by a
                        // some or all values from, or a cardinality
                        Resource prop = restriction.getProperty(OWL.onProperty).getResource();
                        s.add(prop);
                    }
                }
            }
            sIter.close();

            // search all the superclasses for their props
            sIter = aRes.listProperties(RDFS.subClassOf);
            while (sIter.hasNext())
            {
                Statement stmt = sIter.nextStatement();
                if (stmt.getObject() instanceof Resource && !aRes.equals(stmt.getResource()) && !stmt.getResource().isAnon()
                        && !stmt.getResource().toString().startsWith(RDFS.getURI()) && !stmt.getResource().toString().startsWith(OWL.getURI()) && !stmt.getResource().toString().startsWith(RDF.getURI()))
                {
                    Set temp = getDeclaredProps(stmt.getResource());
                    s.addAll(temp);
                } // if
            } // while
            sIter.close();
        } // else
        return s;
    } // getDeclaredProps

    /**
     * Returns the qname for the given resource
     * @param res Resource the resource
     * @return String the qname for the specified resource
     */
    public static String getQName(Resource res)
    {
        return NamespaceManager.getQNameFor(res.toString());
    } // getQName

    /**
     * Returns whether the given term is in the RDF, RDFS, DAML or OWL namespaces
     * @param r Resource
     * @return boolean
     */
    public static boolean isLanguageTerm(Resource r)
    {
        return !r.isAnon() && (r.getURI().startsWith(RDFS.getURI()) || r.getURI().startsWith(RDF.getURI()) ||
            r.getURI().startsWith(OWL.getURI()) || r.getURI().startsWith(DAML_OIL.NAMESPACE_DAML_2001_03_URI));
    }
    /**
     * Returns the local name for the specified RDF Id
     * @param name String - the id
     * @return String - the corresponding local name
     */
    public static String getLocalName(String name) {
        int slashIndex = name.lastIndexOf("/");
        int hashIndex = name.lastIndexOf("#");

        if (slashIndex > hashIndex)
            return name.substring(slashIndex+1);
        else return name.substring(hashIndex+1);
    } // isLanguageTerm

    /**
     * Return the label for a resource with English as the preferred language
     * @param theResource Resource
     * @return String the label of the resource, or null if one is not found
     */
    public static String getLabel(Resource theResource) {
        return getLabel(theResource,"EN");
    }

    /**
     * Get the label for a resource using the given preferred language
     * @param theResource Resource the resource
     * @param thePreferredLang String the preferred language
     * @return String the label of the resource in the preferred language, or null if one is not found
     */
    public static String getLabel(Resource theResource, String thePreferredLang)
    {
        if (theResource.hasProperty(RDFS.label))
        {
            StmtIterator sIter = theResource.listProperties(RDFS.label);
            while (sIter.hasNext())
            {
                Statement aStmt = sIter.nextStatement();
                if (aStmt.getLiteral().getLanguage().equalsIgnoreCase(thePreferredLang))
                    return aStmt.getLiteral().getString();
            }
            sIter.close();

            // we didn't find the preferred language, just just return the first label
            return theResource.getProperty(RDFS.label).getString();
        }
        else return null;
    }

    /**
     * Get the comment for a resource
     * @param theResource Resource the resource
     * @return String the comment, or null if it does not have one
     */
    public static String getComment(Resource theResource)
    {
        if (theResource.hasProperty(RDFS.comment))
            return theResource.getProperty(RDFS.comment).getObject().toString();
        return null;
    }

    /**
     * Return whether or not the given resource is an individual
     * @param theResource Resource the resource
     * @return boolean true if its an individual, false if not
     */
    public static boolean isIndividual(final Resource theResource)
    {
//        Iterator typeIter = getTypes(theResource,false).iterator();
//
//        while (typeIter.hasNext())
//        {
//            Resource aType = (Resource)typeIter.next();
//            if (isLanguageTerm(aType) && !aType.equals(OWL.Thing))
//                continue;
//            else {
//                return true;
//            }
//        }
//        return false;

//            Set aSet = getTypes(theResource,true);
//            if (aSet.isEmpty())
//                return false;
//            else return true;

        StmtIterator sIter = theResource.listProperties(RDF.type);
        Set aSet = new HashSet();
        while (sIter.hasNext())
            aSet.add(sIter.nextStatement().getResource());
        sIter.close();
        aSet.removeAll(java.util.Arrays.asList(new Object[] {
                                               RDFS.Class, RDF.Property, RDFS.Resource, OWL.Class,
                                               OWL.AnnotationProperty, OWL.DatatypeProperty, OWL.ObjectProperty,
                                               OWL.InverseFunctionalProperty, OWL.ObjectProperty, OWL.OntologyProperty,
                                               OWL.Restriction, OWL.SymmetricProperty, OWL.TransitiveProperty, RDFS.Datatype,
                                               OWL.Ontology, OWL.FunctionalProperty
        }));

        if (!aSet.isEmpty())
            return true;
        else return false;

/*
        if (theResource.getModel().listStatements(new com.hp.hpl.jena.rdf.model.SimpleSelector() {
            public boolean selects(Statement stmt) {
                if (stmt.getSubject().equals(theResource) && stmt.getPredicate().equals(RDF.type) &&
                    (stmt.getObject() instanceof Resource &&

                    !stmt.getResource().getURI().startsWith(OWL.getURI()) &&
                    !stmt.getResource().getURI().startsWith(RDFS.getURI()) &&
                    !stmt.getResource().getURI().startsWith(RDF.getURI())

//                    (!stmt.getResource().equals(RDFS.Class) &&
//                     !stmt.getResource().equals(OWL.Class) &&
//                     !stmt.getResource().equals(RDF.Property) &&
//                     !stmt.getResource().equals(OWL.ObjectProperty) &&
//                     !stmt.getResource().equals(OWL.DatatypeProperty) &&
//                     !stmt.getResource().equals(OWL.FunctionalProperty) &&
//                     !stmt.getResource().equals(OWL.InverseFunctionalProperty) &&
//                     !stmt.getResource().equals(OWL.SymmetricProperty) &&
//                     !stmt.getResource().equals(OWL.TransitiveProperty) &&
//                     !stmt.getResource().equals(OWL.AnnotationProperty) &&
//                     !stmt.getResource().equals(OWL.Ontology) &&
//                     !stmt.getResource().equals(RDFS.Datatype) &&
//                     !stmt.getResource().equals(RDFS.Resource))
                    ))
                    return true;
                else return false;
            }
        }).hasNext())
        {
            return true;
        }
        else return false;
*/
    }

    /**
     * List all individuals in the given mode
     * @param theModel Model the model
     * @return Iterator an iterator over the list of individuals
     */
    public static Iterator listIndividuals(Model theModel)
    {
        HashSet list = new HashSet();

        ResIterator rIter = theModel.listSubjectsWithProperty(RDF.type);

int count = 0;
        while (rIter.hasNext())
        {
            final Resource aRes = rIter.nextResource();

long start = System.currentTimeMillis();
            boolean isIndividual = isIndividual(aRes);
long end = System.currentTimeMillis();
System.err.println("is ind time: "+(end-start));
count++;
            if (isIndividual && !list.contains(aRes))
                list.add(aRes);
        }
        rIter.close();
System.err.println("");
        return list.iterator();
    }

    /**
     * Takes an Object and returns it as a Jena RDFNode
     * @param theValue Object
     * @return RDFNode
     */
/*    public static RDFNode objectToRDFNode(Object theValue)
    {
        RDFNode object = null;

        if (theValue instanceof Resource)
            object = (Resource)theValue;
        else if (theValue instanceof RDFNode)
            object = (RDFNode)theValue;
        else object = ModelFactory.createDefaultModel().createLiteral(theValue);

        return object;
    }*/

    /**
     * Returns whether or not the first resource is a subclass of the second
     * @param theChild Resource the child
     * @param theParent Resource the parent
     * @return boolean true if the child is a subclass of the parent, false otherwise
     */
    public static boolean isSubClassOf(Resource theChild, Resource theParent)
    {
        return getAllSubClassesOf(theParent,false).contains(theChild);
    }

    /**
     * this assumes that your results have three vars per binding (s,p,o or subj,pred,obj, or variations)
     * and it will turn each of these bindings into a triple in the resulting model
     * if they vars are not named like this, you will get an empty model
     * @param theResults QueryResults the results
     * @return Model the model from the results
     */
   /* public static Model queryResultsToModel(QueryResults theResults)
    {
        Model aModel = ModelFactory.createDefaultModel();

        while (theResults.hasNext())
        {
            ResultBinding aResult = (ResultBinding)theResults.next();
            Resource subj = (Resource)getNodeFromResult(aResult, new String[] { "s", "subj", "subject" });
            Resource predRes = (Resource)getNodeFromResult(aResult, new String[] { "p", "pred", "predicate" });

            Property pred = aModel.getProperty(predRes.getURI());
            RDFNode obj = getNodeFromResult(aResult,new String[] {"o", "obj", "object"});

            if (subj == null || pred == null || obj == null)
                continue;
            else aModel.add(subj,pred,obj);
        }

        return aModel;
    }

    private static RDFNode getNodeFromResult(ResultBinding theBinding, String[] theNames)
    {
        RDFNode aNode = null;

        for (int i = 0; i < theNames.length; i++)
        {
            aNode = (RDFNode)theBinding.get(theNames[i]);
            if (aNode != null)
                break;
        }
        return aNode;
    }*/

    /**
     * Returns whether or not the instance is of the specified type
     * @param theInst Resource the instance
     * @param theType Resource the type
     * @return boolean true if the instance rdf:type theType, false otherwise
     */
    public static boolean isType(Resource theInst, Resource theType)
    {
        return isType(theInst, theType, theInst.getModel());
    }

    /**
     * Returns whether or not the instance is of the specified type
     * @param theInst Resource the inst
     * @param theType Resource the type
     * @param theModel Model the kb
     * @return boolean true if the instance rdf:type theType, false otherwise
     */
    public static boolean isType(Resource theInst, Resource theType, Model theModel)
    {
        Iterator aIter = getAllTypes(theInst,theModel,false).iterator();
        while (aIter.hasNext())
        {
            Resource aType = (Resource)aIter.next();
            if (aType.equals(theType))
                return true;
        }
        return false;
    }

    /**
     * Return all the resources in a rdf list (like seq, bag or alt) in a Set
     * @param theList Resource the head of the list
     * @return Set the resources in the list
     */
    public static Set collectResourcesFromRDFList(Resource theList)
    {
        LinkedHashSet aSet = new LinkedHashSet();

        Resource rest, first;
        rest = first = null;
        rest = theList;

        if (rest == null || !rest.hasProperty(RDF.first)) // empty list
        {
            if (theList.canAs(Seq.class))
                aSet = getFromSeq(theList);
            return aSet;
        }

        do {
            first = rest.getProperty(RDF.first).getResource();
            rest = rest.getProperty(RDF.rest).getResource();
            aSet.add(first);
        }
        while (!rest.equals(RDF.nil));

        return aSet;
    }

    private static LinkedHashSet getFromSeq(Resource theList)
    {
        LinkedHashSet aSet = new LinkedHashSet();
        Seq aSeq = (Seq)theList.as(Seq.class);
        NodeIterator nIter = aSeq.iterator();
        while (nIter.hasNext())
            aSet.add(nIter.next());
        nIter.close();
        return aSet;
    }

    /**
     * Returns a list of all classes in the model
     * @param theModel Model the model
     * @return Iterator an iterator over the list of all classes
     */
    public static Iterator listClasses(Model theModel) {
        Set classes = new HashSet();
        ResIterator rIter = theModel.listSubjects();
        while (rIter.hasNext())
        {
            Resource aRes = (Resource)rIter.next();

            if (aRes.isAnon() || JenaUtils.isLanguageTerm(aRes))
                continue;

            if (aRes.hasProperty(RDF.type,RDFS.Class) || aRes.hasProperty(RDF.type,OWL.Class))
                classes.add(aRes);

//            Set aTypes = JenaUtils.getTypes(aRes,false);
//            if (aTypes.contains(RDFS.Class) || aTypes.contains(OWL.Class)) {
//                classes.add(aRes);
//            }
        }
        rIter.close();

        return classes.iterator();
    }

    /**
     * List all properties in the model
     * @param theModel Model the model
     * @return Iterator an iterator over the list of all properties in the model
     */
    public static Iterator listProperties(Model theModel) {
        Set classes = new HashSet();
        ResIterator rIter = theModel.listSubjects();
        while (rIter.hasNext())
        {
            Resource aRes = (Resource)rIter.next();

            if (aRes.isAnon() || JenaUtils.isLanguageTerm(aRes))
                continue;

            Set aTypes = JenaUtils.getTypes(aRes,false);
            if (aTypes.contains(RDF.Property) || aTypes.contains(OWL.DatatypeProperty) || aTypes.contains(OWL.ObjectProperty)) {
                classes.add(aRes.as(Property.class));
            }
        }
        rIter.close();

        return classes.iterator();
    }
}
