package org.eaglei.datatools.etl.server;

/**
 * @author Sravan Kumar Cheriyala
 * This class has services related to repository
 */
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.StringWriter;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.User;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.jena.RESTRepositoryProvider;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIInstanceFactory;
import org.eaglei.security.Session;

import com.hp.hpl.jena.rdf.model.Model;

public class RDFtoRepoService {

	private static RDFtoRepoService instance = null;
	private DatatoolsConfiguration config;
	RESTRepositoryProvider repoProvider;
	JenaEIInstanceFactory instanceFactory;
	private static final String DEFAULT_CONFIGURATION = "etl-datatools.prop";
	private static final Log logger = LogFactory.getLog( RDFtoRepoService.class );
	private String token;
	private Session session;

	/* workflow transistions */
	private static final EIEntity defaultCreateTransition = EIEntity.create( EIURI.create( "http://eagle-i.org/ont/repo/1.0/WFT_0" ), "Default Create" );
	private static final EIEntity sentToCurationTransition = EIEntity.create( EIURI.create( "http://eagle-i.org/ont/repo/1.0/WFT_1" ), "Send to Curation" );
	private static final EIEntity PublishTransition = EIEntity.create( EIURI.create( "http://eagle-i.org/ont/repo/1.0/WFT_2" ), "Publish" );

	private RDFtoRepoService(String file, JenaEIInstanceFactory instanceFactory) throws IOException {
		this.instanceFactory = instanceFactory;
		config = getConfiguration( file );
		repoProvider = (RESTRepositoryProvider)new RESTRepositoryProvider( config, instanceFactory );
	}

	private RDFtoRepoService(JenaEIInstanceFactory instanceFactory) throws IOException, URISyntaxException {
		this.instanceFactory = instanceFactory;
		config = getConfiguration();
		repoProvider = (RESTRepositoryProvider)new RESTRepositoryProvider( config, instanceFactory );
	}

	public RepositoryProvider getRepositoryProvider() {
		return repoProvider;
	}

	public DatatoolsConfiguration getDatatoolsConfiguration() {
		return config;
	}

	public static RDFtoRepoService getInstance(String file, JenaEIInstanceFactory instanceFactory) throws IOException {
		if ( instance == null ) {
			instance = new RDFtoRepoService( file, instanceFactory );
		}
		return instance;
	}

	public static RDFtoRepoService getInstance(JenaEIInstanceFactory instanceFactory) throws Exception {
		if ( instance == null ) {
			instance = new RDFtoRepoService( instanceFactory );
		}
		return instance;
	}

	/**
	 * pushed the Model to Repo and promotes it
	 * 
	 * @param rdfModel
	 * @param session
	 * @param uri
	 * @return
	 * @throws Exception
	 */
	public boolean pushtoRepo(Model rdfModel, org.eaglei.security.Session session, String uri, String promoteParameter) throws Exception {

		StringWriter swriter = new StringWriter();
		rdfModel.write( swriter, "RDF/XML" );
		repoProvider.createInstance( session, swriter.toString(), uri, repoProvider.DEFAULT_WORKSPACE_ENTITY );
		token = repoProvider.updateInstance( session, swriter.toString(), uri, null );
		repoProvider.updateInstance( session, swriter.toString(), uri, token );
		if ( promoteParameter.equalsIgnoreCase( "publish" ) ) {
			repoProvider.transition( session, Arrays.asList( EIURI.create( uri ) ), sentToCurationTransition );
			repoProvider.claim( session, Arrays.asList( EIURI.create( uri ) ) );
			repoProvider.transition( session, Arrays.asList( EIURI.create( uri ) ), PublishTransition );
		} else if ( promoteParameter.equalsIgnoreCase( "curation" ) ) {
			repoProvider.transition( session, Arrays.asList( EIURI.create( uri ) ), defaultCreateTransition );
			repoProvider.transition( session, Arrays.asList( EIURI.create( uri ) ), sentToCurationTransition );
			repoProvider.release( session, Arrays.asList( EIURI.create( uri ) ) );
		} else if ( promoteParameter.equalsIgnoreCase( "draft" ) ) {
			repoProvider.release( session, Arrays.asList( EIURI.create( uri ) ) );
		}

		return true;
	}

	/**
	 * gets the new instance from Repo
	 * 
	 * @param countofInstances
	 * @return
	 * @throws Exception
	 */
	public List<EIURI> getNewInstancesFromRepo(org.eaglei.security.Session session, int countofInstances) {
		try {
			return repoProvider.getNewInstanceID( session, countofInstances );
		} catch (Exception e) {
			throw new RuntimeException( e );
		}

	}

	public RepositoryProvider setPropertyFilePathAndgetNewRepoProvider(final String configFilePath) throws IOException {
		final File configFile = new File( configFilePath );
		logger.debug( "loading configuration file: " + configFile );
		config = new DatatoolsConfiguration( configFile );
		repoProvider = new RESTRepositoryProvider( config, instanceFactory );
		return repoProvider;
	}

	/**
	 * get the configuration object by the String as path to file
	 * 
	 * @param DEFAULT_CONFIGURATION
	 * @return
	 * @throws Exception
	 */
	private DatatoolsConfiguration getConfiguration(final String DEFAULT_CONFIGURATION) {
		try {
			final File configFile = new File( DEFAULT_CONFIGURATION );
			logger.debug( "loading configuration file: " + configFile );
			return new DatatoolsConfiguration( configFile );
		} catch (Exception e) {
			logger.error( "Error loading configuration from " + DEFAULT_CONFIGURATION + " " + e );
			throw new RuntimeException( e );
		}
	}

	/**
	 * get the Configuration of default ETL properties file
	 * 
	 * @return
	 * @throws IOException
	 * @throws URISyntaxException
	 * @throws Exception
	 */
	private DatatoolsConfiguration getConfiguration() throws IOException, URISyntaxException {
		final URL configURL = this.getClass().getClassLoader().getResource( DEFAULT_CONFIGURATION );
		if ( configURL == null ) {
			logger.error( "Could not locate " + DEFAULT_CONFIGURATION + " on classpath" );
			throw new FileNotFoundException( "Could not locate " + DEFAULT_CONFIGURATION + " on classpath" );
		}
		try {
			final File configFile = new File( configURL.toURI() );
			logger.debug( "loading configuration file: " + configFile );
			return new DatatoolsConfiguration( configFile );
		} catch (RuntimeException e) {
			logger.error( "Error loading configuration from " + DEFAULT_CONFIGURATION + " " + e );
			throw new RuntimeException( e );
		}
	}

	public Session login(String repoPath) throws Exception {
		Map<String, String> credentialMap = RowConfiguration.getCredentialsForRepo( repoPath );
		User strAry = repoProvider.login( credentialMap.get( "user" ), credentialMap.get( "password" ) );
		if ( strAry == null ) {
			logger.error( "Could not authenticate please check username,password and Repository URL...Exiting System" );
			System.exit( 2 );
		}
		setSession( strAry.getSession() );
		return getSession();
	}

	public Session getSession() {
		return session;
	}

	public void setSession(Session session) {
		this.session = session;
	}

}
