/**
 * @author Sravan Cheriyala
 * 
 * This is the core File for ETL ,This file takes Excel Files and Map Files and converts data of Excel into RDF.
 * 
 */
package org.eaglei.datatools.etl.server;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.ParseException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.table.DefaultTableModel;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.eaglei.datatools.etl.server.ExcelAbstractions.ExcelTab;
import org.eaglei.datatools.etl.server.ExcelFileParser.ExcelFileMetaData;
import org.eaglei.datatools.etl.utils.Configure;
import org.eaglei.datatools.jena.RESTRepositoryProvider;
import org.eaglei.datatools.model.AnnotationFormModel;
import org.eaglei.datatools.provider.EIDataToolsProviderException;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIInstanceFactory;
import org.eaglei.security.Session;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;

public class RdfMaker {

	private static final long serialVersionUID = 1L;
	private static Model prefixModel = null;
	private Model mapModel = null;
	private int count = 0;

	private static org.apache.log4j.Logger logger = Logger.getLogger( RdfMaker.class );
	private static String mapDirectoryPath;
	private static String currentlyProcessingFile;

	private static RDFtoRepoService rdftorepo;
	public static Map<String, List<Resource>> stringToResourceMap;
	private String[] tabOrder;
	private RepositoryProvider provider;
	private JenaEIInstanceFactory instanceFactory;

	private Session session;
	private PrintWriter outputPrinter;
	private PrintWriter errorPrinter;
	private AnnotationFormModel anntModel;
	private Map<String, String> dcCreatorMap;
	private List<EIURI> eiurilst = null;
	private Map<String, Integer> numberOfRowsReadFromTabs;
	EIFileParser.FileMetaData fileMetaData;

	public int getCount() {
		return count;
	}

	public Session getSession() {
		return session;
	}

	public final void setSession(Session session) {
		this.session = session;
	}

	public RdfMaker(AnnotationFormModel anntModel, JenaEIInstanceFactory instanceFactory) throws Exception {
		super();
		dcCreatorMap = RowConfiguration.getdcCreatorFromRepoFile( ETLEntryPoint.getRepoConfigFile() );
		this.instanceFactory = instanceFactory;
		this.anntModel = anntModel;
		// initialize predefined prefix definition
		int prefixCount = Configure.prefixArray.length;
		HashMap predefinedPrefixes = new HashMap();
		for (int row = 0; row < prefixCount; row++) {
			predefinedPrefixes.put( Configure.prefixArray[row][0], Configure.prefixArray[row][1] );
		}
		prefixModel = ModelFactory.createDefaultModel();
		prefixModel.setNsPrefixes( predefinedPrefixes );
		int metaCount = Configure.metaArray.length;
		// getting RepositoryProvider instance
		try {
			rdftorepo = RDFtoRepoService.getInstance( ETLEntryPoint.getRepoConfigFile(), instanceFactory );
			provider = rdftorepo.getRepositoryProvider();
		} catch (FileNotFoundException fe) {
			logger.error( "Repository information file not found" );
			logger.error( fe );
			throw fe;
		}
		/*
		 * getting an instance of rfdtorepo for repository services ,which will be used through out the program
		 */
		setSession( rdftorepo.login( ETLEntryPoint.getRepoConfigFile() ) );
		RdfMaker.mapDirectoryPath = ETLEntryPoint.getMaps();
	}

	/**
	 * Get all files from the specified directory and call generateRDFGraph(which inturn generates the RDF and pushes into Repo)
	 * 
	 * @throws Exception
	 * 
	 */
	public boolean rdfToRepoByExcelStream(final String excelFolder, final String mapFolder) throws Exception {
		Map<String, InputStream> mapIns = new HashMap<String, InputStream>();
		/* excel files */
		File excelDirectory = new File( excelFolder );
		if ( !excelDirectory.isDirectory() ) {
			logger.error( excelFolder + " is not a directory...exiting" );
			System.exit( 2 );
		}

		/* loop through the directories and files */
		String[] lstFiles = excelDirectory.list();

		for (String str : lstFiles) {
			try {
				/* if files are not excel file then ignore */
				if ( !checkForExcelFiles( mapFolder, excelFolder, str ) ) {
					continue;
				}

				InputStream is = getInputStreamOfExcelFile( excelFolder, str );
				if ( is == null ) {
					continue;
				}

				/*
				 * send inputstream of excelfile to ExcelToTableModel() method to convert into csv data structure , ExcelToTableModel(Inputstream) returns Map<String,DefaulTableModel> ,String is the tabname in excel file and DefaultTableModel is the
				 * CSV data container
				 */
				Map<String, ExcelTab> defaultModelMap = ExcelToCSV( is );
				if ( defaultModelMap == null ) {
					continue;
				}
				File mapdirectory = null;
				/*
				 * mapDirecotry variable is populated in ExcelToTableModel method ,-m option takes directory of maps as input,however that directory information is not enough ,location passed to -m is only parent directory of sub directories like
				 * annotation_v1,reagnet_v1,organisms_v1 etc , we can only know which sub-directory to use by reading excel file . "VERSION" column in excel file tells which map directory we need to get maps from.So mapDirectory is populated in
				 * ExcelToTableModel method
				 */

				/*
				 * once we get map directory we can read map files from that directory
				 */
				mapdirectory = new File( RdfMaker.mapDirectoryPath );
				String[] lstMapFiles = mapdirectory.list();

				for (String mapstr : lstMapFiles) {
					if ( mapstr.contains( ".svn" ) ) {
						continue;
					}

					/*
					 * make a map of inputstream of all map files with the filename of mapfile as key NOTE: here filename of map file should be same as tab name in excel file
					 */
					FileInputStream fis = new FileInputStream( RdfMaker.mapDirectoryPath + "/" + mapstr );
					mapIns.put( mapstr.replace( ".rdf", "" ), fis );

				}

				/*
				 * this is where the rdf is generated . generateRDFGraph takes two arguments .1st argument is Map<String,DefaultTableModel> which is returned ExcelToTableModel() here String in Map<String,DefualtTableModel> is name of tab in
				 * excelsheet and 2nd argument is Map<String,InputStream> here String is file name of map file which is same as tab name in excel sheet
				 */
				generateRDFGraph( defaultModelMap, mapIns );

				/*
				 * close this outputprinter after the RDF is generated for file ,new outputstream is opened for new file in next loop
				 */
				if ( outputPrinter != null ) {
					outputPrinter.close();
				}
				if ( errorPrinter != null ) {
					errorPrinter.close();
				}

				/* FIXME:not sure why the below line is */
				RdfMaker.mapDirectoryPath = ETLEntryPoint.getMaps();

			} catch (Exception e) {
				logger.error( e );
				e.printStackTrace();
				throw e;
			}
		}
		return true;
	}

	private boolean checkForExcelFiles(final String mapFolder, final String excelFolder, final String fileName) throws Exception {
		if ( !fileName.contains( ".xlsx" ) && !fileName.contains( ".xls" ) ) {
			if ( fileName.contains( ".svn" ) ) {
				return false;
			}
			File strFIle = new File( excelFolder + "/" + fileName );
			if ( strFIle.isDirectory() ) {
				rdfToRepoByExcelStream( excelFolder + "/" + fileName, mapFolder );
			} else {
				logger.info( fileName + "is not Excel file ...Ignoring" );
				return false;
			}
		}
		return true;
	}

	public InputStream getInputStreamOfExcelFile(final String excelFolder, final String excelFile) throws FileNotFoundException {
		/* get the inputstream of the file once we found excel file */
		File dirfile = new File( excelFolder + "/" + excelFile );
		if ( dirfile.isDirectory() ) {
			logger.info( excelFolder + "/" + excelFile + "is directory ...Ignoring" );
			return null;
			// continue;
		} else {
			logger.info( "processing file :" + excelFolder + "/" + excelFile );
			currentlyProcessingFile = excelFile;
			/*
			 * if -o parameter is passed then open the outputstream to that file to write the generated RDF as output
			 */
			if ( ETLEntryPoint.getOutputdirectory() != null ) {
				FileOutputStream fvec = new FileOutputStream( ETLEntryPoint.getOutputdirectory() + "/" + excelFile + ".rdf" );
				outputPrinter = new PrintWriter( fvec );
			}

			/*
			 * if e parameter is passed then open the outputstream to that file to write the generated RDF as error
			 */
			if ( ETLEntryPoint.getErrorFileDirecotry() != null ) {
				FileOutputStream fvec = new FileOutputStream( ETLEntryPoint.getErrorFileDirecotry() + "/" + excelFile + ".rdf" );
				errorPrinter = new PrintWriter( fvec );
			}

		}

		InputStream is = new BufferedInputStream( new FileInputStream( excelFolder + "/" + excelFile ) );
		return is;

	}

	/**
	 * converts Excel into CSV TableModel
	 * 
	 * @param excelInputStream
	 * @return
	 * @throws ConfigurationException
	 * @throws IOException
	 * @throws ParseException
	 * @throws Exception
	 */
	public Map<String, ExcelTab> ExcelToCSV(InputStream excelInputStream) throws ConfigurationException, IOException, ParseException {

		ExcelFileParser fileParser = new ExcelFileParser( ETLEntryPoint.getMaps() );

		Map<String, ExcelTab> mapToReturn = fileParser.toCSV( excelInputStream );
		mapDirectoryPath = fileParser.getMapDirecotryPath();
		tabOrder = fileParser.getTabOrder();
		numberOfRowsReadFromTabs = fileParser.getNumberOfRowsReadFromTabs();
		fileMetaData = fileParser.getMetaData();

		return mapToReturn;

	}

	/**
	 * Generates RDGGraph by tableModel and mapStream and pushes into Repo
	 * 
	 * @throws IOException
	 * 
	 * @throws Exception
	 * 
	 * @throws Exception
	 */
	public Map<String, Model> generateRDFGraph(Map<String, ExcelTab> tableModelMap, Map<String, InputStream> mapStreamMap) throws IOException {

		Map<String, Model> retMap = new HashMap<String, Model>();
		StringWriter swriter = null;
		for (String keyStr : tabOrder) {
			if ( !keyStr.equalsIgnoreCase( "organisms" ) ) {
				continue;
			}
			logger.info( "Processing " + keyStr );
			ExcelTab excelTab = tableModelMap.get( keyStr.toUpperCase() );
			DefaultTableModel tableModel = excelTab.getTabData();
			if ( tableModel == null ) {
				logger.info( keyStr + " havent got anyrow skipping..." );
				continue;
			}
			InputStream mapStream = mapStreamMap.get( keyStr.toLowerCase() );
			if ( mapStream == null ) {
				continue;
			}
			/* gets the base namspace from properties file */
			String base = rdftorepo.getDatatoolsConfiguration().getDatatoolsRepositoryNamespace();
			String syntax = "RDF/XML";
			/*
			 * RDF123 Expression Engine uses columns numbers to resolve ,map and transform data of excel but At eagle-I they like to create maps with column names .While parsing Excel file we have prepared the MetaData which contains map of Column
			 * name and column number.replaceColumnNamesWithColumnNumbers(InputStream) replaces column names with numbers
			 */
			ExcelFileMetaData excelFileMetaData = (ExcelFileMetaData)fileMetaData;
			InputStream mapStreamWithNumber = RdfMakerUtil.replaceColumnNamesWithColumnNumbers( mapStream, excelFileMetaData.getSheetMetaData( keyStr ).getMapofColumnNameAndNumber() );

			mapModel = ModelFactory.createDefaultModel();
			mapModel.read( mapStreamWithNumber, null, syntax );
			mapModel.removeNsPrefix( "" );
			boolean hasRowHead = true;
			int startRow = 0;
			int endRow = tableModel.getRowCount();
			count = 0;
			for (int row = startRow; row <= endRow; row++) {
				final long startTime = System.nanoTime();
				final long endTime;
				try {

					// list the statements in the graph
					int columnCount = tableModel.getColumnCount();
					int columnsFilled = 0;
					/* if blank row skip it */
					columnsFilled = checkForBlankRow( tableModel, row, columnCount, columnsFilled );
					if ( !( columnsFilled > 0 ) ) {
						continue;
					}
					Set<String> subjectSet = new HashSet<String>();
					StmtIterator subIter = mapModel.listStatements();
					/* process the dependent nodes first,that is our main newly created instance depend on few node ,that should be processed first ,please see the actvity diagram for further understading on flow */
					getMainNodeSubjectSet( subjectSet, subIter );
					stringToResourceMap = new HashMap<String, List<Resource>>();
					MapInterpreter mapInterpretor = new MapInterpreter( mapModel, excelTab.getTabData(), anntModel );
					Statement[] stmtAry = mapInterpretor.interpret( row );

					if ( logger.isDebugEnabled() ) {
						logger.debug( "Map of String Expression to URI" );
						for (Map.Entry<String, List<Resource>> entry : stringToResourceMap.entrySet()) {
							logger.debug( "the processed expressions are" + entry.getKey() + ":" + entry.getValue().toString() );
						}
					}

					/* make Jena RDF Model from the statements returned by mapInterpretor */
					Model rdfModel = null;
					rdfModel = ModelFactory.createDefaultModel();
					rdfModel.setNsPrefixes( mapModel.getNsPrefixMap() );
					rdfModel.add( stmtAry );

					// if ( logger.isDebugEnabled() ) {
					logger.debug( "the returned statements are" );
					for (Statement stmt : stmtAry) {
						logger.debug( stmt.toString() );
					}
					// }

					/* repository wont allow us to insert whole rdf model(with primary and secondary resources) at once ,so first we have to insert secondary instances after that insert primary resources */

					/* create Model to put primary resource statements */
					Model mainRdfModel = null;
					mainRdfModel = ModelFactory.createDefaultModel();
					mainRdfModel.setNsPrefixes( mapModel.getNsPrefixMap() );

					/* get all resources URI's(primary and secondary) form the above created rdfModel */
					List<String> lstResources = RdfMakerUtil.getReourceURIFromModel( rdfModel );

					uploadSecondaryNodes( mapInterpretor, stmtAry, mainRdfModel, stringToResourceMap );

					if ( lstResources.size() == 1 ) {
						mainRdfModel = rdfModel;
					}
					/* add the sourcefile triple to primary model */
					addSourcefileTriple( mainRdfModel, mapInterpretor.getMainSubject() );

					/* for log purpose */
					logger.info( "pushing " + keyStr + " instance into repo" );
					swriter = new StringWriter();
					mainRdfModel.write( swriter, "RDF/XML" );
					logger.debug( swriter );

					/* if program supplied with output directory to print rdf output then doing it */
					if ( ETLEntryPoint.getOutputdirectory() != null ) {
						outputPrinter.println( swriter.toString() );
					}

					/* if the instance is laboratory instance and if it already exsists in repository then don't insert it again,checkMainNodeAndPushToRepo does that check */
					checkMainNodeAndPushToRepo( mainRdfModel, EIURI.create( mapInterpretor.getMainSubject().toString() ), swriter );
					logger.info( keyStr + " instance pushed into repo" );
					count++;

				}

				catch (Exception e) {
					logger.error( "Exception occured in generateRDFGraph() the stack trace is " );
					logger.error( Arrays.toString( e.getStackTrace() ) );
					logger.error( "............................skipping.............................." );
					/* if program supplied with output directory to print rdf output then doing it */
					if ( ETLEntryPoint.getErrorFileDirecotry() != null && swriter != null ) {
						errorPrinter.println( "       " );
						errorPrinter.println( swriter.toString() );
					}
					continue;
				}
				endTime = System.nanoTime();
			}
			if ( count % 50 == 0 ) {
				System.gc();
			}

		}
		return retMap;
	}

	private int uploadSecondaryNodes(MapInterpreter mapInterpretor, Statement[] stmtAry, Model mainRdfModel, Map<String, List<Resource>> resourcesMap) throws Exception {
		int i = 0;
		StringWriter swriter = new StringWriter();
		List<Resource> embeddedResourceList = stringToResourceMap.get( MapInterpreter.EmbeddedClassExpression.EMBEDDED_CLASS_EXPRESSION_MATCHER );
		for (Map.Entry<String, List<Resource>> entry : stringToResourceMap.entrySet()) {

			/* skip the embedded resource cause we are trying to push secondary resource to repository */
			if ( entry.getKey().equals( MapInterpreter.EmbeddedClassExpression.EMBEDDED_CLASS_EXPRESSION_MATCHER ) || entry.getKey().contains( OntologyExpression.ONTOLOGY_EXPRESSION_MATCHER ) ) {
				continue;
			}
			List<Resource> lstResources = entry.getValue();
			/* loop through the resources URI's list */
			for (Resource resourceURI : lstResources) {
				try {
					/* create Jena Model for secondary resource */
					Model subRdfModel = ModelFactory.createDefaultModel();
					subRdfModel.setNsPrefixes( mapModel.getNsPrefixMap() );
					/* loop through statements array we got from mapInterpretor to check if its a secondary resource or primary ,if its secondary then add it to secondary Model(subRdfModel) otherwise add it to primary Model (mainRdfModel) */
					for (Statement stmt : stmtAry) {
						if ( stmt.getSubject().toString().equals( resourceURI.getURI() ) ) {
							subRdfModel.add( stmt );
						} else if ( i < 1 && embeddedResourceList != null && ( stmt.getSubject().equals( mapInterpretor.getMainSubject() ) || embeddedResourceList.contains( stmt.getSubject() ) ) ) {
							mainRdfModel.add( stmt );
						}
					}

					/* is subnode is empty then continue onto to next one */
					if ( subRdfModel.isEmpty() ) {
						continue;
					}

					/* add the sourcefile triple to secondary model */
					addSourcefileTriple( subRdfModel, subRdfModel.createResource( resourceURI.getURI() ) );
					/* push the secondary rdf model we just made to repository */
					/* for log purpose */
					logger.info( "pushing  sub-node instance into repo" );
					subRdfModel.write( swriter, "RDF/XML" );
					logger.debug( swriter );
					rdftorepo.pushtoRepo( subRdfModel, session, resourceURI.getURI(), ETLEntryPoint.getPromoteParameter() );
					i++;
				} catch (Exception e) {
					logger.error( "Exception occured in uploadSecondaryNodes() the stack trace is " );
					logger.error( Arrays.toString( e.getStackTrace() ) );
					logger.error( "............................skipping.............................." );
					/* if program supplied with output directory to print rdf output then doing it */
					if ( ETLEntryPoint.getErrorFileDirecotry() != null && swriter != null ) {
						errorPrinter.println( "       " );
						errorPrinter.println( swriter.toString() );
					}
					throw e;
				}

			}

		}

		return i;
	}

	private void getMainNodeSubjectSet(Set<String> subjectSet, StmtIterator subIter) {
		while ( subIter.hasNext() ) {
			Statement stmtMap = subIter.nextStatement();
			Resource subjectMap = stmtMap.getSubject();
			if ( !subjectMap.getURI().equals( "Ex:e+$0" ) )
				subjectSet.add( subjectMap.getURI() );
		}
	}

	private int checkForBlankRow(DefaultTableModel tableModel, int row, int columnCount, int columnsFilled) {

		try {
			for (int column = 0; column <= columnCount - 1; column++) {
				if ( tableModel.getValueAt( row, column ) != null ) {
					if ( !tableModel.getValueAt( row, column ).equals( "" ) ) {
						columnsFilled++;
						// break;
					}
				}
			}
			return columnsFilled;

		} catch (ArrayIndexOutOfBoundsException e) {
			return 0;
		}
	}

	private void checkMainNodeAndPushToRepo(Model rdfModel, EIURI eiuri, StringWriter swriter) throws Exception {
		try {
			String query = ETLSPARQLQueryUtil.getEIResourcesByOnlyTypeANDRdfsLabel( rdfModel );
			String instance = provider.query( this.session, query.replace( "\n", " " ).replace( "\r", " " ) );
			String[] instanceURI = ETLSPARQLQueryUtil.getInstanceURI( instance );
			if ( query != null && ( instanceURI.length == 0 || !( query.contains( "ERO_0000001" ) || query.contains( "ERO_0000002" ) ) ) ) {
				rdftorepo.pushtoRepo( rdfModel, this.getSession(), eiuri.toString(), ETLEntryPoint.getPromoteParameter() );
			}
		} catch (EIDataToolsProviderException ex) {
			logger.error( ex.getMessage() );
			logger.error( "the offending instnace is " + swriter.toString() );
			logger.info( "skiping..." );
		}
	}

	private void addSourcefileTriple(Model rdfModel, Resource subject) {
		Property filePredicate = rdfModel.createProperty( "http://eagle-i.org/ont/datatools/1.0/source_file" );
		RDFNode fileObject = rdfModel.createLiteral( currentlyProcessingFile );
		rdfModel.add( subject, filePredicate, fileObject );
	}

	/**
	 * converts Excel into CSV TableModel
	 * 
	 * @param excelInputStream
	 * @return
	 * @throws ConfigurationException
	 * @throws IOException
	 * @throws ParseException
	 * @throws Exception
	 */
	/*
	 * public Map<String, ExcelTab> ExcelToTableModel(InputStream excelInputStream) throws ConfigurationException, IOException, ParseException {
	 * 
	 * ExcelFileParser excelParser = new ExcelFileParser( maps );
	 * 
	 * Map<String, ExcelTab> mapToReturn = excelParser.toCSV( excelInputStream ); mapDirectory = excelParser.getMapDirecotryPath(); tabOrder = excelParser.getTabOrder(); return mapToReturn;
	 * 
	 * }
	 */

	public void updateInstanceIfAdditionalDatafound(String instanceID, Model modelFromExcel, String subjectExpression) throws Exception {
		if ( modelFromExcel == null ) {
			return;
		}
		if ( subjectExpression == null ) {
			return;
		}
		if ( instanceID == null ) {
			return;
		}
		boolean foundNewResource = false;
		Resource subjectMap = modelFromExcel.createResource( instanceID );
		EIInstance instance = provider.getInstance( session, EIURI.create( instanceID ) );
		if ( instance == null ) {
			return;
		}
		Model modelFromRepo = instanceFactory.convertToJenaModel( instance );
		Model newModelofExtraData = ModelFactory.createDefaultModel();
		StmtIterator stmtIterator = modelFromExcel.listStatements();
		while ( stmtIterator.hasNext() ) {
			Statement stmtMap = stmtIterator.nextStatement();
			Property predicateMap = stmtMap.getPredicate();
			RDFNode objectMap = stmtMap.getObject();
			if ( !modelFromRepo.contains( subjectMap, predicateMap, objectMap ) && !predicateMap.toString().contains( "ERO_0000024" ) && !instance.getReadOnlyResourceProperties().toString().contains( objectMap.toString() ) ) {
				newModelofExtraData.add( subjectMap, predicateMap, objectMap );
				foundNewResource = true;
			}
		}
		if ( foundNewResource ) {
			StmtIterator stmtFromExcelIterator = modelFromExcel.listStatements();
			while ( stmtFromExcelIterator.hasNext() ) {
				Statement stmtMap = stmtFromExcelIterator.nextStatement();
				Property predicateMap = stmtMap.getPredicate();
				RDFNode objectMap = stmtMap.getObject();
				if ( !newModelofExtraData.contains( subjectMap, predicateMap, objectMap ) && !predicateMap.toString().contains( "ERO_0000024" ) ) {
					newModelofExtraData.add( subjectMap, predicateMap, objectMap );
				}
			}
			StmtIterator stmtFromRepoIterator = modelFromRepo.listStatements();
			while ( stmtFromRepoIterator.hasNext() ) {
				Statement stmtMap = stmtFromRepoIterator.nextStatement();
				Property predicateMap = stmtMap.getPredicate();
				RDFNode objectMap = stmtMap.getObject();
				if ( predicateMap.toString().contains( "ERO_0000024" ) ) {
					newModelofExtraData.add( subjectMap, predicateMap, objectMap );
				}
			}
			try {
				StringWriter swriter = new StringWriter();
				newModelofExtraData.write( swriter, "RDF/XML" );
				logger.info( "updating instance " + subjectMap );
				RESTRepositoryProvider restProvider = (RESTRepositoryProvider)provider;
				String token = restProvider.updateInstance( session, swriter.toString(), instanceID, null );
				restProvider.updateInstance( session, swriter.toString(), instanceID, token );
			} catch (EIDataToolsProviderException ei) {
				ei.printStackTrace();
			}
		}
	}

	/**
	 * @return
	 */
	public static RDFtoRepoService getRdftorepo() {
		// TODO Auto-generated method stub
		return rdftorepo;
	}

}
