/**
 * 
 */
package org.eaglei.datatools.etl.server;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.log4j.Logger;
import org.eaglei.datatools.etl.server.exceptions.BadInputException;
import org.eaglei.datatools.etl.server.exceptions.ExceptionWhileMakingRepoSerivce;
import org.eaglei.datatools.etl.server.exceptions.ExtractorException;
import org.eaglei.datatools.etl.server.exceptions.LoaderException;
import org.eaglei.datatools.etl.server.exceptions.NotADirectoryException;
import org.eaglei.datatools.etl.server.exceptions.UnsupportedFileException;
import org.eaglei.datatools.etl.server.extractor.Extractor;
import org.eaglei.datatools.etl.server.extractor.ExtractorFactory;
import org.eaglei.datatools.etl.server.extractor.FileIterator;
import org.eaglei.datatools.etl.server.extractor.parsers.Data;
import org.eaglei.datatools.etl.server.extractor.parsers.MapFileConfiguration;
import org.eaglei.datatools.etl.server.extractor.parsers.RowIterator;
import org.eaglei.datatools.etl.server.loader.Loader;
import org.eaglei.datatools.etl.server.loader.LoaderImpl;
import org.eaglei.datatools.etl.server.transformer.Transformer;
import org.eaglei.datatools.etl.server.transformer.TransformerImpl;
import org.eaglei.datatools.model.AnnotationFormModel;
import org.eaglei.model.jena.JenaEIInstanceFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.xml.sax.SAXException;

import com.hp.hpl.jena.rdf.model.Statement;

/**
 * @author Sravan Cheriyala
 * 
 */

public class ETLEntryPoint {
	private static org.apache.log4j.Logger errLogger;
	private static org.apache.log4j.Logger logger = Logger.getLogger( ETLEntryPoint.class );
	private static ETLInputs etlInputs;
	private static ETLEntryPoint entryPoint;
	private AnnotationFormModel anntModel;
	private JenaEIInstanceFactory jenaInstanceFactory;
	private RepoService repoService;
	private String currentlyProcessingFile;
	private List<FileStatistics> fileStatisticsList;
	private Loader loader;
	private Transformer transformer;

	public static void main(final String args[]) {

		try {
			if ( args.length > 0 ) {
				String logFile = "";
				String outputDirectoryFile = "";
				String errorFileDirecotry = "";
				String repoConfigFileLocation = "";
				String mapsParentDirectoryPath = "";
				String excelFilesDirectoryPath = "";
				Loader.Promote promoteParameter = Loader.Promote.DRAFT;

				for (int i = 0; i < args.length - 1; i++) {
					if ( args[i].equals( "-d" ) ) {
						i = i + 1;
						excelFilesDirectoryPath = args[i];
					} else if ( args[i].equals( "-o" ) ) {
						i = i + 1;
						outputDirectoryFile = args[i];
					} else if ( args[i].equals( "-r" ) ) {
						i = i + 1;
						repoConfigFileLocation = args[i];

					} else if ( args[i].equals( "-l" ) ) {
						i = i + 1;
						logFile = args[i];
					} else if ( args[i].equals( "-m" ) ) {
						i = i + 1;
						mapsParentDirectoryPath = args[i];
					} else if ( args[i].equals( "-p" ) ) {
						i = i + 1;
						promoteParameter = Loader.Promote.getPromote( args[i] );
					} else if ( args[i].equals( "-e" ) ) {
						i = i + 1;
						errorFileDirecotry = args[i];
					}

				}

				etlInputs = new ETLInputs( excelFilesDirectoryPath, mapsParentDirectoryPath, repoConfigFileLocation, promoteParameter, logFile, outputDirectoryFile, errorFileDirecotry );
				ETLEntryPoint.setEtlInputs( etlInputs );
				entryPoint = ETLEntryPoint.getInstance();
				entryPoint.doETL( etlInputs.getExcelFilesDirectoryPath() );
				logger.info( "finished ETL." );
				entryPoint.printStatistics();
				logger.info( "---------------------done-------------------------" );

			}
		} catch (IOException e) {
			logger.error( "IO error while ETLing " );
			logger.error( e.getMessage() );
			System.exit( 2 );
		} catch (BadInputException bie) {
			logger.error( "Exiting System" );
			logger.error( bie.getMessage() );
			System.exit( 2 );
		}

	}

	public static ETLEntryPoint getInstance() {
		if ( entryPoint == null ) {
			return entryPoint = new ETLEntryPoint();
		} else {
			return entryPoint;
		}

	}

	private ETLEntryPoint() {
		try {
			// create and configure spring beans
			final ApplicationContext context = new ClassPathXmlApplicationContext( new String[] { "datatools-model-config.xml" } );
			// retrieve configured instance
			anntModel = context.getBean( AnnotationFormModel.class );
			jenaInstanceFactory = context.getBean( JenaEIInstanceFactory.class );
			fileStatisticsList = new ArrayList<FileStatistics>();
			repoService = makeRepoService( etlInputs.getRepoConfigFileLocation() );
			loader = new LoaderImpl( repoService, jenaInstanceFactory, anntModel );

			LoggerFactory.intializeLoggers( etlInputs.getErrorFileDirecotry(), etlInputs.getOutputDirectoryFile() );
			errLogger = LoggerFactory.errorFileLogger;

		} catch (ExceptionWhileMakingRepoSerivce e) {
			logger.error( "Cannot make repository serive class" );
			logger.error( e.getMessage() );
			System.exit( 2 );
		}

	}

	public void doETL(String direcotryPath) throws NotADirectoryException {
		final FileIterator fileIter = new FileIterator( direcotryPath );
		while ( fileIter.hasNext() ) {
			File file = fileIter.next();
			currentlyProcessingFile = file.getName();
			FileStatistics statistics = new FileStatistics( file.getName() );
			LoggerFactory.appendErrorFileToLogger( etlInputs.getErrorFileDirecotry() + "/" + file.getName().replaceAll( "\\..*$", ".txt" ).replaceAll( ".xls", ".txt" ) );
			LoggerFactory.appendOutputFileToLogger( etlInputs.getOutputDirectoryFile() + "/" + file.getName().replaceAll( "\\..*$", ".txt" ).replaceAll( ".xls", ".txt" ) );
			try {
				Extractor extractor = getExtracter( file );
				doTransform( extractor, statistics, etlInputs.getMapsParentDirectoryPath() );
			} catch (UnsupportedFileException e) {
				logerror( e, "this file format is currently unsupported " + file.getName() );
				continue;
			} catch (ExtractorException e) {
				logerror( e, "error while extracting file " + file.getName() );
				continue;
			} catch (ParserConfigurationException e) {
				logerror( e, "parse configuration error while parsing file" + file.getName() );
				continue;
			} catch (SAXException e) {
				logerror( e, "Sax parserException while parsing xml file " + file.getName() );
				continue;
			} catch (ConfigurationException e) {
				logerror( e, "error reading configuration information file for this file " + file.getName() );
				continue;
			} catch (ParseException e) {
				logerror( e, "Parsing error while  reading  file " + file.getName() );
				continue;
			} catch (IOException e) {
				logerror( e, "IO error reading  file " + file.getName() );
				continue;
			}
		}
	}

	private void logerror(Exception e, String msg) {
		errLogger.error( msg );
		errLogger.error( e.getMessage() );
		errLogger.error( "continuing onto next file" );
	}

	public Extractor getExtracter(File excelFile) throws ParseException, ParserConfigurationException, SAXException, IOException, UnsupportedFileException {
		Extractor extractor = ExtractorFactory.getExtractor( excelFile );
		return extractor;
	}

	public void doTransform(Extractor extractor, FileStatistics statistics, String mapsParentDirectoryPath) throws IOException, ConfigurationException, ParseException, ExtractorException {
		Map<String, File> mapFiles;
		Map<String, Data> mapOfData;
		String[] tabOrderToProcess;

		String formVersion = extractor.getFormVersion();
		File mapDirectory = extractor.getMapDirectory( formVersion, new File( mapsParentDirectoryPath ) );
		mapFiles = extractor.getMapFiles( mapDirectory );
		mapOfData = extractor.getData( mapDirectory );
		tabOrderToProcess = MapFileConfiguration.getOrderofMapFiles( mapDirectory.getPath() );

		for (String tabName : tabOrderToProcess) {
			Data data = mapOfData.get( tabName.toUpperCase() );
			if ( data == null ) {
				continue;
			}
			RowIterator rowIterator = new RowIterator( data );
			File mapFile = mapFiles.get( tabName.toUpperCase() );
			transformer = new TransformerImpl( anntModel, this.repoService, mapFile );
			int sucessCount = 0;
			int failureCount = 0;
			int rowCount = 0;
			while ( rowIterator.hasNext() ) {
				try {
					rowCount++;
					Map<String, String> rowMap = rowIterator.nextRow();
					Statement[] statements = transformer.transformToRDF( rowMap );
					doLoading( statements );
					statistics.setSucessCount( sucessCount++ );
				} catch (Exception e) {
					errLogger.error( "error when transforming and loading the row " + rowCount );
					errLogger.error( Arrays.toString( e.getStackTrace() ) );
					errLogger.error( e.getCause() );
					errLogger.error( e.getMessage() );
					statistics.setFailureCount( failureCount++ );
					continue;
				}
			}
		}
	}

	public void doLoading(Statement[] statements) throws LoaderException {
		loader.load( statements, this.getCurrentlyProcessingFile(), this.etlInputs.getPromoteParameter() );
	}

	private void printStatistics() {
		logger.info( "the ETL statistics are as follows" );
		logger.info( "File Name          SuccessRows          FailureRows" );
		for (FileStatistics stats : fileStatisticsList) {
			logger.info( stats.getFileName() + "          " + stats.getSucessCount() + "          " + stats.getFailureCount() );
		}
	}

	private RepoService makeRepoService(String repoConfigFileLocation) throws ExceptionWhileMakingRepoSerivce {
		try {
			RepoService repoService = RepoService.getInstance( repoConfigFileLocation, jenaInstanceFactory );

			return repoService;
		} catch (final FileNotFoundException fe) {
			logger.error( "Repository information file not found" );
			logger.error( fe );
			throw new ExceptionWhileMakingRepoSerivce( fe );
		} catch (Exception e) {
			logger.error( "Exception while making repoSerivce" );
			logger.error( e );
			throw new ExceptionWhileMakingRepoSerivce( e );
		}
	}

	public RepoService getRepoService() {
		return repoService;
	}

	public static ETLInputs getEtlInputs() {
		return etlInputs;
	}

	public static ETLInputs setEtlInputs(ETLInputs etlInputs) {
		return ETLEntryPoint.etlInputs = etlInputs;
	}

	public AnnotationFormModel getAnntModel() {
		return anntModel;
	}

	public JenaEIInstanceFactory getInstanceFactory() {
		return jenaInstanceFactory;
	}

	public String getCurrentlyProcessingFile() {
		return currentlyProcessingFile;
	}

}
