/**
 * 
 */
package org.eaglei.datatools.etl.server.extractor.parsers.excelparsers;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.table.DefaultTableModel;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.log4j.Logger;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.eaglei.datatools.etl.server.extractor.parsers.Data;
import org.eaglei.datatools.etl.server.extractor.parsers.EIFileParser;
import org.eaglei.datatools.etl.server.extractor.parsers.MapFileConfiguration;
import org.eaglei.datatools.excel.ExcelParserException;
import org.eaglei.datatools.excel.ExcelUtils;

/**
 * @author Sravan Kumar Cheriyala
 */

public class ExcelFileParser extends EIFileParser<Map<String, Data>> {

	private static org.apache.log4j.Logger logger = Logger.getLogger( ExcelFileParser.class );
	private static String FORM_VERSION_STRING = "FORM VERSION";

	private final Map<String, Integer> numberOfRowsReadFromTabs;

	private String mapsParentDirecotryPath;

	private String[] globalColumnsOrderInExcel;

	private String[] globalColumnsOrderInCSV;
	private boolean globalColumns;

	private Sheet LAB_SHEET;
	private final ExcelFileMetaData fileMetaData;

	private final Workbook excelWorkbook;

	public ExcelFileParser(final File excelFile) {
		numberOfRowsReadFromTabs = new HashMap<String, Integer>();
		fileMetaData = new ExcelFileParser.ExcelFileMetaData();
		excelWorkbook = getWorkbookOfExcel( excelFile );

		for (int sheetNum = 0; sheetNum <= excelWorkbook.getNumberOfSheets() - 1; sheetNum++) {
			if ( !excelWorkbook.isSheetHidden( sheetNum ) ) {
				final Sheet sheet = excelWorkbook.getSheetAt( sheetNum );
				String sheetName = sheet.getSheetName();
				try {
					fileMetaData.sheetNameToSheetMetaData.put( sheetName, readSheetMetaData( sheet ) );
				} catch (ParseException e) {
					logger.error( "parse exception for the sheet" + sheetName );
					logger.info( "ignoring the sheet " + sheetName );
					continue;
				}
			}
		}
	}

	public List<Sheet> getSheets() {
		final List<Sheet> listToReturn = new ArrayList<Sheet>();
		for (int i = 0; i <= excelWorkbook.getNumberOfSheets(); i++) {
			listToReturn.add( excelWorkbook.getSheetAt( i ) );
		}

		return listToReturn;

	}

	public FileMetaData getFileMetaData() throws ParseException {

		return fileMetaData;
	}

	@Override
	public Map<String, Data> parse(final File mapDirectory) throws ConfigurationException, IOException, ParseException {

		final Map<String, Data> ExcelTabNameToTabDataMap = new HashMap<String, Data>();

		LAB_SHEET = getLabSheet();
		ExcelFileMetaData fileMetaData = (ExcelFileMetaData)getFileMetaData();

		readConfigPropertiesOfFile( mapDirectory.getPath() );

		if ( globalColumnsOrderInExcel != null && globalColumnsOrderInExcel.length > 1 ) {
			globalColumns = true;
		}

		final HashMap<String, String> globalColumnsHash = new HashMap<String, String>();

		String[] configuredTabToProcess = MapFileConfiguration.getOrderofMapFiles( mapDirectory.getPath() );

		for (int sheetNum = 0; sheetNum <= excelWorkbook.getNumberOfSheets() - 1; sheetNum++) {
			if ( excelWorkbook.isSheetHidden( sheetNum ) ) {
				continue;
			}

			final Sheet sheet = excelWorkbook.getSheetAt( sheetNum );
			String sheetName = sheet.getSheetName();
			boolean isConfiguredSheet = false;
			for (String configuredTab : configuredTabToProcess) {
				if ( configuredTab.equalsIgnoreCase( sheetName ) ) {
					isConfiguredSheet = true;
				}
			}

			if ( !isConfiguredSheet ) {
				continue;
			}

			SheetMetaData sheetMetaData = fileMetaData.getSheetMetaData( sheetName );

			if ( sheet == null ) {
				logger.info( sheetName + " is not present in excel tab ...continuing onto next tab" );
				continue;
			}

			logger.info( "Reading " + sheet.getSheetName() );

			final int rowcount = sheet.getLastRowNum();
			if ( rowcount == 0 ) {
				logger.info( "no Rows in sheet...ignoring" );
				continue;
			}

			int colcount;
			if ( sheet.getRow( 1 ) == null ) {
				continue;
			} else {
				colcount = sheet.getRow( 1 ).getLastCellNum();
			}
			int startRow = 0;
			int endRow = 0;
			final int rowheadStart = 0;

			endRow = sheet.getPhysicalNumberOfRows();

			final TabConfigProperites properties = readConfigPropertiesForTab( sheet, mapDirectory.getPath() );
			startRow = properties.getStartRow();

			final Data tabData = new Data( 1, colcount + 4 );

			if ( globalColumns && !sheet.getSheetName().equalsIgnoreCase( "lab" ) ) {
				tabData.setColumnCount( tabData.getColumnCount() + globalColumnsOrderInCSV.length + 1 );
			}

			startRow = startRow - 1;
			int rowCount = 0;
			while ( startRow <= endRow - 1 ) {
				final Row row = sheet.getRow( startRow );

				if ( row == null ) {
					startRow++;
					continue;
				}

				if ( ExcelUtils.isBlankRow( row ) ) {
					startRow++;
					continue;
				}
				tabData.setRowCount( rowCount + 1 );

				final Iterator<Cell> cellIter = row.cellIterator();
				int tempLoop = 0;
				int colNum = 0;
				while ( cellIter.hasNext() ) {
					if ( tempLoop >= colcount ) {
						break;
					}
					final Cell cell = cellIter.next();

					if ( sheet.getSheetName().equalsIgnoreCase( "lab" ) && globalColumns ) {
						if ( tempLoop <= globalColumnsOrderInExcel.length - 1 ) {
							globalColumnsHash.put( globalColumnsOrderInExcel[tempLoop], cell.getStringCellValue() );
						}
					}

					final int rowNum = rowCount;

					if ( globalColumns && rowNum >= rowheadStart && globalColumnsOrderInExcel[0] != null && !sheet.getSheetName().equalsIgnoreCase( "lab" ) ) {
						colNum = cell.getColumnIndex() + globalColumnsOrderInCSV.length;
					} else {
						colNum = cell.getColumnIndex();
					}

					String columnName = evaluateColumnNameWhenGolobalColumnPreset( colNum, sheetMetaData );

					addCell( tabData, cell, rowNum, colNum, columnName );
					tempLoop++;
				}

				addGlobalColumnsToRow( globalColumnsHash, sheetMetaData, rowheadStart, tabData, startRow, rowCount );
				startRow++;
				rowCount++;
			}

			numberOfRowsReadFromTabs.put( sheet.getSheetName(), rowCount );

			ExcelTabNameToTabDataMap.put( sheet.getSheetName().toUpperCase(), tabData );

		}
		return ExcelTabNameToTabDataMap;
	}

	public Sheet getLabSheet() {
		final Sheet LAB_SHEET = excelWorkbook.getSheet( "lab" );
		return LAB_SHEET;
	}

	private void addGlobalColumnsToRow(final HashMap<String, String> globalColumnsHash, final SheetMetaData sheetMetaData, final int rowheadStart, final Data tabData, final int startRow, final int rowCount) throws UnsupportedEncodingException {

		if ( globalColumns && globalColumnsOrderInExcel[0] != null && !sheetMetaData.getSheetName().equalsIgnoreCase( "lab" ) ) {

			if ( startRow == rowheadStart && globalColumnsOrderInExcel.length > 0 ) {
				int y = 0;
				for (final String globalValue : globalColumnsOrderInCSV) {
					addCell( tabData, globalValue, rowCount, y, globalValue );
					if ( logger.isDebugEnabled() )
						logger.debug( "adding global row at " + rowCount + " and column" + y + "the value is " + globalValue );
					y++;
				}
			}

			else if ( startRow > rowheadStart && globalColumnsHash != null ) {
				int y = 0;
				for (final String strColumn : globalColumnsOrderInCSV) {
					addCell( tabData, globalColumnsHash.get( strColumn ).toString(), rowCount, y, strColumn );
					logger.debug( "adding global row at " + rowCount + " and column" + y + "the value is " + strColumn );

					y++;
				}
			}
		}
	}

	private Workbook getWorkbookOfExcel(final File excelFile) {
		Workbook wb;
		try {
			wb = WorkbookFactory.create( new FileInputStream( excelFile ) );
		} catch (final FileNotFoundException e) {
			throw new ExcelParserException( e );
		} catch (final InvalidFormatException e) {
			logger.error( "Expecting an Excel file.." + e + "Ignoring...." );
			throw new ExcelParserException( e );
		} catch (final IOException e) {
			logger.error( "IOExpecting an Excel file.." + e + "Ignoring...." );
			throw new ExcelParserException( e );
		}

		return wb;
	}

	protected String readMapDirectoryPathFromFile() throws ParseException {
		return readMapDirectoryPathFromFile( LAB_SHEET );
	}

	private String readMapDirectoryPathFromFile(final Sheet sheet) throws ParseException {

		return mapsParentDirecotryPath + "/" + getFormVersion( sheet );
	}

	public String getFormVersion(final Sheet sheet) throws ParseException {
		String mapDirectoryName = "";
		String mapDirectoryVersion = "";
		boolean formVersionColumnFound = false;

		final Row headerow = getHeaderRow( sheet );
		final Iterator<Cell> cellIter = headerow.cellIterator();
		int cellNum = 0;
		while ( cellIter.hasNext() ) {
			final Cell cell = cellIter.next();
			if ( cell.getStringCellValue().contains( FORM_VERSION_STRING ) ) {
				mapDirectoryName = cell.getStringCellValue().split( FORM_VERSION_STRING )[0].toLowerCase().trim();
				formVersionColumnFound = true;
			}
			if ( formVersionColumnFound ) {
				final Row valueRow = sheet.getRow( headerow.getRowNum() + 1 );
				final Cell valuecell = valueRow.getCell( cellNum );
				if ( !valuecell.getStringCellValue().equals( "" ) ) {
					mapDirectoryVersion = valuecell.getStringCellValue().toLowerCase();
					break;
				}
			}
			cellNum++;
		}
		return mapDirectoryName + "_" + mapDirectoryVersion;
	}

	private Row getHeaderRow(final Sheet sheet) throws ParseException {
		Row row;
		double endRow = 0;
		int k = 0;
		endRow = sheet.getPhysicalNumberOfRows();

		while ( k <= endRow - 1 ) {
			row = sheet.getRow( k );
			if ( row == null ) {
				k++;
				continue;
			}
			final Iterator<Cell> cellIter = row.cellIterator();
			while ( cellIter.hasNext() ) {
				final Cell cell = cellIter.next();

				if ( cell.getCellType() != Cell.CELL_TYPE_STRING ) {
					continue;
				}

				if ( cell.getStringCellValue().contains( FORM_VERSION_STRING ) ) {
					return row;
				}
			}
			k++;
		}
		throw new ParseException( "Header row not found in this Excel File", 0 );
	}

	public SheetMetaData readSheetMetaData(final Sheet sheet) throws ParseException {

		final Map<String, Integer> columnNameToNumberMap = new HashMap<String, Integer>();
		final Map<Integer, String> columnNumberToNameMap = new HashMap<Integer, String>();

		final SheetMetaData sheetMetaData = new SheetMetaData();
		int i = 1;

		if ( globalColumns && !sheet.getSheetName().equalsIgnoreCase( "lab" ) ) {
			for (final String globalColumnName : globalColumnsOrderInCSV) {
				columnNameToNumberMap.put( globalColumnName, i );
				i++;
			}
		}
		final Row headerow = getHeaderRow( sheet );
		final Iterator<Cell> cellIter = headerow.cellIterator();
		while ( cellIter.hasNext() ) {
			final Cell cell = cellIter.next();
			columnNameToNumberMap.put( cell.getStringCellValue(), i );
			columnNumberToNameMap.put( i, cell.getStringCellValue() );
			i++;

		}
		sheetMetaData.columnNameToColumnNumberMap = columnNameToNumberMap;
		sheetMetaData.columnNumberToColumnNameMap = columnNumberToNameMap;
		sheetMetaData.sheetName = sheet.getSheetName();
		return sheetMetaData;
	}

	protected void readConfigPropertiesOfFile(final String mapDirectoryPath) throws IOException, ConfigurationException {
		final Map<String, String> rowConfigMap = ExcelTemplateConfiguration.getFormatingInfoOfFile( mapDirectoryPath );
		final String order = rowConfigMap.get( "order" );

		if ( rowConfigMap.get( "globalcolumnsorderinexcel" ) != null ) {
			globalColumnsOrderInExcel = rowConfigMap.get( "globalcolumnsorderinexcel" ).split( "#" );
		}

		if ( rowConfigMap.get( "globalcolumnsorderintoCSV" ) != null ) {
			globalColumnsOrderInCSV = rowConfigMap.get( "globalcolumnsorderintoCSV" ).split( "#" );
		}

	}

	protected TabConfigProperites readConfigPropertiesForTab(final Sheet sheet, final String mapDirecotryPath) throws ConfigurationException {
		final Map<String, String> rowConfigMap = ExcelTemplateConfiguration.getFormatingInfoOfRow( mapDirecotryPath, sheet.getSheetName() );
		final int startRow = Integer.parseInt( rowConfigMap.get( "startrow" ) );
		return new TabConfigProperites( startRow, sheet.getSheetName() );

	}

	private class TabConfigProperites {

		int startRow;
		String tabName;

		TabConfigProperites(final int startRow, final String tabName) {
			this.startRow = startRow;
			this.tabName = tabName;
		}

		public int getStartRow() {
			return startRow;
		}

		public String getTabName() {
			return tabName;
		}

	}

	private void addCell(final DefaultTableModel tableModel, final Cell cell, final int rowNum, final int colNum, String columnName) throws UnsupportedEncodingException {
		if ( cell.getCellType() == Cell.CELL_TYPE_STRING || cell.getCellType() == Cell.CELL_TYPE_FORMULA || cell.getCellType() == Cell.CELL_TYPE_BOOLEAN ) {
			addCell( tableModel, cell.getStringCellValue(), rowNum, colNum, columnName );
		} else if ( cell.getCellType() == Cell.CELL_TYPE_NUMERIC ) {
			addCell( tableModel, (int)cell.getNumericCellValue() + "", rowNum, colNum, columnName );
		}
	}

	private void addCell(final DefaultTableModel tableModel, final String cellValue, final int rowNum, final int colNum, String columnName) throws UnsupportedEncodingException {
		String columnValue = new String( cellValue.replaceAll( "_x[^_]*?_", "" ).getBytes(), "utf-8" );

		if ( logger.isDebugEnabled() )
			logger.debug( "adding in the cell column name = " + columnName + "  column value = " + columnValue );

		String[] cellData = new String[] { columnName.trim().replaceAll( "\\s+", "_" ), columnValue };

		tableModel.setValueAt( cellData, rowNum, colNum );
	}

	private String evaluateColumnNameWhenGolobalColumnPreset(final int colNum, SheetMetaData sheetMetaData) {
		int keyTofetchColumnNameFromMap = 0;

		if ( !sheetMetaData.getSheetName().equalsIgnoreCase( "lab" ) && globalColumns ) {
			keyTofetchColumnNameFromMap = Math.abs( ( colNum - globalColumnsOrderInCSV.length ) );
		} else {
			keyTofetchColumnNameFromMap = colNum;
		}

		return sheetMetaData.getColumnNameByColumnNumber( keyTofetchColumnNameFromMap + 1 );
	}

	public Map<String, Integer> getNumberOfRowsReadFromTabs() {
		return numberOfRowsReadFromTabs;
	}

	public FileMetaData getMetaData() {
		return fileMetaData;
	}

	public class ExcelFileMetaData implements FileMetaData {
		private final Map<String, SheetMetaData> sheetNameToSheetMetaData;

		public ExcelFileMetaData() {
			sheetNameToSheetMetaData = new HashMap<String, SheetMetaData>();
		}

		public SheetMetaData getSheetMetaData(final String sheetName) {
			return sheetNameToSheetMetaData.get( sheetName );
		}

	}

	public class SheetMetaData {
		private Map<String, Integer> columnNameToColumnNumberMap;
		private Map<Integer, String> columnNumberToColumnNameMap;
		private String sheetName;

		public int getColumnNumberByColumnName(final String columnName) {
			return columnNameToColumnNumberMap.get( columnName );
		}

		public Map<String, Integer> getMapofColumnNameAndNumber() {
			return columnNameToColumnNumberMap;
		}

		public Map<Integer, String> getColumnNumberToColumnNameMap() {
			return columnNumberToColumnNameMap;
		}

		public String getColumnNameByColumnNumber(final Integer columnNumber) {
			return columnNumberToColumnNameMap.get( columnNumber );
		}

		public String getSheetName() {
			return sheetName;
		}

	}

}
