// The MIT License
//
// Copyright (c) 2005 Michael Grove
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.

package org.eaglei.datatools.etl.utils;

import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.BufferedWriter;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;

import java.net.URL;

import java.util.Random;
import java.util.Set;
import java.util.LinkedHashSet;
import java.util.Iterator;

import java.awt.Toolkit;
import java.awt.Point;
import java.awt.Dimension;
import java.awt.Frame;

import javax.swing.JFrame;

/**
 * 
 * <p>
 * Title: BasicUtils
 * </p>
 * 
 * <p>
 * Description: A collection of basic functions for standard operations like saving a string to file, and centering a frame
 * </p>
 * 
 * <p>
 * Copyright: Copyright (c) 2003
 * </p>
 * 
 * <p>
 * Company: Mindswap (http://www.mindswap.org)
 * </p>
 * 
 * @author Michael Grove
 * @version 1.0
 */
public class BasicUtils {

	private static final Random RAND = new Random();

	/**
	 * Given a path to a file on the local disk, return the contents of that file as a String.<br>
	 * <br>
	 * 
	 * @param fn
	 *            Fully qualified file name to a file on the local disk
	 * @return Contents of the file as a String
	 * @throws java.io.IOException
	 *             if there are problems opening or reading from the file
	 * @throws java.io.FileNotFoundException
	 *             if the file cannot be found
	 */
	public static String getFileAsString(String fn) throws java.io.IOException, java.io.FileNotFoundException {
		BufferedReader reader = new BufferedReader( new InputStreamReader( new FileInputStream( fn ) ) );
		return getStringFromBufferedReader( reader );
	} // getFileAsString

	/**
	 * Returns the specified URL as a string. This function will read the contents at the specified URl and return the results.
	 * 
	 * @param theURL
	 *            URL the URL to read from
	 * @return String the String found at the URL
	 * @throws IOException
	 */
	public static String getURLAsString(URL theURL) throws IOException {
		if ( theURL == null )
			return null;

		BufferedReader reader = new BufferedReader( new InputStreamReader( theURL.openStream() ) );
		return getStringFromBufferedReader( reader );
	}

	/**
	 * Encodes the specified URL, changes spaces to %20
	 * 
	 * @param theURL
	 *            URL the url to encode
	 * @return URL the encoded url
	 */
	public static URL encode(URL theURL) {
		try {
			String aURLString = theURL.toExternalForm();
			aURLString = aURLString.replaceAll( " ", "%20" );
			theURL = new URL( aURLString );
		} catch (Exception ex) {
		}

		return theURL;
	}

	/**
	 * Read the contents of the specified reader return them as a String
	 * 
	 * @param aReader
	 *            BufferedReader the reader to read from
	 * @return String the string contained in the reader
	 * @throws IOException
	 */
	public static String getStringFromBufferedReader(BufferedReader aReader) throws IOException {
		StringBuffer theFile = new StringBuffer();
		String line = aReader.readLine();
		while ( line != null ) {
			theFile.append( line + "\n" );
			line = aReader.readLine();
		} // while
		aReader.close();
		return theFile.toString();
	}

	/**
	 * Save the specifed string to the given file name
	 * 
	 * @param toSave
	 *            String the string to save
	 * @param fn
	 *            String the file to save the string to
	 * @throws IOException
	 * @throws FileNotFoundException
	 */
	public static void saveStringToFile(String toSave, String fn) throws IOException, FileNotFoundException {
		BufferedWriter writer = new BufferedWriter( new OutputStreamWriter( new FileOutputStream( fn ) ) );
		java.util.StringTokenizer st = new java.util.StringTokenizer( toSave, "\n", true );
		String s;
		while ( st.hasMoreTokens() ) {
			s = st.nextToken();
			if ( s.equals( "\n" ) )
				writer.newLine();
			else
				writer.write( s );
		} // while
		writer.flush();
		writer.close();
	} // saveStringToFile

	/**
	 * Returns a randomly generated string of up to 15 characters.<br>
	 * <br>
	 * 
	 * @return A randomly generated string
	 */
	public static String getRandomString() {
		return getRandomString( 15 );
	}

	/**
	 * Returns a randomly generated string of up to X characters.<br>
	 * <br>
	 * 
	 * @param numChars
	 *            the number of characters long the random string should be
	 * 
	 * @return A randomly generated string
	 */
	public static String getRandomString(int numChars) {
		int chars = RAND.nextInt( numChars );
		while ( chars == 0 )
			chars = RAND.nextInt( numChars );
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < chars; i++) {
			int index = 97 + RAND.nextInt( 26 );
			char c = (char)index;
			sb.append( c );
		} // for
		return sb.toString();
	} // getRandomString

	/**
	 * Returns whether or not the parameter is a valid URL
	 * 
	 * @param s
	 *            String - the url to test
	 * @return boolean - true if the parameter is a valid url, false otherwise
	 */
	public static boolean isValidURL(String s) {
		try {
			new java.net.URL( s );
			return true;
		} catch (Exception ex) {
			return false;
		}
	}

	/**
	 * Returns string if the parameter represents a valid file URL
	 * 
	 * @param s
	 *            String the url
	 * @return boolean true if it is a valid file URL, false otherwise
	 */
	public static boolean isFileURL(String s) {
		if ( isValidURL( s ) && s.startsWith( "file:" ) )
			return true;
		else
			return false;
	}

	/**
	 * Returns true or false depending on whether or not the specified string is in the array.
	 * 
	 * @param elem
	 *            int to search for
	 * @param array
	 *            int array to search through
	 * @return true if elem is in array, false otherwise
	 */
	public static boolean isInArray(int elem, int[] array) {
		if ( array == null )
			return false;
		for (int i = 0; i < array.length; i++)
			if ( array[i] == elem )
				return true;
		return false;
	} // isInArray

	/**
	 * Converts a regular java string to its unicode string representation<br>
	 * <br>
	 * 
	 * @param s
	 *            Java String to convert to unicode
	 * @return the converted string
	 */
	public static String convertStringToUnicodeString(String s) throws Exception {
		StringBuffer convert = new StringBuffer();
		for (int i = 0; i < s.length(); i++) {
			char ch = s.charAt( i );

			if ( (byte)ch >= 0 && (int)ch <= 255 )
				convert.append( ch );
			else {
				if ( (int)ch > 255 || (int)ch < 0 )
					throw new Exception( "Invalid Character: " + ch );
				String val = "\\u";
				int intVal = (int)ch;
				int bit4 = (int)( intVal / Math.pow( 16, 3 ) );
				intVal -= bit4 * Math.pow( 16, 3 );
				int bit3 = (int)( intVal / Math.pow( 16, 2 ) );
				intVal -= bit3 * Math.pow( 16, 2 );
				int bit2 = (int)( intVal / Math.pow( 16, 1 ) );
				intVal -= bit2 * Math.pow( 16, 1 );
				int bit1 = (int)( intVal / Math.pow( 16, 0 ) );
				intVal -= bit1 * Math.pow( 16, 0 );
				if ( intVal != 0 )
					System.err.println( "Oops, no character to conver to..." );
				convert.append( val + Character.toUpperCase( Integer.toHexString( bit4 ).charAt( 0 ) ) + Character.toUpperCase( Integer.toHexString( bit3 ).charAt( 0 ) ) + Character.toUpperCase( Integer.toHexString( bit2 ).charAt( 0 ) )
						+ Character.toUpperCase( Integer.toHexString( bit1 ).charAt( 0 ) ) );
			} // else
		} // for
		return convert.toString();
	} // convertStringToUnicodeString

	/**
	 * Converts any hex entity type chars ("&#x00E9") to their ascii equivalent
	 * 
	 * @param theString
	 *            String the string to conver
	 * @return String the converted string
	 */
	public static String hexEntityStringConvert(String theString) {
		int index = theString.indexOf( "&#x" );
		while ( index != -1 ) {
			String value = theString.substring( index + 3, index + 7 );

			char newChar = (char)Integer.valueOf( value, 16 ).intValue();
			String charString = String.valueOf( newChar );
			theString = theString.replaceAll( "&#x" + value + ";", charString );
			theString = theString.replaceAll( "&#x" + value, charString );

			index = theString.indexOf( "&#x", index + 3 );
		}

		return theString;
	}

	/**
	 * Moves a frame in front of another frame
	 * 
	 * @param theParent
	 *            JFrame the parent, or reference frame
	 * @param theFrame
	 *            JFrame the frame to move in front
	 */
	public static void toFrontHack(JFrame theParent, JFrame theFrame) {
		if ( theFrame.getState() == Frame.ICONIFIED ) {
			theFrame.setState( Frame.NORMAL );
		} else {
			// save position
			Point origPos = theFrame.getLocation();

			Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
			// move off the visible screen
			theFrame.setLocation( screenSize.width + theParent.getWidth(), screenSize.height + theParent.getHeight() );
			theFrame.setState( Frame.ICONIFIED );
			theFrame.setState( Frame.NORMAL );
			// return to original position
			theFrame.setLocation( origPos );
		}
		// repaint since damage repair is not consistant
		theParent.repaint();
		theFrame.repaint();
	}

	/**
	 * Centers a frame with respect to another window, or the whole screen
	 * 
	 * @param theParent
	 *            Window the reference (parent) frame, or null to center on the entire screen
	 * @param theFrame
	 *            Window the window to center
	 */
	public static void centerFrame(java.awt.Window theParent, java.awt.Window theFrame) {

		if ( theParent != null ) {
			Dimension aSize = theFrame.getSize();
			int x = theParent.getX() + theParent.getWidth() / 2 - aSize.width / 2;
			int y = theParent.getY() + theParent.getHeight() / 2 - aSize.height / 2;

			theFrame.setLocation( x, y );
		} else {
			Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
			Dimension size = theFrame.getSize();
			screenSize.height = screenSize.height / 2;
			screenSize.width = screenSize.width / 2;
			size.height = size.height / 2;
			size.width = size.width / 2;
			int y = screenSize.height - size.height;
			int x = screenSize.width - size.width;
			theFrame.setLocation( x, y );
		}
	}

	/**
	 * Checks to see if any elements of one set are present in another. First parameter is the list of elements to search for, the second parameter is the list to search in. Basically tests to see if the two sets intersect
	 * 
	 * @param theList
	 *            Set the elements to look for
	 * @param toSearch
	 *            Set the search set
	 * @return boolean true if any element in the first set is present in the second
	 */
	public static boolean containsAny(Set theList, Set toSearch) {

		if ( toSearch.isEmpty() )
			return false;
		else {
			Iterator iter = theList.iterator();
			while ( iter.hasNext() ) {
				Object obj = iter.next();
				if ( toSearch.contains( obj ) )
					return true;
			}
			return false;
		}
	}

	/**
	 * Returns the list of elements returned by an iterator in a Set
	 * 
	 * @param theIter
	 *            Iterator
	 * @return Set
	 */
	public static Set collectElements(Iterator theIter) {
		LinkedHashSet aSet = new LinkedHashSet();
		while ( theIter.hasNext() )
			aSet.add( theIter.next() );
		return aSet;
	}

	/**
	 * Changes an images alpha values to the specified value. Works on Windows platforms only.<br>
	 * <br>
	 * 
	 * @param img
	 *            Image to fade
	 * @return copy of the image with reduced alpha values
	 */
	public static java.awt.Image getTranslucentVersionOf(java.awt.Image img) {
		// if (theAlpha > 1)
		// theAlpha = 1;
		// else if (theAlpha < 0)
		// theAlpha = 0;

		int w = img.getWidth( null );
		int h = img.getHeight( null );
		int[] pix = new int[w * h];

		// get the pixels:
		java.awt.image.PixelGrabber pg = new java.awt.image.PixelGrabber( img, 0, 0, w, h, pix, 0, w );
		try {
			pg.grabPixels();
		} // try
		catch (Exception e) {
			return img;
		}

		// loop through the array, shift the alpha values
		for (int i = pix.length - 1; i >= 0; i--)
			pix[i] = 0x99000000 | ( pix[i] & 0x00ffffff );

		// make and return an Image of the array
		java.awt.image.MemoryImageSource mis = new java.awt.image.MemoryImageSource( w, h, pix, 0, w );
		return java.awt.Toolkit.getDefaultToolkit().createImage( mis );
	} // getTranslucentVersionOf

	/**
	 * Given a string, replace all occurances of one string with another.<br>
	 * <br>
	 * 
	 * @param host
	 *            parent string
	 * @param oldchar
	 *            String to remove
	 * @param newchar
	 *            String to insert in the place of oldchar in the String host
	 * @return copy of host with all occurances of param oldchar replaced with param newchar
	 */
	public static String replace(String host, String oldchar, String newchar) {
		int index = 0;
		while ( host.indexOf( oldchar, index ) != -1 ) // diff, no index
		{
			index = host.indexOf( oldchar, index );
			host = host.substring( 0, index ) + newchar + host.substring( index + oldchar.length(), host.length() );
			index += newchar.length(); // diff absent
		} // while
		return host;
	} // replace

	public static  String convertStreamToString(InputStream is) throws IOException {
		/*
		 * To convert the InputStream to String we use the Reader.read(char[] buffer) method. We iterate until the Reader return -1 which means there's no more data to read. We use the StringWriter class to produce the string.
		 */
		if ( is != null ) {
			Writer writer = new StringWriter();

			char[] buffer = new char[1024];
			try {
				Reader reader = new BufferedReader( new InputStreamReader( is, "UTF-8" ) );
				int n;
				while ( ( n = reader.read( buffer ) ) != -1 ) {
					writer.write( buffer, 0, n );
				}
			}
			finally {
				is.close();
			}
			return writer.toString();
		} else {
			return "";
		}
	}

} // Utils
