package org.eaglei.datatools.etl.server;

/**
 * @author Sravan Kumar Cheriyala
 * This class has services related to repository
 */
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.StringWriter;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.User;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.etl.server.loader.Loader.Promote;
import org.eaglei.datatools.jena.RESTRepositoryProvider;
import org.eaglei.datatools.jena.FooRepositorySecurityProvider;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIInstanceFactory;
import org.eaglei.security.Session;
import org.eaglei.services.repository.RepositoryProviderException;
import org.eaglei.services.repository.SecurityProvider;

import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.Resource;

public class RepoService {

	private static RepoService instance = null;
	private DatatoolsConfiguration config;
	RESTRepositoryProvider repoProvider;
	FooRepositorySecurityProvider securityProvider;
	JenaEIInstanceFactory instanceFactory;
	private static final String DEFAULT_CONFIGURATION = "etl-datatools.prop";
	private static final Log logger = LogFactory.getLog( RepoService.class );
	private String token;
	private Session session;
	private final String repoConfigFile;

	/* workflow transistions */
	public static final EIEntity DEFAULT_CREATE_TRANSITION = EIEntity.create( EIURI.create( "http://eagle-i.org/ont/repo/1.0/WFT_0" ), "Default Create" );
	public static final EIEntity SENT_TO_CURATION_TRANSITION = EIEntity.create( EIURI.create( "http://eagle-i.org/ont/repo/1.0/WFT_1" ), "Send to Curation" );
	public static final EIEntity PUBLISH_TRANSITION = EIEntity.create( EIURI.create( "http://eagle-i.org/ont/repo/1.0/WFT_2" ), "Publish" );

	private RepoService(String file, JenaEIInstanceFactory instanceFactory, SecurityProvider generalSecurityProvider) throws IOException {
		this.instanceFactory = instanceFactory;
		config = getConfiguration( file );
		repoConfigFile = file;
		repoProvider = (RESTRepositoryProvider)new RESTRepositoryProvider( instanceFactory, generalSecurityProvider );
		securityProvider = repoProvider.getDatatoolsSecurityProvider();
	}

	private RepoService(JenaEIInstanceFactory instanceFactory, SecurityProvider generalSecurityProvider) throws IOException, URISyntaxException {
		this("", instanceFactory, generalSecurityProvider);
	}

	public RepositoryProvider getRepositoryProvider() {
		return repoProvider;
	}
	
	public FooRepositorySecurityProvider getRepositorySecurityProvider() {
		return securityProvider;
	}

	public DatatoolsConfiguration getDatatoolsConfiguration() {
		return config;
	}

	public static RepoService getInstance(String file, JenaEIInstanceFactory instanceFactory, SecurityProvider generalSecurityProvider) throws Exception {
		if ( instance == null ) {
			instance = new RepoService( file, instanceFactory, generalSecurityProvider );
			instance.login( file );

		}
		return instance;
	}

	public boolean workFlowTansistion(EIURI resource, Promote promote) throws Exception {

		if ( promote == Promote.PUBLISH ) {
			repoProvider.transition( getSession(), Arrays.asList( resource ), RepoService.SENT_TO_CURATION_TRANSITION );
			repoProvider.claim( getSession(), Arrays.asList( resource ) );
			repoProvider.transition( getSession(), Arrays.asList( resource ), RepoService.PUBLISH_TRANSITION );
			return true;
		} else if ( promote == Promote.CURATION ) {
			repoProvider.transition( getSession(), Arrays.asList( resource ), RepoService.DEFAULT_CREATE_TRANSITION );
			repoProvider.transition( getSession(), Arrays.asList( resource ), RepoService.SENT_TO_CURATION_TRANSITION );
			repoProvider.release( getSession(), Arrays.asList( resource ) );
			return true;
		} else if ( promote == Promote.DRAFT ) {
			repoProvider.release( getSession(), Arrays.asList( resource ) );
			return true;
		} else {
			return false;
		}

	}

	public boolean workFlowTansistion(List<EIURI> lstResource, Promote promote) throws Exception {

		if ( promote == Promote.PUBLISH ) {
			repoProvider.transition( getSession(), lstResource, RepoService.SENT_TO_CURATION_TRANSITION );
			repoProvider.claim( getSession(), lstResource );
			repoProvider.transition( getSession(), lstResource, RepoService.PUBLISH_TRANSITION );
			repoProvider.release( getSession(), lstResource );
			return true;
		} else if ( promote == Promote.CURATION ) {
			repoProvider.transition( getSession(), lstResource, RepoService.DEFAULT_CREATE_TRANSITION );
			repoProvider.transition( getSession(), lstResource, RepoService.SENT_TO_CURATION_TRANSITION );
			repoProvider.release( getSession(), lstResource );
			return true;
		} else if ( promote == Promote.DRAFT ) {
			repoProvider.release( getSession(), lstResource );
			return true;
		} else {
			return false;
		}

	}

	public static RepoService getInstance(JenaEIInstanceFactory instanceFactory, SecurityProvider generalSecurityProvider) throws Exception {
		return getInstance("", instanceFactory, generalSecurityProvider);
	}

	/**
	 * pushed the Model to Repo and promotes it
	 * 
	 * @param rdfModel
	 * @param session
	 * @param uri
	 * @return
	 * @throws Exception
	 */
	public boolean pushToRepo(Model rdfModel, org.eaglei.security.Session session, String uri, String promoteParameter) throws Exception {

		StringWriter swriter = new StringWriter();
		rdfModel.write( swriter, "RDF/XML" );
		repoProvider.createInstance( session, swriter.toString(), uri, repoProvider.DEFAULT_WORKSPACE_ENTITY );
		token = repoProvider.updateInstance( session, swriter.toString(), uri, null );
		repoProvider.updateInstance( session, swriter.toString(), uri, token );
		return true;
	}

	public boolean pushToRepo(EIInstance instance) throws Exception {
		repoProvider.createInstance( getSession(), instance, RepositoryProvider.DEFAULT_WORKSPACE_ENTITY );
		
		return true;
	}
	
	public boolean pushToRepo(String rdfString,String instanceUri) throws RepositoryProviderException
	{
		repoProvider.createInstance( session, rdfString, instanceUri, RepositoryProvider.DEFAULT_WORKSPACE_ENTITY );
		return true;
	}

	public String query(final String query) throws Exception {
		return repoProvider.query( getSession(), query );
	}

	/**
	 * gets the new instance from Repo
	 * 
	 * @param countofInstances
	 * @return
	 * @throws Exception
	 */
	public List<EIURI> getNewInstancesFromRepo(int countofInstances) {
		try {
			return repoProvider.getNewInstanceID( this.getSession(), countofInstances );
		} catch (Exception e) {
			throw new RuntimeException( e );
		}

	}

	public void setPropertyFilePathAndgetNewRepoProvider(final String configFilePath) throws IOException {
		final File configFile = new File( configFilePath );
		logger.debug( "loading configuration file: " + configFile );
		config = new DatatoolsConfiguration( configFile );
		securityProvider.setDefaultRepositoryURL(config.getDatatoolsRepositoryURL());
	}

	/**
	 * get the configuration object by the String as path to file
	 * 
	 * @param DEFAULT_CONFIGURATION
	 * @return
	 * @throws Exception
	 */
	private DatatoolsConfiguration getConfiguration(final String DEFAULT_CONFIGURATION) {
		try {
			final File configFile = new File( DEFAULT_CONFIGURATION );
			logger.debug( "loading configuration file: " + configFile );
			return new DatatoolsConfiguration( configFile );
		} catch (Exception e) {
			logger.error( "Error loading configuration from " + DEFAULT_CONFIGURATION + " " + e );
			throw new RuntimeException( e );
		}
	}

	/**
	 * get the Configuration of default ETL properties file
	 * 
	 * @return
	 * @throws IOException
	 * @throws URISyntaxException
	 * @throws Exception
	 */
	private DatatoolsConfiguration getConfiguration() throws IOException, URISyntaxException {
		final URL configURL = this.getClass().getClassLoader().getResource( DEFAULT_CONFIGURATION );
		if ( configURL == null ) {
			logger.error( "Could not locate " + DEFAULT_CONFIGURATION + " on classpath" );
			throw new FileNotFoundException( "Could not locate " + DEFAULT_CONFIGURATION + " on classpath" );
		}
		try {
			final File configFile = new File( configURL.toURI() );
			logger.debug( "loading configuration file: " + configFile );
			return new DatatoolsConfiguration( configFile );
		} catch (RuntimeException e) {
			logger.error( "Error loading configuration from " + DEFAULT_CONFIGURATION + " " + e );
			throw new RuntimeException( e );
		}
	}

	public Session login(String repoPath) throws Exception {
		Map<String, String> credentialMap = ETLRepositoryConfiguration.getCredentialsForRepo( repoPath );
		User strAry = securityProvider.login( credentialMap.get( "user" ), credentialMap.get( "password" ) );
		if ( strAry == null ) {
			logger.error( "Could not authenticate please check username,password and Repository URL...Exiting System" );
			System.exit( 2 );
		}
		setSession( strAry.getSession() );
		return getSession();
	}

	public Resource getNewResourceFromRepository(Model modelTocreateStatments) {
		List<EIURI> lstNewURI = getNewInstancesFromRepo( 1 );
		Resource subject = modelTocreateStatments.createResource( lstNewURI.get( 0 ).toString() );
		return subject;
	}

	public Session getSession() {
		return session;
	}

	public void setSession(Session session) {
		this.session = session;
	}

	public String getRepoConfigFile() {
		return repoConfigFile;
	}

}
