// The MIT License
//
// Copyright (c) 2005 Michael Grove
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.

package org.eaglei.datatools.etl.utils;

import java.awt.BorderLayout;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JFrame;
import javax.swing.JProgressBar;
import javax.swing.BoxLayout;
import javax.swing.Box;
import javax.swing.BorderFactory;

/**
 *
 * <p>Title: </p>
 *
 * <p>Description:  a simple progress frame; it takes a runnable task and executes it.  if the task is not completed right away, a progress frame will pop up until the task is complete.</p>
 *
 * <p>Copyright: Copyright (c) 2006</p>
 *
 * <p>Company: Mindswap (http://www.mindswap.org)</p>
 *
 * @author Michael Grove
 * @version 1.0
 */
public class ProgressFrame extends JDialog
{
    // thread that is doing the job
    private Thread mThread = new Thread();

    // time thread was started
    private long mStartTime;

    // wait this milliseconds before popping up the progress frame
    private static final long WAIT_TIME = 900;

    // this thread will watch the progress and popup the frame if necessary
    private Thread mWaitThread = new Thread();

    /**
     * Create a new ProgressFrame
     * @param parent JFrame the parent
     * @param title String the title of the frame
     * @param msg String the message to display in the frame
     * @param task Runnable the task to monitor
     */
    public ProgressFrame(JFrame parent, String title, String msg, final Runnable task) { this(parent,title,msg,task,true); }

    /**
     * Create a new ProgressFrame
     * @param parent JFrame the parent
     * @param title String the title of the frame
     * @param msg String the message to display in the frame
     * @param task Runnable the task to monitor
     * @param modal boolean the modality of the dialog
     */
    public ProgressFrame(JFrame parent, String title, String msg, final Runnable task, boolean modal)
    {
        // display progress bar and load ontology in new thread
        super(parent, title, modal);
        getContentPane().setLayout(new BorderLayout());
        JProgressBar progressBar = new JProgressBar();
        progressBar.setIndeterminate(true);

        JButton cancelBtn = new JButton("Cancel");
        cancelBtn.setFont(new Font("Tahoma", Font.PLAIN, 11));

        JPanel panel = new JPanel();
        panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
        panel.setBorder(BorderFactory.createEmptyBorder(3, 3, 3, 3));
        panel.add(new JLabel(msg));
        panel.add(Box.createVerticalStrut(5));
        panel.add(progressBar);
        panel.add(Box.createVerticalStrut(5));
        panel.add(cancelBtn);
        getContentPane().add(panel);

        pack();
        setSize(400, getHeight());
        setLocation(80,100);
        setResizable(false);

        setLocation(80,100);
        setResizable(false);

        addWindowListener(new WindowAdapter() {
          public void windowClosing(WindowEvent e)
          {
            mThread.stop();
            dispose();
          } // windowClosing
        });

        mThread = new Thread() {
            public void run() {
                try {
                    task.run();
                } // try
                catch (Throwable ex) {
                    System.err.println("task run exception");
                    ex.printStackTrace();
                } // catch
                finally {
                    try {
                        dispose();
                    } // try
                    catch (Exception ex) {} // if it gets interuppted and barfs, no biggie
                } // finally
            }

        };
        mThread.start();

        cancelBtn.addActionListener (
            new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                  try {
                    mThread.stop();
                    dispose();
                  } // try
                  catch (Exception ex)
                  {
                    System.err.println("task kill exception");
                  } // catch
                } // actionPerformed
            } // ActionListener
        );

        mWaitThread = new Thread() {
            public void run() {
                int sleepTime = 100;
                long elapsedTime;
                do {
                    try {
                        Thread.sleep(sleepTime);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                    elapsedTime = System.currentTimeMillis() - mStartTime;
                }
                while(elapsedTime < WAIT_TIME && mThread.isAlive());

                if(mThread.isAlive())
                    show();
            }
        };
        mStartTime = System.currentTimeMillis();
        mWaitThread.start();
    }

}
