// The MIT License
//
// Copyright (c) 2005 Michael Grove
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.

package org.eaglei.datatools.etl.utils;

import javax.swing.JTextField;
import javax.swing.JPasswordField;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.JOptionPane;

import java.awt.GridLayout;

import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;

import java.net.URLEncoder;
import java.net.URLConnection;
import java.net.Authenticator;
import java.net.PasswordAuthentication;
import java.net.URL;

import java.util.Vector;
import java.util.Iterator;
import java.util.Date;

import java.text.SimpleDateFormat;

/**
 * <p>Title: SubmitRDF - a utility class for submitting rdf data to the mindswap portal </p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: </p>
 * @author Michael Grove
 * @author Ron Alford
 * @version 1.0
 */
public class SubmitRDF {
    public static boolean DEBUG = false;

    private String rdfpayload;
    private String name;
    private String description = "Program Data";
    private Date date = new Date();
    private Vector emails;
    private Vector categories;

    private JTextField mUser;
    private JTextField mDescription;

    private SimpleDateFormat sdf;

    private static String lastUser = null;

    /**
     * Create a new SubmitRDF object with the specified RDF for submission
     * @param rdf String the rdf to submit
     */
    public SubmitRDF(String rdf) {
        rdfpayload = rdf;
        Authenticator.setDefault(new VisualAuthenticator());
        name = null;
        categories = new Vector();
        emails = new Vector();
        sdf = new SimpleDateFormat();
        sdf.applyPattern("yyyy-MM-dd");
    }

    /**
     * Create a new SubmitRDF object with the specified RDF for submission
     * @param rdf String the rdf to submit
     * @param username String user name for gaining access to the submit script
     * @param password String password for gaining access to the submit script
     */
    public SubmitRDF(String rdf, String username, String password) {
        rdfpayload = rdf;
        Authenticator.setDefault(new StaticAuthenticator(username, password));
        name = null;
        categories = new Vector();
        emails = new Vector();
        sdf = new SimpleDateFormat();
        sdf.applyPattern("yyyy-MM-dd");
    }

    /**
     * Adds a creator email to the list of creators for this submission
     * @param email String
     */
    public void addCreator(String email) {
        emails.add(email);
    }

    /**
     * Sets the list of emails for the creators of this submission
     * @param email List
     */
    public void setCreators(java.util.List email) {
        emails = new Vector(email);
    }

    /**
     * Add a new category to the submission
     * @param category String
     */
    public void addCategory(String category) {
        categories.add(category);
    }

    /**
     * Sets the list of categories for the submission
     * @param category List
     */
    public void setCategories(java.util.List category) {
        categories = new Vector(category);
    }

    /**
     * Sets the name of the submitter
     * @param fname String
     */
    public void setName(String fname) {
        name = fname;
    }

    /**
     * Sets the description of the submission
     * @param desc String
     */
    public void setDescription(String desc) {
        description = desc;
    }

    /**
     * Set the date of the submission
     * @param d Date
     */
    public void setDate(Date d) {
        date = d;
    }

    /**
     * Set the date of the submission
     * @param d String
     */
    public void setDate(String d) {
        try {
            date = sdf.parse(d);
        } catch (Exception e) {}
        if (date == null) {
            date = new Date();
        }
    }

    private void writeCreators(PrintWriter out) throws java.io.IOException {
        Iterator iter = emails.iterator();

        while (iter.hasNext()) {
            String encodedEmail = URLEncoder.encode( (String) iter.next(), "UTF-8");
            out.write("&creator=" + encodedEmail);
        }
    }

    private void writeCategories(PrintWriter out) throws java.io.IOException {
        Iterator iter = categories.iterator();

        while (iter.hasNext()) {
            String encodedCategory = URLEncoder.encode( (String) iter.next(), "UTF-8");
            out.write("&category=" + encodedCategory);
        }
    }

    private void writeName(PrintWriter out) {
        if (name != null) {
            out.write("&num=" + name);
        }
    }

    private void writeDescription(PrintWriter out) {
        if (description != null) {
            out.write("&desc=" + description);
        }
    }

    private void writeDate(PrintWriter out) {
        if (date == null) {
            date = new Date();
        }

        String datestring = sdf.format(date);

        out.write("&date=" + datestring);

    }

    private boolean getUserInfo() {
      mUser = new JTextField();
      if (lastUser != null)
          mUser.setText(lastUser);

      mDescription = new JTextField();
      JPanel panel = new JPanel(new GridLayout(2, 2));
      panel.add(new JLabel("Email"));
      panel.add(mUser);
      panel.add(new JLabel("Description of RDF submission"));
      panel.add(mDescription);
      int option =
        JOptionPane.showConfirmDialog(
          null,
          panel,
          "RDF Submit Information",
          JOptionPane.OK_CANCEL_OPTION,
          JOptionPane.PLAIN_MESSAGE);

      if (option == JOptionPane.OK_OPTION)
        return true;
      else
        return false;
    } // getUserInfo

    private String clean(String theString)
    {
        StringBuffer aFixed = new StringBuffer();
        for (int i = 0; i < theString.length(); i++)
        {
            int aVal = theString.charAt(i);
            Character aChar = new Character(theString.charAt(i));

            if (aVal == 0 && Character.isIdentifierIgnorable(aChar.charValue()) &&
                !Character.isLetterOrDigit(aChar.charValue()) && !Character.isWhitespace(aChar.charValue()))
            {
                System.err.println("removing char from submit output, it seems to be invalid: "+aChar);
                // invalid char, dont add it to the clean output
            }
            else aFixed.append(aChar);
        }

        return aFixed.toString();
    }

    /**
     * Submit the RDF to the specified url
     * @param uri String the url of the submit script
     * @return boolean true if the submission was successful, false otherwise
     * @throws IOException
     */
    public boolean submit(String uri) throws IOException {

        if (getUserInfo())
        {
            String encodedrdf = URLEncoder.encode(clean(rdfpayload), "UTF-8");

            URL url = new URL(uri);
            URLConnection connection = url.openConnection();
            connection.setDoOutput(true);

            setDescription(mDescription.getText());
            addCreator(mUser.getText());

            lastUser = mUser.getText();

            PrintWriter out = new PrintWriter(connection.getOutputStream());
            out.println("rdfblob=" + encodedrdf);
            writeCreators(out);
            writeCategories(out);
            writeName(out);
            writeDescription(out);
            writeDate(out);
            out.println("&inputtype=1");
            out.println("&op=Submit");
            out.close();

            return doSubmit(connection,rdfpayload);
            //new ProgressFrame(null, "Submitting RDF...", "Submitting RDF data to " + url, new OpenConnectionTask(connection, model, rdfpayload), false);
        }
        else {
            JOptionPane.showMessageDialog(null,"Submit cannot be completed without user information, please try again.", "User Info Error", JOptionPane.ERROR_MESSAGE);
            return false;
        }
    }

    private boolean doSubmit(URLConnection theConn, String theRDF)
    {
        boolean success = false;
        try {
            theConn.connect();
            InputStream is = theConn.getInputStream();

            BufferedReader in = new BufferedReader(new InputStreamReader(is));

            String output = "";
            String line;

            while ( (line = in.readLine()) != null) {
                output += line;
                if (DEBUG)
                    System.err.println(line);
            }
            if (output.indexOf("Success") != -1){

                //Update the the instances to reflect the uri in which they were stored on the remote server
                theConn.connect();
                String loc = theConn.getHeaderField("Location");

                success = true;
            }
            else  {
                success = false;
            }

            is.close();
        } catch (Exception ex) {
            ex.printStackTrace();
            success = false;
        }

        return success;
    }
}
