// The MIT License
//
// Copyright (c) 2005 Michael Grove
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.

package org.eaglei.datatools.etl.utils;

/**
 * <p>Title: TypeValidator</p>
 *
 * <p>Description: A simple utility class for checking to see if data represented in a String object is in the correct format for the specified datatype and is valid data.  For example, if you were checking to see if the string "foobar" was a valid integer, you would get false.  But it would be a valid string.  The datatypes used are XSD datatypes.</p>
 *
 * <p>Copyright: Copyright (c) 2006</p>
 *
 * <p>Company: Mindswap (http://www.mindswap.org)</p>
 *
 * @author Michael Grove
 * @version 1.0
 */
public class TypeValidator
{
  public static boolean isValidData(String dataType, String data)
  {
      // Notes:
      //   * I'm treating decimals as floats, is that ok?
      //   * Need validation routines for binary (0 or 1?), time and date

      // need to validate:
      // duration, time, dateTime
      // gYear, gYearMonth, gDay, gMonthDay, gMonth
      // hexBinary, base64Binary
      // QName, NOTATION
      //
      // token, language
      // NMTOKEN, NMTOKENS, Name, NCName, ID, IDREF, IDREFS
      // ENTITY, ENTITIES

      // empty data always considered valid, as is a string which can contain anything
      if (data.equals("") || dataType.endsWith("string"))
          return true;
      else if (dataType.endsWith("normalizedString"))
          return data.indexOf("\t") == -1 && data.indexOf("\n") == -1 && data.indexOf("\r") == -1;
      else if (dataType.endsWith("boolean"))
      {
          if (data.equalsIgnoreCase("true") || data.equalsIgnoreCase("false"))
              return true;
          else return false;
      }
      else
      {
          // now we handle the parsing and validation of all numeric formats
          // we'll try to parse them by what they claim to be
          // an exception being thrown means that it was incorrectly formatted
          try {
              if (dataType.endsWith("integer"))
                  Integer.parseInt(data);
              else if (dataType.endsWith("positiveInteger") || dataType.endsWith("unsignedInt") || dataType.endsWith("nonNegativeInteger"))
              {
                  if (data.indexOf("-") != -1) // make sure the negative sign is not there
                      return false;
                  else Integer.parseInt(data);
              }
              else if (dataType.endsWith("nonPositiveInteger") || dataType.endsWith("negativeInteger"))
              {
                  if (data.indexOf("-") == -1) // make sure it has the negative sign and is indeed a negative number
                      return false;
                  else Integer.parseInt(data);
              }
              else if (dataType.endsWith("float") || dataType.endsWith("decimal"))
                  Float.parseFloat(data);
              else if (dataType.endsWith("long") || dataType.endsWith("int"))
                  Long.parseLong(data);
              else if (dataType.endsWith("unsignedLong"))
              {
                  if (data.indexOf("-") != -1) // negative sign not allowed...
                      return false;
                  else Long.parseLong(data);
              }
              else if (dataType.endsWith("byte"))
              {
                  Byte.parseByte(data);
              }
              else if (dataType.endsWith("unsignedByte"))
              {
                  if (data.indexOf("-") != -1) // negative sign not allowed...
                      return false;
                  else Byte.parseByte(data);
              }
              else if (dataType.endsWith("short"))
                  Short.parseShort(data);
              else if (dataType.endsWith("unsignedShort"))
              {
                  if (data.indexOf("-") != -1) // negative sign not allowed...
                      return false;
                  else Short.parseShort(data);
              }
              else if (dataType.endsWith("double"))
                  Double.parseDouble(data);
              else if (dataType.endsWith("timeDuration"))
              {
                  return data.startsWith("P");
              }
              else if (dataType.endsWith("uriReference") || dataType.endsWith("anyURI"))
                  return data.startsWith("http://") || data.startsWith("www.");
              else if (dataType.endsWith("date"))
              {
                  java.text.DateFormat df = new java.text.SimpleDateFormat("MM/DD/yyyy");
                  df.setLenient(true);
                  df.parse(data);
              }
              return true;
          }
          catch (Exception e) { return false; }
      }
  }
}
