/**
 * eagle-i Project
 * Harvard University
 * Jan 27, 2010
 */
package org.eaglei.datatools.interim.cores;

import static org.eaglei.datatools.interim.cores.Cores3Mappings.*;
import static org.eaglei.datatools.model.DataToolsOntConstants.*;
import static org.eaglei.datatools.interim.cores.UniversityConstants.*;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EagleIOntConstants;
import org.eaglei.model.jena.EagleIOntDocumentManager;
import org.eaglei.model.jena.EagleIOntUtils;

import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.ResIterator;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;
import com.hp.hpl.jena.vocabulary.OWL;
import com.hp.hpl.jena.vocabulary.RDF;
import com.hp.hpl.jena.vocabulary.RDFS;

//TODO revise exception throwing vs ignoring missing or incorrect values (as we are processing incomplete entry forms)

/**
 * @author Daniela Bourges-Waldegg
 *
 * This class Wraps a Jena Model containing all resource instances associated to one lab, as captured in an
 * Excel annotation form Cores V3
 */

public class Cores3Model {
	private static final Log logger = LogFactory.getLog(Cores3Model.class);
	//Model to hold the generated RDF
	private Model model;
	//OntModel for the eagle-i ontology
	private final OntModel ontModel = 
		EagleIOntDocumentManager.INSTANCE.getOntModel(EAGLE_I_DT_URI);
	private String university;
	private final Property directType;
	public Cores3Model() {
		model = ModelFactory.createDefaultModel();
		model.setNsPrefix("eagle-i", EAGLE_I_URI);
		model.setNsPrefix("eagle-i-dt-ont", EAGLE_I_DT_URI);

		//TODO: set other prefixes
		directType = model.createProperty(EagleIOntConstants.DIRECT_TYPE_URI);
	}
	

	public void createResourceInstance(Mapping mapping, Map<String,String> columns) {
		if(university == null)
			university = EAGLE_I_SITES.get(columns.get("UNIVERSITY"));
		Resource resourceInstance = createInstance();
		setSubclassesAndProperties(mapping, resourceInstance, columns);
	}
	
	//TODO need to incorporate these exceptions
	
/*	public void createInstrumentResourceInstance(Map<String,String> columns) {

		//Exceptions
		//Manufacturer URL needs to be attached to manufacturer resource, not to instrument
		String manufacturerUrl = columns.remove("MANUFACTURER_URL");
		//Generic cases - need to do these first in order to create all nodes
		setSubclassesAndProperties(instrumentClass, resourceInstance, columns);
		//Continue with exceptions
		//if manufacturer url is null, it was skipped in the spreadsheet; no big deal
		if(manufacturerUrl != null) {
			//get manufacturer resource instance set in previous step
			OntProperty op = ontModel.getOntProperty(idMap.get("MANUFACTURER"));
			Statement ms = resourceInstance.getProperty(op);
			Resource manufacturer=null;
			if(ms != null)
				 manufacturer = (Resource)ms.getObject();		
			//set its URL
			if(manufacturer != null) {
				OntProperty mp = ontModel.getOntProperty(idMap.get("MANUFACTURER_URL"));
				model.add(model.createStatement(manufacturer, mp, manufacturerUrl));
			}
		}
	}*/
	

/*	public void createSoftwareResourceInstance(Map<String,String> columns){
		//Exceptions: input data and processing objective
		String inputDataLabel = columns.remove("INPUT_DATA");
		//If input data is null it was skipped in the spreadsheet; no big deal
		if(inputDataLabel != null) {
			OntProperty op = ontModel.getOntProperty(idMap.get("INPUT_DATA"));
			OntClass dataclass = op.getRange().asClass();
			connectToType(resourceInstance, dataclass, idMap.get("INPUT_DATA"), inputDataLabel);
		}
		String processingObjective = columns.remove("DATA_PROCESSING_OBJECTIVE");
		//If processing objective is null it was skipped in the spreadsheet; no big deal
		if(inputDataLabel != null) {
			OntProperty op = ontModel.getOntProperty(idMap.get("DATA_PROCESSING_OBJECTIVE"));
			OntClass dataclass = op.getRange().asClass();
			connectToType(resourceInstance, dataclass, idMap.get("DATA_PROCESSING_OBJECTIVE"), processingObjective);
		}		
		//Generic cases
		setSubclassesAndProperties(softwareClass, resourceInstance, columns);
	}*/
	

	
	
	public Model getModel(){
		return model;
	}			


	
	/**
	 * Create a resource instance with a type, equivalent to a row in the table
	 * @param uri the class of the resource - if not in the eagle-i ontology, the class is created in an Additional namespace
	 * @return
	 */
	
	private Resource createInstance(String classUri) {
			Resource r = createInstance();
			setType(r, classUri);
			return r;
		}
	
	private Resource createInstance() {
		//TODO change instance ID generation to eagle-i official mechanism
		String uri = "http://" + university + EAGLE_I_DATA_URI_FRAGMENT + "INSTANCE_" 
		+ UUID.randomUUID().toString();
		if(logger.isDebugEnabled()) logger.debug("Creating resource with URI= " + uri); 
		final Resource resourceInstance = model.createResource(uri);
		return resourceInstance;
	}
	
	/**
	 * Set the RDF:type of a resource
	 * @param r
	 * @param classUri
	 */
	private void setType(Resource r, String classUri) {
		OntClass resourceClass = ontModel.getOntClass(classUri);
		if (resourceClass != null)
			model.add(model.createStatement(r, RDF.type, resourceClass));
		else
			model.add(model.createStatement(r, RDF.type,
					model.createResource(classUri)));
	}
	
	private void setDirectType(Resource r, String classUri) {
		OntClass resourceClass = ontModel.getOntClass(classUri);
		if (resourceClass != null)
			model.add(model.createStatement(r, directType, resourceClass));
		//Set only for Eagle-i resources
		//else
			//model.add(model.createStatement(r, directType,
					//model.createResource(classUri)));
	}
	
	/**
	 * Processing that can be generalized across resources; columns map contains only those
	 * columns in spreadsheet that have uniform mapping (column is a property or subclass of resource)
	 * @param classUri
	 * @param resourceInstance
	 * @param columns
	 */
	private void setSubclassesAndProperties(Mapping mapping, 
			Resource resourceInstance, Map<String,String> columns) {
		if(logger.isDebugEnabled())logger.debug("Resource of class: " + mapping.getTargetUri());
		OntClass resourceClass = ontModel.getOntClass(mapping.getTargetUri());
		for(Map.Entry<String, String> entry : columns.entrySet()){
			String key = entry.getKey();
			String value = entry.getValue();
			String targetUri = mapping.getTargetUri(key);	
			if(logger.isDebugEnabled())logger.debug("Processing column: " + key +", for property: " + targetUri + " with value: " + value);
			if(targetUri == null) {
				// a Property that is not in the loaded ontology
				// all the non-ontology properties are literals
				Property p = model.createProperty(NON_ONTOLOGY_URI, key.toLowerCase());
				model.add(model.createStatement(resourceInstance, p, value));
				continue;
			}
			//Type?
			if (RDF.type.getURI().equals(targetUri)) {
				//Attempt to match a subclass label with value
				OntClass subclass = findSubclass(resourceClass, value, false);
				//If subclass is null it was incorrectly entered in the spreadsheet; no big deal
				//TODO move to setType
				if(subclass != null) {
					if(logger.isDebugEnabled())logger.debug("Setting subtype to: " + subclass.getURI());
					model.add(model.createStatement(resourceInstance, RDF.type, subclass));	
					//TODO see if we will need DirectType in the future
					//model.add(model.createStatement(resourceInstance, directType, subclass));
					//materializeTypes(resourceInstance, subclass);
				}
				else {
					if(logger.isDebugEnabled()) logger.debug("A subclass of "
							+ resourceClass.getLabel(null) +" with label " 
							+ value + " was not found in the eagle-i ontology. Doing nothing");		
				}
				continue;		
			}
			// Label?
			if(RDFS.label.getURI().equals(targetUri)) {
				if(logger.isDebugEnabled())logger.debug("Setting label to: " + value);
				model.add(model.createStatement(resourceInstance, RDFS.label, value));
				continue;
			}

			//All other ontology properties
			OntProperty o = ontModel.getOntProperty(targetUri);
			//TODO - revise this exception, maybe just silently ignore
			if (o==null)
				continue;
				//throw new GeneratorException("a Property with URI " + targetUri + " was not found in the eagle-i ontology");
			String[] values = value.split(";");
			if(o.isDatatypeProperty() || o.isAnnotationProperty()) {
				if(logger.isDebugEnabled())logger.debug("Processing Datatype/Annotation property");
				for(String v : values)
					model.add(model.createStatement(resourceInstance, o, v.trim()));
				continue;
			} else if (o.isObjectProperty()){
				if(logger.isDebugEnabled())logger.debug("Processing Object property");
				String t = mapping.getTargetRangeURI(key);
				logger.debug("URI= " + t);
				OntResource or = ontModel.getOntClass(t);
				for(String v : values) {
					Resource r = findInstanceByLabel(or.getURI(), v.trim());
					if (r == null) {
						r = createInstance(or.getURI());
						//TODO do I need to set direct Type?
						//TODO need to set values of the resource's properties instead of label
						model.add(model.createStatement(r, RDFS.label, v.trim()));
					}
					model.add(model.createStatement(resourceInstance,o,r));
				}
				continue;
			} else {
				if(logger.isDebugEnabled())logger.debug("Property is not defined in OWL");
			}
		}//for

		//Check if type info was not found in source data; use type as label if label was left empty
		if(resourceInstance.getProperty(RDF.type) == null && resourceClass != null) {
			setType(resourceInstance, resourceClass.getURI());
			//setDirectType(resourceInstance, resourceClass.getURI());
			if(resourceInstance.getProperty(RDFS.label) == null)
				setTypeAsLabel(resourceInstance, resourceClass);
		}		
	}

	/**
	 * Find a subclass of resourceClass, with label subclassLabel
	 * @param resourceClass
	 * @param subclassLabel
	 * @param direct -if true, only direct subclasses are returned
	 * @return
	 */
	private OntClass findSubclass(OntClass resourceClass, String subclassLabel, boolean direct){
		if (subclassLabel == null)
			return null;
		final ExtendedIterator<OntClass> extIt = resourceClass.listSubClasses(direct);
		while (extIt.hasNext()) {
			OntClass subclass = extIt.next();
			//TODO this might not happen if the subclass names in the form do not match the labels
			if(subclassLabel.equalsIgnoreCase(subclass.getLabel(null))) {
				return subclass;
			}		
		} 
		return null;
	}

	/**
	 * Connect a resource to a (sub)class,  via a property with propertyUri
	 * @param r
	 * @param rootClassUri
	 * @param propertyUri
	 * @param subclassLabel
	 */
	private void connectToType(Resource r, String rootClassUri, String propertyUri, String subclassLabel) {
		OntClass rootClass = ontModel.getOntClass(rootClassUri);
		connectToType(r, rootClass, propertyUri, subclassLabel);
	}
	
	/**
	 * Connect a resource to a (sub)class,  via a property with propertyUri
	 * @param r -  the resource
	 * @param classUri -  root of the range class
	 * @param propertyUri -  property to use
	 * @param subclassLabel - type label of range class
	 */
	private void connectToType(Resource r, OntClass rootClass, String propertyUri, String subclassLabel) {
		if(subclassLabel!= null) {
			OntProperty op = ontModel.getOntProperty(propertyUri);
			//TODO check if range matches
			//find subclass in ontclass; need deep search, so direct is set to false
			//if subclass is not in ontology, enter it as a non-ontology term
			OntClass subclass = findSubclass(rootClass, subclassLabel, false);
			if (subclass != null)
				model.add(model.createStatement(r, op, subclass));
			else {
				if(logger.isDebugEnabled()) logger.debug("A subclass of " 
						+ rootClass.getLabel(null) + " with label " 
						+ subclassLabel + " was not found in the eagle-i ontology. Creating a resource in unclassified namespace.");
				//TODO replace when we know what to do
				Resource ur = model.createResource(NON_ONTOLOGY_URI + subclassLabel.replaceAll("\\s{1,}", "_"));
				model.add(model.createStatement(ur, RDF.type, rootClass));
				model.add(model.createStatement(ur, directType, rootClass));
				model.add(model.createStatement(ur, RDFS.label, subclassLabel));
				model.add(model.createStatement(r, op, ur));	
			}
		}
	}

	private Resource findInstanceByLabel(String targetClass, String targetLabel) {
		ResIterator it = model.listResourcesWithProperty(RDF.type, ontModel.getOntClass(targetClass));
		while (it.hasNext()) {
			final Resource tr = it.next();
			final String label = tr.getProperty(RDFS.label).getString();
			if (label.equalsIgnoreCase(targetLabel)) {
				return tr;
			}		
		}
		return null;
	}
	
	/**
	 * Connect a resource to an instance of a targetClass, via a property with propertyUri
	 * @param r
	 * @param targetClassUri
	 * @param propertyUri
	 * @param targetLabel
	 */
	private void connectToInstance(Resource r, String targetClassUri, String propertyUri, String targetLabel) {
		Property p = ontModel.getOntProperty(propertyUri);
		//TODO check if range matches
		Resource tr = findInstanceByLabel(targetClassUri, targetLabel);
		if (tr != null) {
				model.add(model.createStatement(r, p, tr));		
		} else {
			//if instance wasn't found, create one 
			if(p == null)
				p = model.createProperty(propertyUri);
			tr = createInstance(targetClassUri);
			model.add(model.createStatement(tr, RDFS.label, targetLabel));
			model.add(model.createStatement(r, p, tr));
		}		
	}
	
	/**
	 * @param r
	 * @param columns
	 */

	private void materializeTypes(Resource resourceInstance, OntClass ontClass) {
        // get all of the parent classes
        final ExtendedIterator extIt = ontClass.listSuperClasses();
        while (extIt.hasNext()) {
            OntClass parent = (OntClass) extIt.next();
            if (parent.getURI() == null) {
                // Some reasoners return resources with null
                // URIs in the superclass list
                continue;
            }
            // don't add Thing
            if (parent.getURI().equals(OWL.Thing.getURI())) {
                continue;
            }
            if (parent.getURI().equals(RDFS.Resource.getURI())) {
                // Some reasoners return RDFS.Resource in superclass list.
                continue;
            }
            model.add(model.createStatement(resourceInstance, RDF.type, parent));
        }

	}
	
	private void setTypeAsLabel(Resource resourceInstance, OntClass ontClass) {
		if(resourceInstance.getProperty(RDFS.label) == null) {
			model.add(model.createStatement(resourceInstance, RDFS.label, ontClass.getLabel(null)));
			if(logger.isDebugEnabled()) logger.debug("Label is empty; setting it to type label instead");
		}
	}	
}