/**
 * eagle-i Project
 * Harvard University
 * Feb 18, 2010
 */
package org.eaglei.datatools.interim.cores;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.hp.hpl.jena.rdf.model.Model;

/**
 * @author Daniela Bourges-Waldegg
 *
 */
public class Cores3RDFGenerator {
	private static final Log logger = LogFactory.getLog(Cores3RDFGenerator.class);
	
	//Directories
	public static void main(String[] args) throws Exception {
		if (args.length <= 1 || args[0].length() == 0 || args[1].length() == 0) {
			logger.error("Cores3RDFGenerator <Excel file> <output dir> [RDF/XML | N-TRIPLE]");
			return;
		}
		String format = "RDF/XML";
		//defaults to RDF/XML
		if(args.length >2 && args[2] != null && args[2].length() > 0)
			if(!"RDF/XML".equals(args[2]) && !"N-TRIPLE".equals(args[2]))
				logger.error("Unsupported format: " + args[2]);
			else
				format = args[2];
		generateToFile(args[0], args[1], format);
	}

	public static Model generateToModel(String excel) throws IOException {
		if (!excel.endsWith(".xlsx") || excel.startsWith("~$")) {
			throw new IllegalArgumentException("Expecting an Excel file");
		}
		//final File excelFile = new File(excel);
		if(logger.isDebugEnabled()) {
			logger.debug("Reading Excel file: " + excel);
		}

		final Cores3ExcelParser parser = new Cores3ExcelParser(excel);
		//This wraps a Model for one core, represented by one excel file
		final Cores3Model core = new Cores3Model();
		List<Cores3ExcelParser.ProcessedTab> processedTabs = parser.getProcessedTabs();
		for(Cores3ExcelParser.ProcessedTab p : processedTabs) {
			List<Map<String, String>> tabData = p.getTabData();
			for(Map<String, String> columns : tabData)
				core.createResourceInstance(p.getClassMapping(), columns);
		}
		return core.getModel();
	}

	public static void generateToFile(String excel, String outputPath, String format) throws IOException {
		final Model generatedModel = generateToModel(excel);
		final File outputDir= new File(outputPath);
		// check if directory exists; create if it doesn't
		OldFileUtils.validateDirectory(outputDir, true);
		if(logger.isDebugEnabled()) {
			logger.debug("Writing RDF data to directory: " + outputDir.getAbsolutePath());
		}
		
		String extension=null;
		if("RDF/XML".equals(format))
			extension = ".rdf";
		else if("N-TRIPLE".equals(format))
			extension = ".ntriple";
		
		final File excelFile = new File(excel);
		int i = excelFile.getName().lastIndexOf(".");

		final String rdfFile;
		if(i>0)
			rdfFile = excelFile.getName().substring(0,i) + extension;
		else
			rdfFile = excelFile.getName() + extension;

		final OutputStream os = new FileOutputStream(new File(outputDir, rdfFile));
		try {
			generatedModel.write(os, format);
			os.flush();
		} finally {
			os.close();
		}
		logger.info("Done!");
	}
}


