/**
 * eagle-i Project
 * Harvard University
 * Apr 23, 2010
 */
package org.eaglei.datatools.jena;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIDatatypeProperty;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIOntModel;

import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.SimpleSelector;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.vocabulary.RDF;
import com.hp.hpl.jena.vocabulary.RDFS;

/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 */
 
 
public class EIInstanceFactory {
	private static final Log logger = LogFactory
			.getLog(EIInstanceFactory.class);
	
	public static final EIInstanceFactory INSTANCE = new EIInstanceFactory();

	private OntModel ontModel = JenaEIOntModel.INSTANCE.getOntModel();
    public static EIInstanceFactory getInstance()
    {
        return INSTANCE;
    }


    private  EIInstanceFactory()
    {
    }

    
    /**
     * Serialize an EInstance into an RDF language
     * @param instance
     * @param lang - one of the RDF languages supported by Jena; predefined values are 
     * "RDF/XML", "RDF/XML-ABBREV", "N-TRIPLE", "TURTLE", (and "TTL") and "N3". The default value, represented by null is "RDF/XML".
     * @return - the serialized EIInstance as a STring
     */
    
    public String serialize(final EIInstance instance, final String lang)   {
    	if(instance == null)
    		return null;
    	Model model = convertToJenaModel(instance);
        //Serialize model to String in specified format
        StringWriter sw = new StringWriter();
        try {
        	//TODO validate format
        	model.write(sw, lang);
        	String s = sw.toString();
        	sw.flush();
        	return s;
        } finally {
        	try {if(sw != null) sw.close(); } catch (IOException e){/*not much to do*/}
        }      
    	
    }
    public Model convertToJenaModel(final EIInstance instance)   {
    	if(instance == null)
    		return null;
        Model model = ModelFactory.createDefaultModel();
        //Create the resource that will be the subject of all statements
        Resource resourceInstance = model.createResource(instance.getInstanceURI().toString());

        //Set its type and label
        Resource type = model.createResource(instance.getInstanceClass().getEntity().getURI().toString());
        model.add(model.createStatement(resourceInstance, RDF.type, type));
        model.add(model.createStatement(resourceInstance, RDFS.label, instance.getInstanceLabel()));

        Map<EIEntity, Set<String>> dataprop = instance.getDatatypeProperties();
        Map<EIEntity, Set<EIURI>> objectprop = instance.getObjectProperties();

        //Loop through the map of data/object properties and create appropriate Jena objects
        
        for(Map.Entry<EIEntity, Set<String>> entry : dataprop.entrySet())
        {
            Property p = model.createProperty(entry.getKey().getURI().toString());
            final Set<String> values = entry.getValue();
            for(String value : values)
            	model.add(model.createStatement(resourceInstance, p, value));
        }

        for(Map.Entry<EIEntity, Set<EIURI>> entry : objectprop.entrySet())
        {
            Property p = model.createProperty(entry.getKey().getURI().toString());
            final Set<EIURI> values = entry.getValue();
            for(EIURI value : values) {
            	Resource valueResource = model.createResource(value.toString());
            	model.add(model.createStatement(resourceInstance, p, valueResource));
            }
        }
        return model;
    }
  
    

    /**
     * Create an EIInstance from an RDF String serialization in one of the RDF languages supported by Jena
     * @param instanceUri - the instance uri if known (all the triples in the rdf parameter should have it as subject)
     * @param rdf 
     * @param lang
     * @return
     */
    
    //TODO deal with cases where the RDF does not represent an EIInstance
    
    public EIInstance create(final EIURI instanceUri, final String rdf, final String lang) {
    	if(instanceUri == null || instanceUri.toString().length() ==0 || rdf == null || rdf.length() == 0)
    		return null;
    	Model model = ModelFactory.createDefaultModel();
    	model.read(new StringReader(rdf), null, lang);
    	return create(instanceUri, model);
    }
    
    public static EIInstance createEmpty(final EIURI classUri, final EIEntity instanceEntity) {
    	if(classUri == null || classUri.toString().length() == 0 || instanceEntity == null) 
    		return null;
    	EIClass instanceClass = JenaEIOntModel.INSTANCE.getClass(classUri);
    	if( instanceClass == null)
    		return null;
    	return new EIInstance(instanceClass, instanceEntity);
    }
    
    public EIInstance create(final EIURI instanceUri, final Model model)
    {
    	if(instanceUri == null || model == null)
    		return null;
    	final Resource subject = model.getResource(instanceUri.toString());
    	//Get resource will create a resource if one doens't exist already, so need to check
    	if(!model.contains(subject, null, (RDFNode)null))
    		return null;
    	//Instances can have multiple types; the superclass should go in the EIInstance
    	//Get all the types and find the superclass to put in instance
    	
    	final List<Statement> typeStatements = model.listStatements(subject, RDF.type, (Resource)null).toList();
    	EIClass instanceClass = null;
    	Resource instanceType=null;
    	if(typeStatements.size() > 1) {
    		if(logger.isDebugEnabled())logger.debug("Found more than one type: " + typeStatements.size());
    		for(Statement st : typeStatements) {
    			final Resource type = (Resource)st.getObject();
    			final EIURI typeUri = EIURI.create(type.getURI());
    			final EIClass eiClass = JenaEIOntModel.INSTANCE.getClass(typeUri);
    			if(eiClass!=null && !eiClass.hasSuperClass()) {
    				instanceClass = eiClass;
    				instanceType = type;
    				break;
    			}
    		}
    	} else {
    		final Statement typeStatement = subject.getProperty(RDF.type);
    		if(typeStatement == null) {
    			if(logger.isDebugEnabled())logger.debug("Type is not set for instance in model");
    			return null;
    		}
    		instanceType = typeStatement.getResource();
    		if(instanceType == null || instanceType.getURI() == null) {
       			if(logger.isDebugEnabled())logger.debug("Type is not correctly set for instance in model");
    			return null;
    		}
    		instanceClass = JenaEIOntModel.INSTANCE.getClass(EIURI.create(instanceType.getURI()));
    	}
    	
    	//Nothing in the received model matches an EIClass
    	if(instanceClass == null) {
    		if(logger.isDebugEnabled())logger.debug("No EIClass found in the model");
    		return null;
    	}
    		
    	final EIURI instanceClassUri = instanceClass.getEntity().getURI();
    	final Statement labelStatement = subject.getProperty(RDFS.label);
    	if(labelStatement == null) {
    		if(logger.isDebugEnabled())logger.debug("RDFS Label is not set for instance in RDF");
    		return null;
    	}
        final String instanceLabel = labelStatement.getString();
        if(logger.isDebugEnabled()) logger.debug("Creating an instance of class: " + instanceClassUri.toString() + " with URI: " + instanceUri + " and label: " + instanceLabel);

        // The entity and classes of our new EIInstance
        final EIEntity instanceEntity = EIEntity.create(instanceUri, instanceLabel);

        // create IEInstance
        final EIInstance ei = new EIInstance(instanceClass, instanceEntity);

        // query the model for all statements about this subject
        final StmtIterator iter = model.listStatements(subject, null, (RDFNode) null);
        if(logger.isDebugEnabled()) logger.debug("Searching for statements with subject: " + instanceUri.toString() + " of class: " + instanceClassUri + " and label: " + instanceLabel);
       
        while(iter.hasNext())
        {
            final Statement statement = iter.nextStatement();
            final Property predicate = statement.getPredicate();
            //No need to process label, as it was done previously
            if(predicate.equals(RDFS.label))
            	continue;
            
            final EIURI propertyUri = EIURI.create(predicate.getURI());
            final RDFNode o = statement.getObject();
            if(predicate.equals(RDF.type) && o.isResource() && ((Resource)o).equals(instanceType))
            	continue;

            // Check if property is in ontology

            EIProperty p = getEIOntProperty(instanceClassUri, propertyUri);
            if(p!= null) {
            	final String preferredLabel = JenaEIOntModel.INSTANCE.getPreferredLabel(propertyUri);
            	final EIEntity propEntity = EIEntity.create(propertyUri, preferredLabel);
            	if(p instanceof EIDatatypeProperty) {
            		final String value = o.toString();
            		if(logger.isDebugEnabled())logger.debug("adding property: [" + propertyUri + "] with literal value : [" + value + "] ");
            		ei.addDatattypePropertyToInstance(propEntity, value);
            	} else if (p instanceof EIObjectProperty){
            		final EIURI objectUri = EIURI.create(((Resource)o).getURI());
            		ei.addObjectPropertyToInstance(propEntity, objectUri);
            		if(logger.isDebugEnabled())logger.debug("adding property: [" + propertyUri + "] with object value : [" + objectUri + "] ");
            	}
            } else {
            	// Property is not in ontology; use its local anme as label
            	final String predicateLabel = predicate.getLocalName();
            	final EIEntity propEntity = EIEntity.create(propertyUri, predicateLabel);
            	if(logger.isDebugEnabled()) logger.debug("Property not found in ontology; we'll still try to add it: " + propertyUri);
            	if(o.isLiteral())
            		ei.addNonOntologyLiteralProperty(propEntity, o.toString());
            	else if (o.isResource())
            		ei.addNonOntologyResourceProperty(propEntity, EIURI.create(((Resource)o).getURI()));
            	else
            		if(logger.isDebugEnabled()) logger.debug("Could not add property " + propertyUri);
            }
        }
        return ei;
    }
    
    /**
     * Batch creation of a list of EIInstances from a model
     * @param model
     * @return
     */
    public List<EIInstance> create(final Model model) {
    	if(model == null)
    		return Collections.EMPTY_LIST;
    	final List<EIInstance> instances = new ArrayList<EIInstance>();
    	final Set<Resource> subjects = model.listSubjects().toSet();
		//create an EIInstance per subject
		for(Resource r : subjects) {
			final Model results = model.query(new SimpleSelector(r, null, (RDFNode)null));			
			EIInstance ei = create(EIURI.create(r.getURI()), results);
			if(ei != null)
				instances.add(ei);
		}
		return instances;
    }
    
    /**
     * Find a property for a class in the eagle-i ontology.  
     * @param instanceClassUri
     * @param propertyUri
     * @return the property if found, null otherwise
     */
    private EIProperty getEIOntProperty(final EIURI instanceClassUri, final EIURI propertyUri) {
    	if(instanceClassUri == null || propertyUri == null) 
    		return null;

    	final List<EIProperty> properties = JenaEIOntModel.INSTANCE.getProperties(instanceClassUri);
    	if(properties == null || properties.isEmpty())
    		return null;
    	for (EIProperty p : properties) {
    		final EIURI propUri = p.getEntity().getURI();
    		if(propUri.equals(propertyUri))
    			return p;   		
    	}
    	//If property wasn't found in domain
    	return null;
    }
}
