package org.eaglei.datatools.jena;

import static org.eaglei.datatools.model.DataToolsOntConstants.DC_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.RDFS_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.RDF_URI;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringBufferInputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIOntModel;

import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFormatter;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;

import org.eaglei.datatools.jena.SPARQLQueryUtil;


/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * 
 * 
 *       In-memory stub repository containing sample RDF data for the purposes
 *       of testing
 */

public final class MemoryRepositoryProvider implements RepositoryProvider
{
    private static final Log logger = LogFactory.getLog(MemoryRepositoryProvider.class);
    private static final String DEFAULT_USER = "eaglei";
    private static final String DEFAULT_SESSION = "valid_sessions";
    private Model model = ModelFactory.createDefaultModel();
    

    public MemoryRepositoryProvider(final InputStream is)
    {


        populateFakeDataModel(is);

    }

    private void populateFakeDataModel(final InputStream is)
    {

        try
        {
            if(logger.isDebugEnabled())
            {
                logger.debug("Reading data file....");
            }
            model.read(is, null, "N-TRIPLE");
        }
        finally
        {
            try
            {
                if(is != null)
                {
                    is.close();
                }
            }
            catch(final Exception e)
            {
                if(logger.isDebugEnabled())
                {
                    logger.debug("Error reading data file: " + e);
                }
            }
        }

        if(logger.isDebugEnabled())
        {
            logger.debug("populated in memory repo...");
        }
    }

    @Override
    public String updateInstance(final EIInstance instance, String token) throws Exception
    {
        if(instance == null)
        {
            return null;           
            
        }
        
        // this means the user is requesting a token for this instance
        
        if(token == null) {
            return UUID.randomUUID().toString();
            
        }
        
        final Model receivedModel = EIInstanceFactory.INSTANCE.convertToJenaModel(instance);
        if(receivedModel == null)
        {
            throw new Exception("updateInstance: model is null");
        }

        // Delete instance related statements from model so we don't end up with
        // duplicate properties
        final Resource subject = model.createResource(instance.getEntity().getURI().toString());
        model.remove(model.listStatements(subject, null, (RDFNode) null));

        // add new statements to model
        final List<Statement> statements = receivedModel.listStatements().toList();
        model.add(statements);
        final StringWriter sw = new StringWriter();
        try
        {
            // TODO validate format
            model.write(sw, "RDF/XML");
            final StringBuffer s = sw.getBuffer();
            sw.flush();
            logger.debug("updateInstance: RDF/XML size: " + s.length());

        }
        finally
        {
            try
            {
                if(sw != null)
                {
                    sw.close();
                }
            }
            catch(final IOException e)
            {
                logger.error("updateInstance: problem updating: " + e);
                throw e;
            }
        }
        return "true";
        
    }

    @Override
    public EIInstance getInstance(final EIURI uri) throws Exception
    {
        final Resource r = model.getResource(uri.toString());
        // resource wasn't found in repo
        if(r == null)
        {
            return null;
        }
        final List<Statement> statements = model.listStatements(r, null, (RDFNode) null).toList();
        final Model results = ModelFactory.createDefaultModel();
        for(final Statement s : statements)
        {
            results.add(s);
        }
        return EIInstanceFactory.INSTANCE.create(EIURI.create(r.getURI()), results);
    }


    @Override
    public List<EIInstance> getAllEIResourcesForUser(final String user)
    {      
        return SPARQLQueryUtil.getInstance().getAllEIResourcesForUser(user, model);
    }

    /**
     * @param rnav
     * @param classUri
     * @return
     */
    @Override
    public List<EIInstance> getUserInstances(final String user, final EIURI classUri)
    {
        return SPARQLQueryUtil.getInstance().getUserInstances(user,classUri, model);
    }

    @Override
    public void createInstance(final EIInstance instance) throws Exception
    {
        // the same method in this implementation
        updateInstance(instance,"non-null token");

    }

    @Override
    public void deleteInstance(final EIInstance instance) throws Exception
    {
        // TODO: for testing purposes we should implement this if we have time

    }

    @Override
    public List<EIURI> getNewInstanceID(final int count) throws Exception
    {

        final List<EIURI> instances = new ArrayList<EIURI>(count);

        for(int i = 0; i < count; i++)
        {
            final String id = "http://harvard.data.eagle-i.org/i/" + UUID.randomUUID().toString();
            logger.debug("creating: " + id);
            instances.add(EIURI.create(id));
        }

        logger.debug("Created " + instances.size() + " ids");

        return instances;
    }

    @Override
    public EIInstance getEmptyEIInstance(EIURI classUri, EIEntity instanceEntity) throws Exception {

        return EIInstanceFactory.createEmpty(classUri, instanceEntity);
    }
    
    
    
    @Override
    public String[] login(final String user, final String password) throws Exception
    {
        // returns defaults user because memorymodel doesn't really have any
        // users

        return new String[] {DEFAULT_USER, DEFAULT_SESSION };
    }

    @Override
    public void logout() throws Exception
    {

    }

    @Override
    public String whoami() throws Exception
    {
        return DEFAULT_USER;
    }

    @Override
    public boolean isOnline()
    {
        boolean state =  model == null ? false : true;
        return state;
    }

    @Override
    public String query(String sparql) throws Exception
    {
        return SPARQLQueryUtil.getInstance().serializeQuery(sparql, model);
    }

    @Override
    public void uploadInstances(String rdf) throws Exception
    {
        assert rdf !=null;
        
        StringBufferInputStream in = new StringBufferInputStream(rdf);
        
        model.read(in,"RDF/XML");
        
    }

}
