package org.eaglei.datatools.jena;

import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.CreateInstance;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.GetInstance;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.GetNewInstances;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Logout;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.WhoAmI;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Query;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Online;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Upload;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.httpclient.Credentials;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIURI;


import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.query.ResultSetFormatter;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.sparql.util.ResultSetUtils;

/**
 * 
 * @author Ricardo De Lima
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 */

public final class RESTRepositoryProvider implements RepositoryProvider
{
    private static final Log log = LogFactory.getLog(RESTRepositoryProvider.class);
    private static String DEFAULT_REPOSITORY = "http://alaska.data.eagle-i.org:8080/";
    private static String DEFAULT_NAMESPACE = "http://alaska.data.eagle-i.org/i/";
    private static String DEFAULT_USER = "superuser";
    private static String DEFAULT_PASSWD = "superuser";

    enum RestCommands
    {
        GetNewInstances("repository/new"), WhoAmI("repository/whoami"), Logout("repository/logout"), GetInstance("i"), CreateInstance("repository/update"), Query("repository/sparql"), Online(""), Upload("repository/graph");

        private RestCommands(final String propKey)
        {
            key = propKey;
        }

        public String getURL()
        {

            return DEFAULT_REPOSITORY + key;
        }

        private final String key;

    }

    private ResultSet results;
    //TODO: fix this
    private UsernamePasswordCredentials defaultcreds = new UsernamePasswordCredentials("superuser", "superuser");

    /*
     * Private constructor, must access via static INSTANCE field
     */

    public RESTRepositoryProvider(DatatoolsConfiguration config)
    {

        if(config != null)
        {

            // TODO: at some point we need to get the auth information via
            // Sessions or other method

            DEFAULT_USER = config.getDefaultUser();
            DEFAULT_PASSWD = config.getDefaultPassword();
            DEFAULT_REPOSITORY = config.getDatatoolsRepositoryURL();
            DEFAULT_NAMESPACE = config.getDatatoolsRepositoryNamespace();
        }



    }

    private HttpClient getHttpClient() {
        HttpClient client = new HttpClient();
        client.getState().setCredentials(AuthScope.ANY, defaultcreds);
        client.getParams().setAuthenticationPreemptive(true);
        
        return client;
        
    }
    
    @Override
    public void createInstance(final EIInstance instance) throws Exception
    {
        assert instance != null;
        assert instance.getInstanceURI() != null;

        final String id = instance.getInstanceURI().toString();

        int status = 0;
        final PostMethod method = new PostMethod(CreateInstance.getURL());
        method.setParameter("uri", id);
        method.setParameter("format", "application/xml");
        method.setParameter("action", "create");
        method.setParameter("workspace", "http://eagle-i.org/ont/repo/1.0/NG_Published");
        // log.debug("dump data: " +
        // EIInstanceFactory.INSTANCE.serialize(instance, "RDF/XML"));
        method.setParameter("insert", EIInstanceFactory.INSTANCE.serialize(instance, "RDF/XML"));

        log.debug("Trying to get create instance at " + id);

        try
        {

            HttpClient client = getHttpClient();
            
            status = client.executeMethod(method);
            final String resp = getStringFromInputStream(method.getResponseBodyAsStream());

            if(status == HttpStatus.SC_OK)
            {
                log.info("get instance succeded with status: " + status + " response: " + resp);

            }
            else
            {
                log.error("create instance failed with status: " + status + " response: " + resp);

            }

        }
        catch(final Exception e)
        {
            log.error("problem creating instance " + id + " at URL  " + CreateInstance.getURL() + " because " + e);

        }
        finally
        {
            method.releaseConnection();
        }

    }

    @Override
    public void deleteInstance(final EIInstance instance) throws Exception
    {
        assert instance != null;
        assert instance.getInstanceURI() != null;
        assert instance.getInstanceURI().toString() != null;

        final String id = instance.getInstanceURI().toString();

        int status = 0;
        final PostMethod method = new PostMethod(CreateInstance.getURL());
        method.setParameter("uri", id);
        method.setParameter("format", "application/xml");
        method.setParameter("action", "update");
        method.setParameter("workspace", "http://eagle-i.org/ont/repo/1.0/NG_Published");
        // log.debug("dump data: " +
        // EIInstanceFactory.INSTANCE.serialize(instance, "RDF/XML"));
        method.setParameter("delete", EIInstanceFactory.INSTANCE.serialize(instance, "RDF/XML"));

        log.debug("Trying to get delete instance at " + id);

        try
        {
            HttpClient client = getHttpClient();
            status = client.executeMethod(method);
            final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
            if(status == HttpStatus.SC_OK)
            {
                log.info("delete instance succeded with status: " + status + " response: " + resp);

            }
            else
            {
                log.error("delete instance failed with status: " + status + " response: " + resp);

            }

        }
        catch(final Exception e)
        {
            log.error("problem deleting instance " + id + " at URL  " + CreateInstance.getURL() + " because " + e);

        }
        finally
        {
            method.releaseConnection();
        }

    }

    @Override
    public List<EIInstance> getAllEIResourcesForUser(final String rnav) throws Exception
    {

        log.debug("getAllEIResourcesForUser: " + rnav);

        String q = SPARQLQueryUtil.getInstance().getAllEIResourcesForUserQuery(rnav);

        // Now Post to SPARQL Endpoint

        String results = postQuery(q);


        // log.debug("getAllEIResourcesForUser results " + results);

        // Serialize the RDF into a List of EIInstances

        Model m = ModelFactory.createDefaultModel();
        m.read(new StringReader(results), "RDF/XML");

        // debugging:
        
//        StringWriter s = new StringWriter();
//        m.write(s,null);
//        log.debug("model dump: " + s);

        return EIInstanceFactory.INSTANCE.create(m);

    }

    private synchronized String postQuery(String q)
    {
        assert q != null;

        int status = 0;
        final PostMethod method = new PostMethod(Query.getURL());

        method.setParameter("format", "application/xml");
        method.setParameter("default-graph-uri", "http://eagle-i.org/ont/repo/1.0/NG_Published");
        method.setParameter("query", q);

        try
        {
            HttpClient client = getHttpClient();
            status = client.executeMethod(method);
            final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
            if(status == HttpStatus.SC_OK)
            {
                log.debug("query sparql endpoint succeded with status: " + status);
                return resp;

            }
            else
            {
                log.error("query sparql endpoint  failed with status: " + status);

            }

        }
        catch(final Exception e)
        {
            log.error("problem query sparql endpoint at URL  " + Query.getURL() + " because " + e);

        }
        finally
        {
            method.releaseConnection();
        }

        return null;
    }

    @Override
    public List<EIInstance> getUserInstances(final String rnav, final EIURI classUri) throws Exception
    {
        String q = SPARQLQueryUtil.getInstance().getUserInstancesQuery(rnav, classUri);

        log.debug("getUserInstances: " + rnav + " URI: " + classUri);

        // Now Post to SPARQL Endpoint

        String results = postQuery(q);

        Model m = ModelFactory.createDefaultModel();
        
        m.read(new StringReader(results), "RDF/XML");
        
        log.debug("getUserInstances results: " + results);

        // Serialize the RDF into a List of EIInstances

      // debugging:
        
        StringWriter s = new StringWriter();
        m.write(s,null);
        log.debug("model dump: " + s);
       


        return EIInstanceFactory.INSTANCE.create(m);
    }

    @Override
    public EIInstance getInstance(final EIURI instanceID) throws Exception
    {
        assert instanceID != null;
        assert instanceID.toString() != null;

        String id = null;

        try
        {
            id = instanceID.toString().split(DEFAULT_NAMESPACE)[1];
        }
        catch(Exception e)
        {
            log.error("Problem parsing the namespace: " + DEFAULT_NAMESPACE + " out of the EIRUI: " + instanceID.toString());
            return null;
        }

        int status = 0;
        final GetMethod method = new GetMethod(GetInstance.getURL() + "/" + id);
        HttpClient client = getHttpClient();
        client.getParams().setParameter("accept", "text/plain");

        log.debug("Trying to get instance " + GetInstance.getURL() + "/" + id);

        try
        {

            status = client.executeMethod(method);

            if(status == HttpStatus.SC_OK)
            {
                log.debug("reading in input stream (RDF/XML)as a model");

                // read the inputstream RDF/XML into EIInstanceFactory which
                // will return a EIInstance

                // TODO: change this method signature to only return one
                // EIInstance, there is no need
                // to return a list...
                InputStream in = method.getResponseBodyAsStream();
                StringWriter writer = new StringWriter();
                IOUtils.copy(in, writer);
                return EIInstanceFactory.INSTANCE.create(instanceID, writer.toString(), "RDF/XML");

            }
            else
            {
                log.error("get instance failed with status: " + status);
                return null;
            }

        }
        catch(final Exception e)
        {
            log.error("problem getting instance " + instanceID.toString() + " at URL  " + GetInstance.getURL() + " because " + e);
            return null;
        }
        finally
        {
            method.releaseConnection();
        }

    }

    @Override
    public List<EIURI> getNewInstanceID(final int count) throws Exception
    {
        int status = 0;
        ResultSet results = null;
        final PostMethod method = new PostMethod(GetNewInstances.getURL());
        final List<EIURI> instances = new ArrayList<EIURI>(count);

        log.info("getting " + count + " instance IDs");

        try
        {

            method.setParameter("accept", "application/sparql-results+xml");
            method.setParameter("count", Integer.toString(count));
            HttpClient client = getHttpClient();
            status = client.executeMethod(method);

            // We'll get this as a resource stream and convert into EIURIs

            //log.debug("RESPONSE: " + method.getResponseBodyAsString());

            if(status == HttpStatus.SC_OK)
            {

                final String response = getStringFromInputStream(method.getResponseBodyAsStream());
                results = ResultSetFactory.fromXML(response);
            }
            else
            { // TODO: check for an unathenticated session
                log.debug("Error getting instances HTTP Status: " + status);
                return null;
            }

        }
        catch(final Exception e)
        {
            log.error("problem obtaning a new instance from URL: " + GetNewInstances.getURL() + " " + e);
        }
        finally
        {
            method.releaseConnection();
        }

        for(; results.hasNext();)
        {
            final QuerySolution soln = results.nextSolution();

            final Resource r = soln.getResource("new");

            final String id = r.getURI();
            instances.add(EIURI.create(id));
            log.info("Created new instance id: " + id);
        }

        log.debug("Created " + instances.size() + " ids");

        return instances;

    }

    @Override
    public String[] login(final String user, final String password) throws Exception   
    {
        int status = 0;
        ResultSet results = null;
        final GetMethod method = new GetMethod(WhoAmI.getURL());
        HttpClient client = getHttpClient();
        client.getParams().setParameter("accept", "application/sparql-results+xml");

        client.getParams().setAuthenticationPreemptive(true);
        client.getState().setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(user, password));

        // now let's try to do a whoami if it succeeds then we are logged in
        // otherwise return null

        log.debug("Trying to login at " + WhoAmI.getURL() + " with username " + user);
        try
        {

            status = client.executeMethod(method);

            if(status == HttpStatus.SC_OK)
            {
                results = ResultSetFactory.fromXML(getStringFromInputStream(method.getResponseBodyAsStream()));
            }
            // in = method.getResponseBodyAsStream();
            // ResultSet results = ResultSetFactory.fromXML(in);

        }
        catch(final Exception e)
        {
            log.error("problem login into " + WhoAmI.getURL() + " " + e);
        }
        finally
        {
            method.releaseConnection();
        }

        // check to see if the status is UNATHENTICATED

        if(status != HttpStatus.SC_UNAUTHORIZED)
        {
            // we are authorized so let's return the user name

            final QuerySolution soln = results.nextSolution();

            final Literal l = soln.getLiteral("username");

            // Repo right now doesn't contain First/Last names so let's just
            // return the username

            log.info("Authenticated user: " + l.getString());

             defaultcreds = new UsernamePasswordCredentials(user, password);

            return new String[] {l.getString(), "temp_session"};
        }
        else
        {
            log.info("Could not authenticate user: " + user);
            return null;

        }

    }

    @Override
    public void logout() throws Exception
    {
        int status = 0;
        final PostMethod method = new PostMethod(Logout.getURL());

        // now let's try to do a whoami if it succeeds then we are logged in
        // otherwise return null

        log.debug("Trying to logout at " + Logout.getURL());
        try
        {
            HttpClient client = getHttpClient();
            status = client.executeMethod(method);

            // log.debug("data dump: " + method.getResponseBodyAsString());

        }
        catch(final Exception e)
        {
            log.error("problem with logout  " + Logout.getURL() + " " + e);
        }
        finally
        {
            method.releaseConnection();
        }

        // check to see if we succeeded

        if(status == HttpStatus.SC_OK)
        {

            log.debug("logout succeded");
            // clear credentials
            defaultcreds = null;
            

        }
        else
        {
            log.info("Could not logout user: HTTP Status: " + status);

        }

    }

    @Override
    public String updateInstance(final EIInstance instance, String token) throws Exception
    {

        assert instance != null;
        assert instance.getInstanceURI().toString() != null;

        int status = 0;
        String id;
        final PostMethod method = new PostMethod(CreateInstance.getURL());
        method.setParameter("format", "application/xml");
        method.setParameter("workspace", "http://eagle-i.org/ont/repo/1.0/NG_Published");
        id = instance.getInstanceURI().toString();
        method.setParameter("uri", id);

        // log.debug("dump data: " +
        // EIInstanceFactory.INSTANCE.serialize(instance, "RDF/XML"));
        method.setParameter("insert", EIInstanceFactory.INSTANCE.serialize(instance, "RDF/XML"));
        log.debug("Trying to get update instance at " + CreateInstance.getURL() + "/" + id.split(DEFAULT_NAMESPACE)[1]);

        if(token == null)
        {
            method.setParameter("action", "gettoken");

        }
        else
        {

            method.setParameter("action", "update");
            method.setParameter("token", token);

        }

        try
        {
            HttpClient client = getHttpClient();
            status = client.executeMethod(method);
            final String resp = getStringFromInputStream(method.getResponseBodyAsStream());

            if(status == HttpStatus.SC_OK)
            {
                
                
                // check to see if we asked for token then return it
                // otherwuse just return the response
                
                if(token == null) {
                 
                    results = ResultSetFactory.fromXML(resp);
                    final QuerySolution soln = results.nextSolution();

                    final Resource r = soln.getResource("token");

                    final String tokenid = r.getURI();

                    log.info("Retrieved new token " + tokenid);

                    return tokenid;

                    
                    
                    
                } else 
                {
                    log.info("update succeded with status: " + status + " response size: " + resp.length());
                    return resp;
                }
            }
            else
            {
                log.error("update instance failed with status: " + status + " response: " + resp);
                return null;
            }

        }
        catch(final Exception e)
        {
            log.error("problem updating instance " + id + " at URL  " + CreateInstance.getURL() + " because " + e);
            return null;
        }
        finally
        {
            method.releaseConnection();
        }

    }

    @Override
    public String whoami() throws Exception
    {
        int status = 0;
        ResultSet results = null;
        final GetMethod method = new GetMethod(WhoAmI.getURL());
        HttpClient client = getHttpClient();
        client.getParams().setParameter("accept", "application/sparql-results+xml");

        // now let's try to do a whoami if it succeeds then we are logged in
        // otherwise return null

        log.debug("Trying to whoami at " + WhoAmI.getURL());
        try
        {

            status = client.executeMethod(method);

            if(status == HttpStatus.SC_OK)
            {
                results = ResultSetFactory.fromXML(getStringFromInputStream(method.getResponseBodyAsStream()));
            }
            // in = method.getResponseBodyAsStream();
            // ResultSet results = ResultSetFactory.fromXML(in);

        }
        catch(final Exception e)
        {
            log.error("problem with whoami into " + WhoAmI.getURL() + " " + e);
        }
        finally
        {
            method.releaseConnection();
        }

        // check to see if the status is UNATHENTICATED

        if(status != HttpStatus.SC_UNAUTHORIZED)
        {
            // we are authorized so let's return the user name

            final QuerySolution soln = results.nextSolution();

            final Literal l = soln.getLiteral("username");

            // Repo right now doesn't contain First/Last names so let's just
            // return the username

            log.info("Authenticated user: " + l.getString());

            return l.getString();
        }
        else
        {
            log.info("Could not whoami user: http status: " + status);
            return null;

        }

    }

    @Override
    public EIInstance getEmptyEIInstance(EIURI classUri, EIEntity instanceEntity) throws Exception
    {
        return EIInstanceFactory.createEmpty(classUri, instanceEntity);
    }

    @Override
    public boolean isOnline()
    {

        int status = 0;
        boolean online = false;
        final GetMethod method = new GetMethod(Online.getURL());
        HttpClient client = getHttpClient();

        log.debug("Trying to see if Repository is available: " + Online.getURL());
        try
        {

            status = client.executeMethod(method);

            if(status == HttpStatus.SC_OK || status == HttpStatus.SC_UNAUTHORIZED)
            {
                online = true;
                log.debug("Repository is available: " + Online.getURL() + " is available with status: " + status);
            }
            else
            {
                log.debug("Repository is unavailable: " + Online.getURL() + " is available with status: " + status);
            }

        }
        catch(final Exception e)
        {
            log.error("problem checking online status of repository: " + Online.getURL() + " " + e);
        }
        finally
        {
            method.releaseConnection();
        }

        return online;

    }

    @Override
    public String query(String sparql) throws Exception
    {
        return postQuery(sparql);
    }

    private String getStringFromInputStream(InputStream in) throws IOException
    {

        StringWriter writer = new StringWriter();
        IOUtils.copy(in, writer);
        return writer.toString();

    }

    @Override
    public void uploadInstances(String rdf) throws Exception
    {
        assert rdf != null;

        int status = 0;
        final PostMethod method = new PostMethod(Upload.getURL());

        method.setParameter("format", "text/plain");
        method.setParameter("name", "http://eagle-i.org/ont/repo/1.0/NG_Published");
        method.setParameter("type", "published");
        method.setParameter("action", "add");
        method.setParameter("content", rdf);

        try
        {
            HttpClient client = getHttpClient();
            status = client.executeMethod(method);
            final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
            if(status == HttpStatus.SC_OK)
            {
                log.info("upload succeed with status: " + status + " response: " + resp);

            }
            else
            {
                log.error("upload sparql endpoint  failed with status: " + status + " response: " + resp);

            }

        }
        catch(final Exception e)
        {
            log.error("problem with upload endpoint at URL  " + Upload.getURL() + " because " + e);

        }
        finally
        {
            method.releaseConnection();
        }

    }

}
