// The MIT License
//
// Copyright (c) 2003 Michael Grove
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.

package org.eaglei.datatools.etl.utils;

/**
 * <p>Title: </p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: </p>
 * @author Michael Grove
 * @version 1.0
 */

import java.util.Hashtable;
import java.util.Map;
import java.util.Set;
import java.util.Iterator;
import java.util.Enumeration;

/**
 * A small utility class for managing namespaces and qnames
 */
public class NamespaceManager {
    private static Hashtable mNamespaces = new Hashtable();

    private NamespaceManager()
    {
    } // cons

    /**
     * Adds all the namespace key/value pairs in the map to the list of all current namespaces
     * @param m Map - a set of namespace key/value pairs
     */
    public static void addAll(Map m)
    {
        Set aSet  = m.keySet();
        Iterator iter = aSet.iterator();
        while (iter.hasNext())
        {
            String key = iter.next().toString();
            String value = m.get(key).toString();
            if (!mNamespaces.containsKey(key))
                add(key,value);
        } // while
    } // addAll

    /**
     * Adds a single namespace key/value pair to the list
     * @param prefix String - the abbreviation for the namespace
     * @param url String - the url of the namespace
     */
    public static void add(String prefix, String url)
    {
        if (!mNamespaces.containsKey(prefix))
            mNamespaces.put(prefix,url);
    } // add

    /**
     * Encodes an rdf:ID URL as a Qname
     * @param id String - the id to convert
     * @return String - the resulting qname
     */
    public static String getQNameFor(String id)
    {
        int slashIndex = id.lastIndexOf("/");
        int hashIndex = id.lastIndexOf("#");

        String name = null;
        String prefix = null;
        String ns = null;

        if (hashIndex == -1 && slashIndex == -1)
            return id;

        if (hashIndex == -1)
        {
            // uri
            ns = id.substring(0,slashIndex);

            // local name
            name = id.substring(slashIndex+1);

            prefix = getPrefixForURI(ns);
            if (prefix == null)
            {
                int lastSlash = ns.lastIndexOf("/");
                // get the name after last slash
                prefix = ns.substring(lastSlash + 1);

                try {
                    // check to see if the thing before the name is a version number
                    // for example: http://xmlns.com/foaf/1.4/Document

                    Float.parseFloat(prefix);

                    // if it parses, then its a version number, so get the string before that "foaf" in case above
                    int prevSlash = ns.substring(0, lastSlash).lastIndexOf("/");
                    // append version number for more accurate name
                    prefix = ns.substring(prevSlash + 1, lastSlash) + "-" + prefix;
                } // try
                catch (Exception ex)
                {
                    // not a version number, just ignore exception, we have a good enough prefix
                } // catch
            } // if
        } // if
        else
        {
            // uri
            ns = id.substring(0,hashIndex+1);

            // local name
            name = id.substring(hashIndex+1);

            prefix = getPrefixForURI(ns);

            if (prefix == null)
            {
                // get the file name
                prefix = ns.substring(ns.lastIndexOf("/") + 1, ns.length() - 1);

                // strip off extension if necessary
                if (prefix.indexOf(".") != -1)
                    prefix = prefix.substring(0,prefix.lastIndexOf("."));
            } // if
        } // else

        if (mNamespaces.containsKey(prefix))
        {
            // handle dupes here
        } // if
        else add(prefix,ns);

        if (prefix.equals(""))
            return name;
        else return prefix + ":" + name;
    } // getQNameFor

    /**
     * Returns the namespace abbreviation used for the specified identifier
     * @param url String - the Id
     * @return String - the abbreviation for the id
     */
    public static String getPrefixForURI(String url)
    {
        Enumeration e = mNamespaces.keys();
        boolean hasBlank = false;
        while (e.hasMoreElements())
        {
            String prefix = e.nextElement().toString();
            String value = mNamespaces.get(prefix).toString();
            if (value.equals(url))
            {
                if (prefix.equals(""))
                    hasBlank = true;
                else return prefix;
            } // if
        } // while

        if (hasBlank)
            return "";
        else return null;
    } // getPrefixForURI

    /**
     * Returns the full expanded URI for a given qname
     * @param qname String the qname to expand
     * @return String the expanded URI, or null if the expansion cannot be performed
     */
    public static String getURIForQName(String qname) {
        int split = qname.indexOf(":");
        String prefix = qname.substring(0,split);
        String url = (String) mNamespaces.get(prefix);
        if (url == null)
            return null;
        else return url+qname.substring(split+1);
    }

    /**
     * Clear all known namespaces from the manager
     */
    public static void clear() {
        mNamespaces.clear();
    }
} // NamespaceManager
