package org.eaglei.datatools.client.rpc;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eaglei.datatools.SortByProperties;
import org.eaglei.datatools.User;
import org.eaglei.datatools.WorkFlowTransition;
import org.eaglei.datatools.client.ApplicationState;
import org.eaglei.datatools.client.DatatoolsCookies;
import org.eaglei.datatools.client.WorkFlowConstants;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.datatools.client.status.ClientSideRepositoryException;
import org.eaglei.datatools.client.ui.WidgetUtils;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.ModelService;
import org.eaglei.model.gwt.rpc.ModelServiceAsync;
import org.eaglei.search.provider.AuthSearchRequest;
import org.eaglei.security.Session;

import com.google.gwt.core.client.GWT;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;

/**
 * Maintains a client-side cache of EIInstance. Proxies all model RPC methods.
 * 
 * It is critical that all model RPC calls go through this class. All methods in this class MUST call getCached(EIInstance) on all EIInstance objects it receives from the server to ensure that there is only one instance of an EIInstance per URI in
 * the client.
 * 
 * 
 */
public class ClientRepositoryToolsManager {

	public interface SessionListener {

		void onLogIn(String username, String userUri);

		void onLogOut(boolean isSessionExpired); // Notification that a logout occurred

	}

	// TODO this is the only specific callback that seems to make sense. Revise.

	public interface BulkWorkflowCallback {
		void onSuccess(List<EIInstanceMinimal> successes);

		void onFailure(Throwable caught);

		void needsRefresh(String message);
	}

	public static final ClientRepositoryToolsManager INSTANCE = new ClientRepositoryToolsManager();
	private static RepositoryToolsModelServiceAsync repositoryService;
	private static RepositorySecurityServiceAsync securityService;
	private static ModelServiceAsync modelService;
	private Session session;
	private User user;
	private ArrayList<SessionListener> listeners;
	private static final GWTLogger log = GWTLogger.getLogger( "ClientRepositoryToolsManager" );

	private ClientRepositoryToolsManager() {
		repositoryService = GWT.create( RepositoryToolsModelService.class );
		securityService = GWT.create( RepositorySecurityService.class );
		modelService = GWT.create( ModelService.class );
	}

	public boolean isLoggedIn() {
		return Session.isValid( session );
	}

	public void initializeAfterRefresh() {
		session = DatatoolsCookies.getSession();
	}

	private void handleLogOut() {
		session = null;
		user = null;
		Cookies.removeCookie( DatatoolsCookies.DATATOOLS_USER_COOKIE_ID );
		Cookies.removeCookie( DatatoolsCookies.DATATOOLS_USER_URI_ID );
		Cookies.removeCookie( DatatoolsCookies.DATATOOLS_SESSION_ID );
		log.info( "removed cookies" );
		if ( listeners != null ) {
			for (final SessionListener listener : listeners) {
				listener.onLogOut( false );
			}
		}
	}

	public void handleExpiredSession() {
		session = null;
		user = null;
		Cookies.removeCookie( DatatoolsCookies.DATATOOLS_USER_COOKIE_ID );
		Cookies.removeCookie( DatatoolsCookies.DATATOOLS_USER_URI_ID );
		Cookies.removeCookie( DatatoolsCookies.DATATOOLS_SESSION_ID );
		log.info( "removed cookies" );
		if ( listeners != null ) {
			for (final SessionListener listener : listeners) {
				listener.onLogOut( true );
			}
		}
	}

	public void addSessionListener(final SessionListener listener) {
		if ( listeners == null ) {
			listeners = new ArrayList<SessionListener>();
		}
		listeners.add( listener );

	}

	public void logOut() {
		if ( !isLoggedIn() ) {
			// ?? Probably should fire logOut even in deferred command
			return;
		}
		// Need to wait for callback, or ok to just assume logged out?
		securityService.logout( session, new AsyncCallback<Void>() {

			@Override
			public void onFailure(final Throwable caught) {
				handleLogOut();
			}

			@Override
			public void onSuccess(final Void result) {
				session = null;
				user = null;
				handleLogOut();
			}
		} );

	}

	public void logIn(final String username, final String password, final RootAsyncCallback<User> callback) {
		if ( isLoggedIn() ) {
			securityService.whoami( session, new AsyncCallback<User>() {
				
				@Override
				public void onFailure(final Throwable caught) {
					user = null;
					session = null;
					callback.onFailure( caught );
				}

				@Override
				public void onSuccess(final User retrievedUser) {
					log.info( "user " + retrievedUser + " had " + user.getAllowedTransitionsForState( WorkFlowConstants.NEW_URI ).size() + " transitions from New" );
					if ( isUserDeniedAccess( retrievedUser ) ) {
						handleAccessDenied( callback );
					}// TODO: this is identical to a clause below. extract

					if ( username.equals( retrievedUser.getUserName() ) ) {
						session = retrievedUser.getSession();
						Cookies.setCookie( DatatoolsCookies.DATATOOLS_SESSION_ID, session.getSessionId() );
						callback.onSuccess( retrievedUser );
					} else {
						user = null;
						session = null;
						// FIXME is this the right exception?
						callback.onFailure( new ClientSideRepositoryException( "Error logging in: already logged in as different user.  Please log out first." ) );
					}
				}
			} );

		} else { // FIXME verify
			securityService.login( username, password, new AsyncCallback<User>() {

				@Override
				public void onFailure(final Throwable caught) {
					user = null;
					session = null;
					callback.onFailure( caught );
				}

				@Override
				public void onSuccess(final User result) {
					session = result.getSession();
					final String userUri = result.getUserURI().toString();
					user = result;

					if ( isUserDeniedAccess( user ) ) {
						handleAccessDenied( callback );
						return;
					}

					Cookies.setCookie( DatatoolsCookies.DATATOOLS_USER_COOKIE_ID, username );
					Cookies.setCookie( DatatoolsCookies.DATATOOLS_USER_URI_ID, userUri );
					Cookies.setCookie( DatatoolsCookies.DATATOOLS_SESSION_ID, session.getSessionId() );
					// Suggestion from GWT security is to use
					// Cookies.setCookie(DatatoolsCookies.DATATOOLS_USER_COOKIE_ID,
					// username, expires, null, "/", true);
					// and call checkValidSession() on server
					// --presumably call whoami & check it's valid not
					// invalid
					log.info( "set cookies; userUri = " + Cookies.getCookie( DatatoolsCookies.DATATOOLS_USER_URI_ID ) + " session = " + Cookies.getCookie( DatatoolsCookies.DATATOOLS_SESSION_ID ) );
					if ( listeners != null ) {
						for (final SessionListener listener : listeners) {
							listener.onLogIn( result.getUserName(), result.getUserURI().toString() );
						}
					}

					callback.onSuccess( result );
				}
			} );

		}
	}

	public void getToken(final EIInstance instance, final RootAsyncCallback<String> callback) {
		repositoryService.getToken( session, instance, callback );
	}

	public void getTokens(final List<EIInstance> instances, final RootAsyncCallback<Map<EIInstance, String>> callback) {
		repositoryService.getTokens( session, instances, callback );
	}

	public void updateInstance(final EIInstance eiInstance, final String token, final RootAsyncCallback<Void> callback) {
		repositoryService.updateInstance( session, eiInstance, token, callback );
	}

	public void updateInstances(final Map<EIInstance, String> instancesWithTokens, final RootAsyncCallback<Void> callback) {
		repositoryService.updateInstances( session, instancesWithTokens, callback );
	}

	public void getInstance(final EIURI eiURI, final RootAsyncCallback<EIInstance> callback) {
		repositoryService.getOneInstance( session, eiURI, true, callback );
	}

	// FIXME for now I'm putting this here; to be moved to the instance code in common
	public void setReferencingResources(final EIInstance instance, RootAsyncCallback<EIInstance> callback) {
		repositoryService.setReferencingResources( session, instance, callback );
	}

	public void deleteInstance(final EIURI instanceUri, final RootAsyncCallback<Void> callback) {
		repositoryService.deleteInstance( session, instanceUri, callback );
	}

	public boolean canEdit(final EIURI workflowState) {
		return user.canEdit( workflowState );
	}

	public boolean canClaim(final EIInstance instance) {
		// If instance has no state it means it's a new instance; by definition user owns it
		if ( isNotNull( instance.getWFState() ) ) {
			return !instanceHasAnyOwner( instance ) && canEdit( instance.getWFState().getURI() );
		} else {
			return true;
		}
	}

	public boolean canClaim(final EIInstanceMinimal instance) {
		return !instanceHasAnyOwner( instance ) && canEdit( instance.getWFStateUri() );
	}

	public boolean canEdit(final EIInstanceMinimal instance) {
		return instanceHasCurrentOwner( instance ) && canEdit( instance.getWFStateUri() );
	}

	public boolean canEdit(final EIInstance instance) {
		// If instance has no state it means it's a new instance; by definition user owns it
		if ( isNotNull( instance.getWFState() ) ) {
			return instanceHasCurrentOwner( instance ) && canEdit( instance.getWFState().getURI() );
		} else {
			return true;
		}
	}

	public void listResourcesForObjectPropertyValue(final EIURI classUri, final EIURI provider, final EIURI state, final boolean onlyProvider, final RootAsyncCallback<List<EIInstanceMinimal>> callback) {
		repositoryService.listResourcesForObjectPropertyValue( session, classUri, provider, state, onlyProvider, callback );
	}

	public void listResources(final AuthSearchRequest queryRequest, final SortByProperties orderBy, boolean isAscending, final boolean strictOwnerFilter, final boolean stubsOnly, final RootAsyncCallback<List<EIInstanceMinimal>> callback) {
		repositoryService.listResources( session, queryRequest, orderBy, isAscending, strictOwnerFilter, stubsOnly, callback );
	}

	public void listReferencingResources(final EIURI resourceUri, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter, final RootAsyncCallback<List<EIInstanceMinimal>> callback) {
		repositoryService.listReferencingResources( session, resourceUri, queryRequest, orderBy, strictOwnerFilter, callback );
	}

	public void getNewInstanceID(final int count, final RootAsyncCallback<List<EIURI>> callback) {
		repositoryService.getNewInstanceID( session, count, callback );
	}

	public void createInstances(final List<EIInstance> instances, final RootAsyncCallback<List<EIInstance>> callback) {

		repositoryService.createInstances( session, instances, ApplicationState.getInstance().getWorkspaceEntity(), new AsyncCallback<Void>() {
			@Override
			public void onFailure(final Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void onSuccess(final Void result) {
				log.info( "creation succeeded" );
				for (EIInstance instance : instances) {
					instance.setWFState( WorkFlowConstants.DRAFT_ENTITY );
				}
				callback.onSuccess( instances );
			}
		} );

	}

	public void createInstance(final EIInstance instance, final RootAsyncCallback<EIInstance> callback) {

		log.info( "creating instance " + instance.getEntity() + " with type " + instance.getInstanceClass() );
		repositoryService.createInstance( session, instance, ApplicationState.getInstance().getWorkspaceEntity(), new AsyncCallback<Void>() {

			@Override
			public void onFailure(final Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void onSuccess(final Void result) {
				log.info( "creation succeeded" );
				instance.setWFState( WorkFlowConstants.DRAFT_ENTITY );
				callback.onSuccess( instance );
			}
		} );

	}

	public void getEmptyEIInstance(final EIURI classUri, final EIEntity instanceEntity, final RootAsyncCallback<EIInstance> callback) {
		repositoryService.getEmptyEIInstance( session, classUri, instanceEntity, new AsyncCallback<EIInstance>() {

			@Override
			public void onFailure(final Throwable caught) {
				log.warn( "failed to get empty instance with class " + classUri );
				callback.onFailure( caught );
			}

			@Override
			public void onSuccess(final EIInstance instance) {
				instance.setWFOwner( user.getUserEntity() );
				// FIXME can we get away from hardcoding this first state?
				instance.setWFState( WorkFlowConstants.DRAFT_ENTITY );
				callback.onSuccess( instance );
			}
		} );
	}

	public void getEmptyEIInstance(final EIURI classUri, final RootAsyncCallback<EIInstance> callback) {
		repositoryService.getEmptyEIInstance( session, classUri, callback );
	}

	public void deepCopyInstance(final EIURI originalUri, final RootAsyncCallback<EIInstance> callback) {
		repositoryService.deepCopy( session, originalUri, callback );
	}

	public void whoami(final RootAsyncCallback<User> callback) {
		if ( securityService == null ) {
			securityService = GWT.create( RepositorySecurityService.class );
		}
		final Session session = DatatoolsCookies.getSession();
		securityService.whoami( session, new AsyncCallback<User>() {

			@Override
			public void onFailure(final Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void onSuccess(final User userInfo) {
				if ( userInfo != null && !userInfo.equals( user ) ) {
					user = userInfo;
				}
				callback.onSuccess( userInfo );
			}
		} );
	}

	public void claim(final EIInstanceMinimal instance, final RootAsyncCallback<EIInstance> callback) {
		claim( new EIInstanceMinimal[] { instance }, new BulkWorkflowCallback() {

			@Override
			public void onSuccess(final List<EIInstanceMinimal> successes) {
				if ( successes == null || successes.size() == 0 ) {
					callback.onFailure( new ClientSideRepositoryException( "claim failed for " + instance.getInstanceLabel() ) );
				}
				getInstance( successes.get( 0 ).getInstanceURI(), callback );

			}

			@Override
			public void onFailure(Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void needsRefresh(final String message) {
				callback.onFailure( new ClientSideRepositoryException( message ) ); // TODO: add a needsRefresh?

			}
		} );
	}

	public void claim(final EIInstanceMinimal[] instances, final BulkWorkflowCallback callback) {

		final Map<EIURI, EIInstanceMinimal> instanceUriMap = new HashMap<EIURI, EIInstanceMinimal>();
		for (final EIInstanceMinimal instance : instances) {
			instanceUriMap.put( instance.getInstanceURI(), instance );
		}

		repositoryService.getModifiedDates( session, new ArrayList<EIURI>( instanceUriMap.keySet() ), new AsyncCallback<Map<EIURI, String>>() {

			@Override
			public void onFailure(final Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void onSuccess(final Map<EIURI, String> result) {

				final List<EIURI> unchangedUris = new ArrayList<EIURI>();
				final List<EIURI> outdatedUris = new ArrayList<EIURI>();

				partitionOKUris( instanceUriMap, result, unchangedUris, outdatedUris );

				if ( outdatedUris.size() > 0 ) {
					callback.needsRefresh( outdatedUris.size() + " resources have been edited since first load" );
					return;
				}

				// TODO: need to warn about outdated uris

				repositoryService.claim( session, unchangedUris, new AsyncCallback<List<EIURI>>() {

					@Override
					public void onFailure(final Throwable caught) {
						callback.onFailure( caught );
					}

					@Override
					public void onSuccess(final List<EIURI> result) {
						final List<EIInstanceMinimal> successes = new ArrayList<EIInstanceMinimal>();
						for (final EIURI succeededUri : result) {
							final EIInstanceMinimal succeeded = instanceUriMap.get( succeededUri );
							succeeded.setWFOwner( user.getUserEntity() );
							successes.add( succeeded );
						}
						callback.onSuccess( successes );
					}
				} );

			}
		} );

	}

	protected void partitionOKUris(final Map<EIURI, EIInstanceMinimal> instanceUriMap, final Map<EIURI, String> uriModificationDateMap, final List<EIURI> unchangedUris, final List<EIURI> outdatedUris) {
		final DateTimeFormat dateFormat = DateTimeFormat.getFormat( "yyyy-MM-dd'T'HH:mm:ss.SSSZ" );

		for (final EIURI uri : instanceUriMap.keySet()) {
			final String repositoryDateString = uriModificationDateMap.get( uri );
			if ( repositoryDateString == null ) {
				log.warn( "no modification date from repository for " + uri );
				continue;
			}

			String instanceDateString = instanceUriMap.get( uri ).getModifiedDate();
			if ( instanceDateString == null ) {
				outdatedUris.add( uri );
				return;
			}
			if ( instanceDateString.indexOf( "^" ) != -1 ) {
				instanceDateString = instanceDateString.substring( 0, instanceDateString.indexOf( "^" ) );
			}

			final Date repositoryModification = dateFormat.parse( repositoryDateString );
			final Date instanceModification = dateFormat.parse( instanceDateString );

			if ( repositoryModification.after( instanceModification ) ) {
				outdatedUris.add( uri );
			} else {
				unchangedUris.add( uri );
			}

		}
	}

	public void release(final EIInstanceMinimal instance, final RootAsyncCallback<EIInstance> callback) {
		release( new EIInstanceMinimal[] { instance }, new BulkWorkflowCallback() {

			@Override
			public void onSuccess(final List<EIInstanceMinimal> successes) {
				if ( successes == null || successes.size() == 0 ) {
					callback.onFailure( new ClientSideRepositoryException( "release failed for " + instance.getInstanceLabel() ) );
				}

				getInstance( successes.get( 0 ).getInstanceURI(), callback );

			}

			@Override
			public void onFailure(Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void needsRefresh(final String message) {
				callback.onFailure( new ClientSideRepositoryException( message ) ); // TODO: add a needsRefresh here too?
			}
		} );
	}

	public void release(final EIInstanceMinimal[] instances, final BulkWorkflowCallback callback) {

		final Map<EIURI, EIInstanceMinimal> instanceUriMap = new HashMap<EIURI, EIInstanceMinimal>();
		for (final EIInstanceMinimal instance : instances) {
			instanceUriMap.put( instance.getInstanceURI(), instance );
		}

		repositoryService.release( session, new ArrayList<EIURI>( instanceUriMap.keySet() ), new AsyncCallback<List<EIURI>>() {

			@Override
			public void onFailure(final Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void onSuccess(final List<EIURI> result) {
				final List<EIInstanceMinimal> successes = new ArrayList<EIInstanceMinimal>();
				for (final EIURI succeededUri : result) {
					final EIInstanceMinimal succeeded = instanceUriMap.get( succeededUri );
					succeeded.setWFOwner( EIEntity.NULL_ENTITY );
					successes.add( succeeded );
				}
				callback.onSuccess( successes );
			}
		} );

	}

	public static boolean isNotNull(final EIEntity entity) {
		return entity != null && !EIEntity.NULL_ENTITY.equals( entity );
	}

	public static boolean isNotNull(final EIInstance instance) {
		return instance != null && !EIInstance.NULL_INSTANCE.equals( instance );
	}

	public boolean instanceHasCurrentOwner(final EIInstance instance) {
		return isNotNull( instance.getWFOwner() ) && instance.getWFOwner().equals( user.getUserEntity() );
	}

	public boolean instanceHasCurrentOwner(final EIInstanceMinimal instance) {
		return isNotNull( instance.getWFOwner() ) && instance.getWFOwner().equals( user.getUserEntity() );
	}

	public boolean instanceHasAnyOwner(final EIInstance instance) {
		return isNotNull( instance.getWFOwner() );
	}

	public boolean instanceHasAnyOwner(final EIInstanceMinimal instance) {
		return isNotNull( instance.getWFOwner() );
	}

	public void transition(final EIInstance instance, final WorkFlowTransition transition, final RootAsyncCallback<EIInstance> callback) {
		transition( new EIInstanceMinimal[] { EIInstanceMinimal.create( instance ) }, transition, new BulkWorkflowCallback() {
			@Override
			public void onSuccess(final List<EIInstanceMinimal> successes) {
				if ( successes == null || successes.size() == 0 ) {
					callback.onFailure( new ClientSideRepositoryException( "claim failed for " + instance.getInstanceLabel() ) );
				}
				getInstance( successes.get( 0 ).getInstanceURI(), callback );

			}

			@Override
			public void onFailure(Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void needsRefresh(final String message) {
				callback.onFailure( new ClientSideRepositoryException( message ) );
			}
		} );
	}

	public void transition(final EIInstanceMinimal[] eiInstanceMinimals, final WorkFlowTransition transition, final BulkWorkflowCallback callback) {

		if ( transition == null ) {
			log.error( "null transition!" );
			// FIXME deal with the failure differently
			callback.onFailure( new ClientSideRepositoryException( "Invalid parameter" ) );
		}

		final Map<EIURI, EIInstanceMinimal> instanceUriMap = new HashMap<EIURI, EIInstanceMinimal>();
		for (final EIInstanceMinimal instance : eiInstanceMinimals) {
			instanceUriMap.put( instance.getInstanceURI(), instance );
		}

		repositoryService.transition( session, new ArrayList<EIURI>( instanceUriMap.keySet() ), transition.getEntity(), new AsyncCallback<List<EIURI>>() {

			@Override
			public void onFailure(final Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void onSuccess(final List<EIURI> result) {
				final List<EIInstanceMinimal> successes = new ArrayList<EIInstanceMinimal>();
				for (final EIURI succeededUri : result) {
					final EIInstanceMinimal succeeded = instanceUriMap.get( succeededUri );
					succeeded.setWFState( WorkFlowConstants.WORKFLOW_URI_MAP.get( transition.getToStateURI() ) );
					succeeded.setWFOwner( EIEntity.NULL_ENTITY ); // Force this: we know it's true because that's what transitioning does
					successes.add( succeeded );
				}
				callback.onSuccess( successes );
			}
		} );

	}

	public void retrieveLabel(final EIURI uri, final RootAsyncCallback<String> callback) {
		repositoryService.retrieveLabel( session, uri, callback );
	}

	public void getRootSuperclassForInstanceUri(final EIURI instanceUri, final RootAsyncCallback<EIClass> callback) {
		repositoryService.getRootSuperclassForInstanceUri( session, instanceUri, callback );
	}

	public void getClassAndSuperclassesForInstanceUri(final EIURI instanceUri, final RootAsyncCallback<List<EIClass>> callback) {
		repositoryService.getClassAndSuperclassesForInstanceUri( session, instanceUri, callback );
	}

	public List<WorkFlowTransition> getAllowedTransitions() {
		return user.getAllowedTransitions();
	}

	public List<WorkFlowTransition> getAllowedTransitions(final EIInstance instance) {
		return getAllowedTransitions( instance.getWFState() );
	}

	public List<WorkFlowTransition> getAllowedTransitions(final EIInstanceMinimal instance) {
		return getAllowedTransitions( instance.getWFState() );
	}

	private List<WorkFlowTransition> getAllowedTransitions(final EIEntity workflowState) {
		if ( isNotNull( workflowState ) ) {
			return user.getAllowedTransitionsForState( workflowState.getURI() );
		} else { // state may be null if instance is new and hasn't been pushed to repo
			// FIXME can we move from hardcoding this here?
			return user.getAllowedTransitionsForState( WorkFlowConstants.DRAFT_URI );
		}
	}

	/*
	 * public void getRootSuperClass(final EIClass eclass, final RootAsyncCallback<EIClass> callback) { repositoryService.getRootSuperClass( eclass, callback); }
	 */

	public void getRootSuperClass(final EIEntity eiEntity, final RootAsyncCallback<EIClass> callback) {
		modelService.getSuperClasses( eiEntity.getURI(), new AsyncCallback<List<EIClass>>() {

			@Override
			public void onFailure(Throwable caught) {
				callback.onFailure( caught );
			}

			@Override
			public void onSuccess(List<EIClass> result) {
				if ( result == null || result.isEmpty() ) {
					modelService.getClass( eiEntity.getURI(), new AsyncCallback<EIClass>() {

						@Override
						public void onFailure(Throwable caught) {
							callback.onFailure( caught );

						}

						@Override
						public void onSuccess(EIClass result) {
							callback.onSuccess( result );

						}
					} );
				} else {
					callback.onSuccess( result.get( result.size() - 1 ) );
				}
			}

		} );
	}

	public void getEquivalentClasses(final List<EIProperty> propertiesToPopulate, final RootAsyncCallback<List<EIProperty>> callback) {
		repositoryService.getAllEquivalentClasses( propertiesToPopulate, callback );
	}

	public void isModelClassURI(final EIURI eiuri, final RootAsyncCallback<Boolean> callback) {
		repositoryService.isModelClassURI( eiuri, callback );
	}

	private boolean isUserDeniedAccess(User thisUser) {
		List<WorkFlowTransition> creates = thisUser.getAllowedTransitionsForState( WorkFlowConstants.NEW_URI );
		log.info( "user " + thisUser + " had " + ( creates == null ? "null" : creates.size() ) + " transitions from New" );
		return ( creates == null || creates.size() == 0 );

	}

	private void handleAccessDenied(final RootAsyncCallback<User> callback) {
		log.info( "disabling login" );
		user = null;
		session = null;
		// If you can't create anything, you don't belong here (for now)
		// callback.onFailure( new ClientSideRepositoryException( "Error logging in: user not authorized for data tool." ) );
		Window.alert( "Error logging in: user not authorized for data tool." ); // TODO: HACK!
		Window.Location.assign( WidgetUtils.getSearchURL() );
	}
}
